/*
**  NewMailboxPanelController.m
**
**  Copyright (c) 2001, 2002, 2003
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "NewMailboxPanelController.h"

#include "Filter.h"
#include "FilterManager.h"
#include "Constants.h"
#include "MailboxManager.h"
#include "MailboxManagerController.h"

#ifndef MACOSX
#include "NewMailboxPanel.h"
#endif

#include "Utilities.h"

#include <Pantomime/Constants.h>
#include <Pantomime/Store.h>
#include <Pantomime/IMAPStore.h>
#include <Pantomime/LocalStore.h>
#include <Pantomime/NSString+Extensions.h>
#include <Pantomime/URLName.h>

#define UPDATE_PATH(name, theOldPath, thePath) ({ \
 if ( [[allValues objectForKey: name] isEqualToString: theOldPath] ) \
   { \
     [allValues setObject: thePath  forKey: name]; \
   } \
})


//
//
//
@implementation NewMailboxPanelController

- (id) initWithWindowNibName: (NSString *) windowNibName
		  folderNode: (FolderNode *) theFolderNode
{
#ifdef MACOSX
  
  self = [super initWithWindowNibName: windowNibName];
 
#else
  NewMailboxPanel *aNewMailboxPanel;

  aNewMailboxPanel = [[NewMailboxPanel alloc] initWithContentRect: NSMakeRect(200,200,345,150)
					      styleMask: NSTitledWindowMask|NSMiniaturizableWindowMask
					      backing: NSBackingStoreBuffered
					      defer: YES];
  
  self = [super initWithWindow: aNewMailboxPanel];
  
  [aNewMailboxPanel layoutPanel];
  [aNewMailboxPanel setDelegate: self];
  
  // We link our outlets
  mailboxNameLabel = (NSTextField *)aNewMailboxPanel->mailboxNameLabel;
  mailboxNameField = aNewMailboxPanel->mailboxNameField;
  popUpButton = aNewMailboxPanel->popUpButton;
  
  RELEASE(aNewMailboxPanel);
#endif

  // We link our ivar, no need to retain it
  folderNode = theFolderNode;

  // We set our window title
  [[self window] setTitle: _(@"New Mailbox")];

  // We add our standard popup items
  [popUpButton removeAllItems];
  [popUpButton insertItemWithTitle: _(@"Mailbox") atIndex: 0];
  [popUpButton insertItemWithTitle: _(@"Maildir") atIndex: 1];
  [popUpButton insertItemWithTitle: _(@"Folder")  atIndex: 2];

  return self;
}


//
//
//
- (void) dealloc
{
  NSDebugLog(@"NewMailboxPanelController: -dealloc");
  TEST_RELEASE(name);

  [super dealloc];
}


//
// action methods
//
- (IBAction) okClicked: (id) sender
{
  // If we are renaming a mailbox..
  if ( [self name] )
    {
      // We must be sure our names are different
      if ( ![[self name] isEqualToString: [[mailboxNameField stringValue] stringByTrimmingWhiteSpaces]] ) 
	{
	  // "name" is the old name of the mailbox while the content of our text field is the new
	  // mailbox name
	  NSString *pathOfFolder, *aSource, *aDestination;
	  Store *aStore;
	  
	  //
	  // We first get our right store
	  //
	  if ( [[Utilities completePathForFolderNode: folderNode  separator: @"/"]
		 hasPrefix: [NSString stringWithFormat: @"/%@", _(@"Local")]] )
	    {
	      aStore = [[MailboxManagerController singleInstance]
			 storeForName: @"GNUMAIL_LOCAL_STORE"
			 username: NSUserName()];
	    }
	  else
	    {
	      NSString *aServerName, *aUsername;

	      [Utilities storeKeyForFolderNode: folderNode
			 serverName: &aServerName
			 username: &aUsername];
	      
	      aStore = [[MailboxManagerController singleInstance] storeForName: aServerName
								  username: aUsername];
	    }


	  //
	  // pathOfFolder will hold a value like:  folderA
	  //                                       folderA/folderB
	  //                                       folderA/folderB/folderC
	  //                                  or:
	  //                                       folderA.folderB
	  //                                       folderA.folderB.folderC
	  //
	  pathOfFolder = [Utilities pathOfFolderFromFolderNode: folderNode
				    separator: [(id<Store>)aStore folderSeparator]];

	  if ( !pathOfFolder )
	    {
	      aSource = [self name];
	      aDestination = [[mailboxNameField stringValue] stringByTrimmingWhiteSpaces];
	    }
	  else 
	    {
	      NSRange aRange;

	      aSource = pathOfFolder;
	      
	      // We set the right destination folder, first by trimming the last path component
	      // using the right separator.
	      aRange = [pathOfFolder rangeOfString: [(id<Store>)aStore folderSeparator]
				     options: NSBackwardsSearch];
	      
	      if ( aRange.length )
		{
		  aDestination = [pathOfFolder substringToIndex: aRange.location];
		}
	      else
		{
		  aDestination = @"";
		}
	      
	      if ( [aDestination length] == 0 )
		{
		  aDestination = [[mailboxNameField stringValue] stringByTrimmingWhiteSpaces];
		}
	      else
		{
		  aDestination = [NSString stringWithFormat: @"%@%@%@", 
					   aDestination,
					   [(id<Store>)aStore folderSeparator],
					   [[mailboxNameField stringValue] stringByTrimmingWhiteSpaces]];
		}
	    }	      

	  //
	  // We _rename_ our folder...
	  //
	  if ( [(id<Store>)aStore renameFolderWithName: aSource
			   toName: aDestination] )
	    {
	      NSString *aSourceURL, *aDestinationURL;
	      
	      // We build our right URLs
	      if ( [(id<NSObject>)aStore isKindOfClass: [LocalStore class]] )
		{
		  aSourceURL = [NSString stringWithFormat: @"local://%@/%@",
					 [[NSUserDefaults standardUserDefaults] objectForKey: @"LOCALMAILDIR"],
					 aSource];
		  aDestinationURL = [NSString stringWithFormat: @"local://%@/%@",
					      [[NSUserDefaults standardUserDefaults] objectForKey: @"LOCALMAILDIR"],
					      aDestination];
		}
	      else
		{
		  aSourceURL = [NSString stringWithFormat: @"imap://%@@%@/%@", 
					 [(IMAPStore *)aStore username], 
					 [(IMAPStore *)aStore name],
					 aSource];
		  aDestinationURL = [NSString stringWithFormat: @"imap://%@@%@/%@", 
					      [(IMAPStore *)aStore username], 
					      [(IMAPStore *)aStore name], 
					      aDestination];
		}
	      
	      NSDebugLog(@"aSourceURL = |%@| aDestinationURL = |%@|", aSourceURL, aDestinationURL); 
	      
	      // We update our filters.
	      [self _updateFiltersFromOldPath: aSourceURL  toPath: aDestinationURL];
	      
	      // We update our "MAILBOXES" for all accounts
	      [self _updateMailboxesFromOldPath: aSourceURL  toPath: aDestinationURL];
	      
	      
	      //
	      // Then, we must verify if we must rename our IMAP cache file.
	      //
	      if ( [(id<NSObject>)aStore isKindOfClass: [IMAPStore class]] )
		{
		  NSString *aKey, *aSourcePath, *aDestinationPath;
		  
		  aKey = [NSString stringWithFormat: @"%@ @ %@", 
				   [(IMAPStore *)aStore username], 
				   [(IMAPStore *)aStore name]];
		  
		  aSourcePath = [NSString stringWithFormat: @"%@/IMAPCache_%@_%@",
					  GNUMailUserLibraryPath(),
					  [Utilities flattenPathFromString: aKey
						     separator: @"/"],
					  [Utilities flattenPathFromString: aSource
						     separator: [(id<Store>)aStore folderSeparator]] ];
		  
		  aDestinationPath = [NSString stringWithFormat: @"%@/IMAPCache_%@_%@", 
					       GNUMailUserLibraryPath(),
					       [Utilities flattenPathFromString: aKey
							  separator: @"/"],
					       [Utilities flattenPathFromString: aDestination
							  separator: [(id<Store>)aStore folderSeparator]] ];
		  
		  [[NSFileManager defaultManager] movePath: aSourcePath
						  toPath: aDestinationPath
						  handler: nil];
		  
		}
	    }
	  else
	    {
	      NSRunInformationalAlertPanel(_(@"Error!"),
					   _(@"The mailbox rename operation failed."),
					   _(@"OK"),
					   NULL,
					   NULL,
					   NULL);
	      [NSApp abortModal];
	      [self close];
	    }
	}
    }
  //
  // We are creating a mailbox, Local or IMAP.
  //
  else
    {
      NSString *pathOfFolder, *aString;
      id aStore;
      
      //
      // We first get the right store.
      //
      if ( [[Utilities completePathForFolderNode: folderNode  separator: @"/"] 
	     hasPrefix: [NSString stringWithFormat: @"/%@", _(@"Local")]] )
	{
	  aStore = [[MailboxManagerController singleInstance]
		     storeForName: @"GNUMAIL_LOCAL_STORE"
		     username: NSUserName()];
	}
      else
	{
	  NSString *aServerName, *aUsername;
	  
	  [Utilities storeKeyForFolderNode: folderNode
		     serverName: &aServerName
		     username: &aUsername];
	  
	  aStore = [[MailboxManagerController singleInstance] storeForName: aServerName
							      username: aUsername];
	}

      
      pathOfFolder = [Utilities pathOfFolderFromFolderNode: folderNode
				separator: [aStore folderSeparator]];
      
      NSDebugLog(@"pathOfFolder = %@", pathOfFolder);

      if ( !pathOfFolder || [pathOfFolder length] == 0)
	{
	  aString = [[mailboxNameField stringValue] stringByTrimmingWhiteSpaces];
	}
      else
	{
	  aString = [NSString stringWithFormat: @"%@%@%@",
			      pathOfFolder,
			      [aStore folderSeparator],
			      [[mailboxNameField stringValue] stringByTrimmingWhiteSpaces]];
	}
      
      NSDebugLog(@"new pathOfFolder = %@", aString);

      // We can now proceed with the creation of our new folder
      if ( ![aStore createFolderWithName: aString 
		    type: [[self popUpButton] indexOfSelectedItem]
		    contents: nil] )
	{
	  NSRunInformationalAlertPanel(_(@"Error!"),
				       _(@"The mailbox create operation failed."),
				       _(@"OK"),
				       NULL,
				       NULL,
				       NULL);
	  [NSApp abortModal];
	  [self close];
	}
    }
  
  [NSApp stopModal];
  [self close];
}


//
//
//
- (IBAction) cancelClicked: (id) sender
{
  [NSApp abortModal];
  [self close];
}


//
// delegate methods
//
- (void) windowWillClose: (NSNotification *) theNotification
{
  // We do nothing
}

- (void) windowDidLoad
{
  name = nil;
}


//
// access/mutation methods
//
- (MailboxManagerController *) mailboxManagerController
{
  return mailboxManagerController;
}


//
// We must NOT retain the mailbox manager controller here.
//
- (void) setMailboxManagerController: (MailboxManagerController *) theMailboxManagerController
{
  mailboxManagerController = theMailboxManagerController;
}


//
//
//
- (NSString *) name
{
  return name;
}


//
//
//
- (void) setName: (NSString *) theName
{
  if (theName)
    {
      RETAIN(theName);
      RELEASE(name);
      name = theName;     
      [mailboxNameField setStringValue: name];
    }
  else
    {
      DESTROY(name);
      [mailboxNameField setStringValue: @""];
    }
}


//
//
//
- (NSTextField *) mailboxNameLabel
{
  return mailboxNameLabel;
}


//
//
//
- (NSTextField *) mailboxNameField
{
  return mailboxNameField;
}


//
//
//
- (NSPopUpButton *) popUpButton
{
  return popUpButton;
}

@end


//
// Private methods
//
@implementation NewMailboxPanelController (Private)

- (void) _updateFiltersFromOldPath: (NSString *) theOldPath
			    toPath: (NSString *) thePath
{
  FilterManager *aFilterManager;
  Filter *aFilter;
  int i;
  
  aFilterManager = [FilterManager singleInstance];
  
  for (i = 0; i < [[aFilterManager filters] count]; i++)
    {   
      aFilter = [aFilterManager filterAtIndex: i];
      
      if ([aFilter action] == TRANSFER_TO_FOLDER)
	{
	  if ( [[aFilter actionFolderName] isEqualToString: theOldPath] )
	    {
	      [aFilter setActionFolderName: thePath];
	    }
	}
    }
  
  [aFilterManager synchronize];	
}


- (void) _updateMailboxesFromOldPath: (NSString *) theOldPath
                              toPath: (NSString *) thePath

{
  NSMutableDictionary *allAccounts, *theAccount, *allValues;
  NSEnumerator *theEnumerator;
  NSString *aKey;
  
  allAccounts = [[NSMutableDictionary alloc] initWithDictionary: [[NSUserDefaults standardUserDefaults] 
								   dictionaryForKey: @"ACCOUNTS"]];
  
  theEnumerator = [allAccounts keyEnumerator];
  
  while ( (aKey = [theEnumerator nextObject]) )
    {
      theAccount = [[NSMutableDictionary alloc] initWithDictionary: [allAccounts objectForKey: aKey]];
      allValues = [[NSMutableDictionary alloc] initWithDictionary: [theAccount objectForKey: @"MAILBOXES"]];
      
      UPDATE_PATH(@"DRAFTSFOLDERNAME", theOldPath, thePath);
      UPDATE_PATH(@"INBOXFOLDERNAME", theOldPath, thePath);
      UPDATE_PATH(@"SENTFOLDERNAME", theOldPath, thePath);
      UPDATE_PATH(@"TRASHFOLDERNAME", theOldPath, thePath);
      
      [theAccount setObject: allValues  forKey: @"MAILBOXES"];
      RELEASE(allValues);
      
      [allAccounts setObject: theAccount  forKey: aKey];
      RELEASE(theAccount);
    }
  
  [[NSUserDefaults standardUserDefaults] setObject: allAccounts  forKey: @"ACCOUNTS"];
  RELEASE(allAccounts);
}

@end

#undef UPDATE_PATH
