#include "editor.h"
#include <gst/common/gste-debug.h>

/* class functions */
static void gst_editor_link_class_init (GstEditorLinkClass * klass);
static void gst_editor_link_init (GstEditorLink * link);

static void gst_editor_link_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec);
static void gst_editor_link_get_property (GObject * object, guint prop_id,
    GValue * value, GParamSpec * pspec);
static void gst_editor_link_realize (GnomeCanvasItem * citem);
static void gst_editor_link_resize (GstEditorLink * link);

/* callbacks from gstreamer */
static void on_pad_unlink (GstPad * pad, GstPad * peer, GstEditorLink * link);
static void on_new_pad (GstElement * element, GstPad * pad,
    GstEditorLink * link);

/* callbacks from editor pads */
static void on_editor_pad_position_changed (GstEditorPad * pad,
    GstEditorLink * link);

/* utility */
static void make_dynamic_link (GstEditorLink * link);


enum
{
  PROP_0,
  PROP_X,
  PROP_Y,
  PROP_X1,
  PROP_Y1,
  PROP_X2,
  PROP_Y2,
  PROP_SRCPAD,
  PROP_SINKPAD,
  PROP_GHOST,
};

enum
{
  LAST_SIGNAL
};

static GObjectClass *parent_class;

/* static guint gst_editor_link_signals[LAST_SIGNAL] = { 0 }; */

GType
gst_editor_link_get_type (void)
{
  static GType link_type = 0;

  if (!link_type) {
    static const GTypeInfo link_info = {
      sizeof (GstEditorLinkClass),
      (GBaseInitFunc) NULL,
      (GBaseFinalizeFunc) NULL,
      (GClassInitFunc) gst_editor_link_class_init,
      NULL,
      NULL,
      sizeof (GstEditorLink),
      0,
      (GInstanceInitFunc) gst_editor_link_init,
    };

    link_type =
	g_type_register_static (gnome_canvas_line_get_type (), "GstEditorLink",
	&link_info, 0);
  }
  return link_type;
}

static void
gst_editor_link_class_init (GstEditorLinkClass * klass)
{
  GObjectClass *object_class;

  object_class = G_OBJECT_CLASS (klass);

  parent_class = g_type_class_ref (gnome_canvas_line_get_type ());

  object_class->set_property = gst_editor_link_set_property;
  object_class->get_property = gst_editor_link_get_property;

  g_object_class_install_property (object_class, PROP_X,
      g_param_spec_double ("x", "x", "x",
	  -G_MAXDOUBLE, G_MAXDOUBLE, 0, G_PARAM_READWRITE));
  g_object_class_install_property (object_class, PROP_Y,
      g_param_spec_double ("y", "y", "y",
	  -G_MAXDOUBLE, G_MAXDOUBLE, 0, G_PARAM_READWRITE));
  g_object_class_install_property (object_class, PROP_X1,
      g_param_spec_double ("x1", "x1", "x1",
	  -G_MAXDOUBLE, G_MAXDOUBLE, 0, G_PARAM_READABLE));
  g_object_class_install_property (object_class, PROP_Y1,
      g_param_spec_double ("y1", "y1", "y1",
	  -G_MAXDOUBLE, G_MAXDOUBLE, 0, G_PARAM_READABLE));
  g_object_class_install_property (object_class, PROP_X2,
      g_param_spec_double ("x2", "x2", "x2",
	  -G_MAXDOUBLE, G_MAXDOUBLE, 0, G_PARAM_READABLE));
  g_object_class_install_property (object_class, PROP_Y2,
      g_param_spec_double ("y2", "y2", "y2",
	  -G_MAXDOUBLE, G_MAXDOUBLE, 0, G_PARAM_READABLE));
  g_object_class_install_property (object_class, PROP_SRCPAD,
      g_param_spec_object ("src-pad", "src-pad", "src-pad",
	  gst_editor_pad_get_type (), G_PARAM_READWRITE));
  g_object_class_install_property (object_class, PROP_SINKPAD,
      g_param_spec_object ("sink-pad", "sink-pad", "sink-pad",
	  gst_editor_pad_get_type (), G_PARAM_READWRITE));
  g_object_class_install_property (object_class, PROP_GHOST,
      g_param_spec_boolean ("ghost", "ghost", "ghost",
	  FALSE, G_PARAM_READWRITE));

  GNOME_CANVAS_ITEM_CLASS (klass)->realize = gst_editor_link_realize;
}

static void
gst_editor_link_init (GstEditorLink * link)
{
  link->points = gnome_canvas_points_new (2);

  gnome_canvas_item_raise ((GnomeCanvasItem *) link, 10);
}

static void
gst_editor_link_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec)
{
  GstEditorPad *srcpad, *sinkpad;
  GstEditorLink *link = GST_EDITOR_LINK (object);

  switch (prop_id) {
    case PROP_X:
      if (link->srcpad && link->sinkpad)
	g_warning ("Settting link drag x without having one unset pad");

      link->dragging = TRUE;
      link->x = g_value_get_double (value);
      break;

    case PROP_Y:
      if (link->srcpad && link->sinkpad)
	g_warning ("Settting link drag y without having one unset pad");

      link->dragging = TRUE;
      link->y = g_value_get_double (value);
      break;

    case PROP_SRCPAD:
      srcpad = (GstEditorPad *) link->srcpad;
      sinkpad = (GstEditorPad *) link->sinkpad;

      if (srcpad) {
	if (link->ghost)
	  srcpad->ghostlink = NULL;
	else
	  srcpad->link = NULL;
      }

      srcpad = (GstEditorPad *) g_value_get_object (value);

      if (srcpad) {
	if (link->ghost)
	  srcpad->ghostlink = link;
	else
	  srcpad->link = link;

	if (sinkpad)
	  link->dragging = FALSE;
      }

      link->srcpad = (GstEditorItem *) srcpad;
      break;

    case PROP_SINKPAD:
      srcpad = (GstEditorPad *) link->srcpad;
      sinkpad = (GstEditorPad *) link->sinkpad;

      if (sinkpad) {
	if (link->ghost)
	  sinkpad->ghostlink = NULL;
	else
	  sinkpad->link = NULL;
      }

      sinkpad = (GstEditorPad *) g_value_get_object (value);

      if (sinkpad) {
	if (link->ghost)
	  sinkpad->ghostlink = link;
	else
	  sinkpad->link = link;

	if (srcpad)
	  link->dragging = FALSE;
      }

      link->sinkpad = (GstEditorItem *) sinkpad;
      break;

    case PROP_GHOST:
      link->ghost = g_value_get_boolean (value);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }

  /* check if we're ready to resize */
  if (((link->srcpad || link->sinkpad) && link->dragging) ||
      (link->srcpad && link->sinkpad))
    gst_editor_link_resize (link);
}

static void
gst_editor_link_get_property (GObject * object, guint prop_id, GValue * value,
    GParamSpec * pspec)
{
  GstEditorLink *link = GST_EDITOR_LINK (object);
  gdouble d = 0.0, blah = 0.0;

  switch (prop_id) {
    case PROP_X:
      g_value_set_double (value, link->x);
      break;

    case PROP_Y:
      g_value_set_double (value, link->y);
      break;

    case PROP_X1:
      if (link->srcpad) {
	g_object_get (link->srcpad, "x", &d, NULL);
	d += link->srcpad->width;
	gnome_canvas_item_i2w (GNOME_CANVAS_ITEM (link->srcpad)->parent, &d,
	    &blah);
      } else if (link->dragging) {
	d = link->x;
      } else {
	g_warning ("no src pad");
      }
      g_value_set_double (value, d);
      break;

    case PROP_X2:
      if (link->sinkpad) {
	g_object_get (link->sinkpad, "x", &d, NULL);
	gnome_canvas_item_i2w (GNOME_CANVAS_ITEM (link->sinkpad)->parent, &d,
	    &blah);
      } else if (link->dragging) {
	d = link->x;
      } else {
	g_warning ("no sink pad");
      }

      g_value_set_double (value, d);
      break;

    case PROP_Y1:
      if (link->srcpad) {
	g_object_get (link->srcpad, "y", &d, NULL);
	d += link->srcpad->height / 2;
	gnome_canvas_item_i2w (GNOME_CANVAS_ITEM (link->srcpad)->parent, &blah,
	    &d);
      } else if (link->dragging) {
	d = link->y;
      } else {
	g_warning ("no src pad");
      }
      g_value_set_double (value, d);
      break;

    case PROP_Y2:
      if (link->sinkpad) {
	g_object_get (link->sinkpad, "y", &d, NULL);
	d += link->sinkpad->height / 2;
	gnome_canvas_item_i2w (GNOME_CANVAS_ITEM (link->sinkpad)->parent, &blah,
	    &d);
      } else if (link->dragging) {
	d = link->y;
      } else {
	g_warning ("no sink pad");
      }
      g_value_set_double (value, d);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}


static void
gst_editor_link_realize (GnomeCanvasItem * citem)
{
  GstEditorLink *link = GST_EDITOR_LINK (citem);

  link->points->coords[0] = 0.0;
  link->points->coords[1] = 0.0;
  link->points->coords[2] = 0.0;
  link->points->coords[3] = 0.0;

  /* we need to be realized before setting properties */
  if (GNOME_CANVAS_ITEM_CLASS (parent_class)->realize)
    GNOME_CANVAS_ITEM_CLASS (parent_class)->realize (citem);

  /* see gnome-canvas-line.h for the docs */
  gnome_canvas_item_set (citem,
      "points", link->points,
      "width-units", 2.0,
      "line-style", GDK_LINE_ON_OFF_DASH,
      "first-arrowhead", TRUE,
      "arrow-shape-a", 5.0, "arrow-shape-b", 5.0, "arrow-shape-c", 5.0, NULL);
  gnome_canvas_item_raise (citem, 10);
}

static void
gst_editor_link_resize (GstEditorLink * link)
{
  gdouble x1, y1, x2, y2;

  g_object_get (link, "x1", &x1, "y1", &y1, "x2", &x2, "y2", &y2, NULL);
  gnome_canvas_item_w2i (GNOME_CANVAS_ITEM (link)->parent, &x1, &y1);
  gnome_canvas_item_w2i (GNOME_CANVAS_ITEM (link)->parent, &x2, &y2);

  /* we do this in reverse so that with dot-dash lines it gives the illusion of
     pulling out a rope from the element */
  link->points->coords[2] = x1;
  link->points->coords[3] = y1;
  link->points->coords[0] = x2;
  link->points->coords[1] = y2;
  gnome_canvas_item_set (GNOME_CANVAS_ITEM (link),
      "points", link->points, NULL);
}

static void
on_pad_unlink (GstPad * pad, GstPad * peer, GstEditorLink * link)
{
  GstEditorBin *srcbin, *sinkbin;

  /* this function is called when unlinking dynamic links as well */
  if (peer && pad)
    EDITOR_DEBUG ("Unlink pad signal (%s:%s from %s:%s) with link %p",
	GST_DEBUG_PAD_NAME (pad), GST_DEBUG_PAD_NAME (peer), link);
  else
    EDITOR_DEBUG ("Unlinking dynamic link");

  g_signal_handlers_disconnect_by_func (link->srcpad,
      on_editor_pad_position_changed, link);
  g_signal_handlers_disconnect_by_func (link->sinkpad,
      on_editor_pad_position_changed, link);
  g_signal_handlers_disconnect_by_func (pad, on_pad_unlink, link);

  srcbin = GST_EDITOR_BIN (GNOME_CANVAS_ITEM (link->srcpad)->parent->parent);
  sinkbin = GST_EDITOR_BIN (GNOME_CANVAS_ITEM (link->sinkpad)->parent->parent);

  sinkbin->links = g_list_remove (sinkbin->links, link);
  if (sinkbin != srcbin)
    srcbin->links = g_list_remove (srcbin->links, link);

  GST_EDITOR_PAD (link->srcpad)->link = NULL;
  GST_EDITOR_PAD (link->sinkpad)->link = NULL;
  link->srcpad = NULL;
  link->sinkpad = NULL;
  /* i have bad luck with actually killing the GCI's */
  gnome_canvas_item_hide (GNOME_CANVAS_ITEM (link));
}

static void
on_editor_pad_position_changed (GstEditorPad * pad, GstEditorLink * link)
{
  g_return_if_fail (GST_IS_EDITOR_LINK (link));

  gst_editor_link_resize (link);
}

gboolean
gst_editor_link_link (GstEditorLink * link)
{
  GObject *src, *sink;
  GstPad *srcpad = NULL, *sinkpad = NULL;

  g_return_val_if_fail (GST_IS_EDITOR_LINK (link), FALSE);

  if (!link->srcpad || !link->sinkpad)
    goto error;

  src = (GObject *) GST_EDITOR_ITEM (link->srcpad)->object;
  sink = (GObject *) GST_EDITOR_ITEM (link->sinkpad)->object;

  if (!GST_EDITOR_PAD (link->srcpad)->istemplate) {
    if (!GST_EDITOR_PAD (link->sinkpad)->istemplate) {
      if (GST_PAD_PEER (src) || GST_PAD_PEER (sink)) {
	if (GST_PAD_PEER (src) != (GstPad *) sink) {
	  g_warning ("The src pad is linked, but not to the sink pad");
	  goto error;
	}
	if (GST_PAD_PEER (sink) != (GstPad *) src) {
	  g_warning ("The sink pad is linked, but not to the src pad");
	  goto error;
	}
	srcpad = GST_PAD (src);
	sinkpad = GST_PAD (sink);
	goto linked;
	/* yay goto */
      }
    } else if (GST_PAD_PEER (src)) {
      /* the to pad is a template */
      g_warning ("The src pad is linked, but not to the sink pad");
      goto error;
    }
  } else if (!GST_EDITOR_PAD (link->sinkpad)->istemplate && GST_PAD_PEER (sink)) {
    /* from pad is a template */
    g_warning ("The sink pad is linked, but not to the src pad");
    goto error;
  }

  if (link->ghost) {
    gnome_canvas_item_set (GNOME_CANVAS_ITEM (link), "fill-color",
	"grey70", NULL);
    goto linked;
  } else if (GST_IS_EDITOR_PAD_SOMETIMES (link->srcpad) ||
      GST_IS_EDITOR_PAD_SOMETIMES (link->sinkpad)) {
    make_dynamic_link (link);
    gnome_canvas_item_set (GNOME_CANVAS_ITEM (link), "fill-color",
	"grey50", NULL);
    goto linked;
  } else {
    if (!GST_EDITOR_PAD (link->srcpad)->istemplate) {
      srcpad = GST_PAD (src);
    } else if (GST_IS_EDITOR_PAD_REQUEST (link->srcpad)) {
      srcpad =
	  gst_element_get_request_pad ((GstElement *)
	  GST_EDITOR_ITEM (GNOME_CANVAS_ITEM (link->srcpad)->parent)->object,
	  GST_PAD_TEMPLATE (src)->name_template);
      /* the new_pad signal will cause a new pad to made automagically in the
         element */
      gnome_canvas_item_set (GNOME_CANVAS_ITEM (link), "src-pad",
	  gst_editor_item_get ((GstObject *) srcpad), NULL);
    } else {
      goto error;
    }

    if (!srcpad)
      goto error;

    if (!GST_EDITOR_PAD (link->sinkpad)->istemplate) {
      sinkpad = GST_PAD (sink);
    } else if (GST_IS_EDITOR_PAD_REQUEST (link->sinkpad)) {
      sinkpad =
	  gst_element_get_request_pad ((GstElement *)
	  GST_EDITOR_ITEM (GNOME_CANVAS_ITEM (link->sinkpad)->parent)->object,
	  GST_PAD_TEMPLATE (sink)->name_template);
      gnome_canvas_item_set (GNOME_CANVAS_ITEM (link), "sink-pad",
	  gst_editor_item_get ((GstObject *) sinkpad), NULL);
    } else {
      goto error;
    }

    if (!sinkpad)
      goto error;

    if (gst_pad_link (srcpad, sinkpad)) {
      GstEditorBin *srcbin, *sinkbin;

      gnome_canvas_item_set (GNOME_CANVAS_ITEM (link), "fill-color",
	  "black", NULL);

    linked:

      srcbin = sinkbin = NULL;

      gnome_canvas_item_set (GNOME_CANVAS_ITEM (link), "line-style",
	  GDK_LINE_SOLID, NULL);

      g_signal_connect (link->srcpad, "position-changed",
	  G_CALLBACK (on_editor_pad_position_changed), G_OBJECT (link));
      g_signal_connect (link->sinkpad, "position-changed",
	  G_CALLBACK (on_editor_pad_position_changed), G_OBJECT (link));

      /* don't connect a signal on a ghost or dynamic link */
      if (srcpad && sinkpad) {
	EDITOR_DEBUG ("link pad signal (%s:%s from %s:%s) with link %p",
	    GST_DEBUG_PAD_NAME (srcpad), GST_DEBUG_PAD_NAME (sinkpad), link);
	g_signal_connect (srcpad, "unlinked", G_CALLBACK (on_pad_unlink), link);
      }

      if (GST_IS_EDITOR_BIN (GNOME_CANVAS_ITEM (link->srcpad)->parent->parent))
	srcbin =
	    GST_EDITOR_BIN (GNOME_CANVAS_ITEM (link->srcpad)->parent->parent);
      if (GST_IS_EDITOR_BIN (GNOME_CANVAS_ITEM (link->sinkpad)->parent->parent))
	sinkbin =
	    GST_EDITOR_BIN (GNOME_CANVAS_ITEM (link->sinkpad)->parent->parent);

      if (sinkbin)
	sinkbin->links = g_list_prepend (sinkbin->links, link);
      if (srcbin && sinkbin != srcbin)
	srcbin->links = g_list_prepend (srcbin->links, link);

      return TRUE;
    }
  }

error:
  g_message ("could not link");

  if (link->srcpad)
    GST_EDITOR_PAD (link->srcpad)->link = NULL;
  if (link->sinkpad)
    GST_EDITOR_PAD (link->sinkpad)->link = NULL;
  return FALSE;
}

void
gst_editor_link_unlink (GstEditorLink * link)
{
  GST_EDITOR_PAD (link->srcpad)->unlinking = FALSE;
  GST_EDITOR_PAD (link->sinkpad)->unlinking = FALSE;

  if (link->ghost) {
    g_warning ("this function should not be called for ghost links..");
  } else if (!GST_EDITOR_PAD (link->srcpad)->istemplate &&
      !GST_EDITOR_PAD (link->sinkpad)->istemplate) {
    GstPad *src = NULL, *sink = NULL;

    g_object_get (link->srcpad, "object", &src, NULL);
    g_object_get (link->sinkpad, "object", &sink, NULL);

    gst_pad_unlink (src, sink);

    /* 'unlinked' signal takes care of calling on_pad_unlink */
  } else {
    /* we must have a dynamic link, then.. */

    g_signal_handlers_disconnect_by_func (link->srcpad->object,
	on_new_pad, link);
    g_signal_handlers_disconnect_by_func (link->sinkpad->object,
	on_new_pad, link);

    on_pad_unlink (NULL, NULL, link);
  }

  /* potential threadsafety issues here, can't assume that the signal has
   * already been handled. you might not even be able to assume you can
   * unlink the pads. maybe the pipeline should only be modifiable in
   * PAUSED? */

  /* but then again calling gst_pad_unlink on a PLAYING threaded element
     isn't safe anyway. i guess we're ok.. */

  /* 'link' is now invalid */
}

/* this function is only linked for dynamic links */
static void
on_new_pad (GstElement * element, GstPad * pad, GstEditorLink * link)
{
  GstPadTemplate *src = NULL, *sink = NULL;

  if (GST_IS_EDITOR_PAD_SOMETIMES (link->srcpad))
    src = GST_PAD_TEMPLATE (link->srcpad->object);

  if (GST_IS_EDITOR_PAD_SOMETIMES (link->sinkpad))
    sink = GST_PAD_TEMPLATE (link->sinkpad->object);

  g_message ("new pad");
  if (pad->padtemplate) {
    g_message ("from a template");

    /* can't do pointer comparison -- some templates appear to be from the
       elementfactories, some from template factories... have to compare
       names */
    if (src
	&& strcmp (pad->padtemplate->name_template, src->name_template) == 0)
      gnome_canvas_item_set (GNOME_CANVAS_ITEM (link), "src-pad",
	  gst_editor_item_get (GST_OBJECT (pad)), NULL);
    else if (sink
	&& strcmp (pad->padtemplate->name_template, sink->name_template) == 0)
      gnome_canvas_item_set (GNOME_CANVAS_ITEM (link), "sink-pad",
	  gst_editor_item_get (GST_OBJECT (pad)), NULL);
    else
      return;

    g_message ("we made it, now let's link");

    gst_element_set_state ((GstElement *)
	gst_element_get_managing_bin (element), GST_STATE_PAUSED);
    gst_editor_link_link (link);
    gst_element_set_state ((GstElement *)
	gst_element_get_managing_bin (element), GST_STATE_PLAYING);
  }
}

static void
make_dynamic_link (GstEditorLink * link)
{
  GstElement *srce, *sinke;
  GstPadTemplate *src = NULL, *sink = NULL;

  if (GST_IS_EDITOR_PAD_SOMETIMES (link->srcpad))
    src = GST_PAD_TEMPLATE (link->srcpad->object);

  if (GST_IS_EDITOR_PAD_SOMETIMES (link->sinkpad))
    sink = GST_PAD_TEMPLATE (link->sinkpad->object);

  srce =
      GST_ELEMENT (GST_EDITOR_ITEM (GNOME_CANVAS_ITEM (link->srcpad)->parent)->
      object);
  sinke =
      GST_ELEMENT (GST_EDITOR_ITEM (GNOME_CANVAS_ITEM (link->sinkpad)->parent)->
      object);

  g_return_if_fail (src || sink);

  if (src)
    g_signal_connect_after (srce, "new-pad", G_CALLBACK (on_new_pad), link);
  if (sink)
    g_signal_connect_after (sinke, "new-pad", G_CALLBACK (on_new_pad), link);

  g_print ("dynamic link\n");
}
