/*
 * Copyright (C) 1997,98 Free Software Foundation
 * Copyright (C) 1994,95,96 Eric M. Ludlam
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, you can either send email to this
 * program's author (see below) or write to:
 *
 *              The Free Software Foundation, Inc.
 *              675 Mass Ave.
 *              Cambridge, MA 02139, USA. 
 *
 * Please send bug reports, etc. to zappo@gnu.org
 *
 * Purpose:
 *   This file contains device functions for handling IO on all types
 * of descriptors.  It manages a static list of all descriptors and
 * their attributes, has a generic selector, and then reads in that
 * data and calls the appropriate object functions.  Also available
 * are the functions sending output to a given object, reguardless of
 * it's type.
 *
 * $Log: gtl_dev.c,v $
 * Revision 1.29  1998/04/28 22:35:55  zappo
 * Added support for a quicktimer.  This timer provides a very fast
 * timeout without updating IO structures.  This is used by the GTK
 * interface to handle it's funny timout requirements.
 *
 * Revision 1.28  1998/02/15 14:16:02  zappo
 * Replaced use of gets w/ fgets.
 *
 * Revision 1.27  1998/01/04 13:33:51  zappo
 * Fixed warnings
 *
 * Revision 1.26  1997/12/14 19:21:09  zappo
 * Renamed package to gtalk, renamed symbols and files apropriately
 * Fixed copyright and email address.
 *
 * Revision 1.25  1997/10/18 03:06:08  zappo
 * changed select loops to not use NULL for infinite wait, but to just
 * use 60 seconds.  The loops are timeout nothing safe.  This fixes problem
 * on HPUX where select returns immediatly for some reason.
 * As such, fixed select_one to reset the return flag because of the timeout.
 *
 * Revision 1.24  1997/10/15 02:10:32  zappo
 * Added new status variable.  Keeps track of when an IO loop needs to start
 * over.  It gets set to DEV_Reset whenever the main list of IO devices gets
 * modified.
 *
 * Revision 1.23  1997/07/23 01:22:12  zappo
 * Changed to use the list library.
 *
 * Revision 1.22  1997/07/11 21:28:06  zappo
 * Modified to use the new generic list structure
 *
 * Revision 1.21  1997/03/23 14:50:52  zappo
 * In selection main loop, the current loop variable is cached, and then
 * advanced, and the cached value is used when calling it's callbacks.  That
 * way a callback can destroy it's structure, and the main loop will survive.
 *
 * Revision 1.20  1997/02/23 03:24:53  zappo
 * API change on GT_clean_dev changes.  It now takes a CLEAN flag which determnes
 * if GT_clean is also called.
 *
 * Revision 1.19  1996/03/02 03:26:56  zappo
 * Removed extraneous variable
 *
 * Revision 1.18  1996/02/01  02:43:37  zappo
 * Fixed close_all to close everything EXCEPT a specified port, and added
 * many pipe related things throughout
 *
 * Revision 1.17  1995/12/10  15:49:01  zappo
 * Fixed some comments
 *
 * Revision 1.16  1995/12/10  03:41:17  zappo
 * Changed all error messages to use DISP_message
 *
 * Revision 1.15  1995/11/27  00:06:07  zappo
 * GT_select_one resets RECURSIVE_EXIT before exiting, as this is the
 * last stop when exiting in most cases.
 *
 * Revision 1.14  1995/11/21  03:54:12  zappo
 * Added outfile IO type support, removed useless debug code, added
 * GT_select_one to block other threads during a wait/read.
 *
 * Revision 1.13  1995/07/16  16:18:45  zappo
 * Fixed a typo in a print message
 *
 * Revision 1.12  1995/04/08  20:02:43  zappo
 * Removed memcmp and replaced with direct tests of a structs fields
 *
 * Revision 1.11  1995/04/01  18:58:49  zappo
 * Added cast for buffer + rin in gtl_dev so that std_c likes it.
 *
 * Revision 1.10  1995/03/30  02:35:22  zappo
 * now handle case where fd_set is not a defined typedef.
 *
 * Revision 1.9  1995/03/25  04:12:09  zappo
 * Updated copyright.
 *
 * Revision 1.8  1995/03/04  14:46:30  zappo
 * Added use of syslog to report errors for use in daemon.
 *
 * Revision 1.7  1995/02/12  14:01:24  zappo
 * Added check for EINTR error return from select in main-loop. This
 * occurs when a sigchild handler is called in gtalkd.
 *
 * Revision 1.6  1995/02/11  17:19:33  zappo
 * Made all etl code require VERBOSE flag to be set before printing.
 *
 * Revision 1.5  1994/12/07  23:19:46  zappo
 * Added loop when writing to a TCP socket to make sure everything gets
 * sent before exiting the write command.
 *
 * Revision 1.4  1994/11/19  17:06:13  zappo
 * Moved empty state to UNUSED state.
 *
 * Revision 1.3  1994/11/15  04:11:25  zappo
 * Moved use of select.h and sys/select.h from general headers to this
 * file, since this is the only place it is used.
 *
 * Revision 1.2  1994/11/13  21:10:36  zappo
 * Added extra return case in main-loop.  Also fixed many comment spacing
 * areas to make it more compatible with cparse auto-comment
 * generation/fixing routines.
 *
 * Revision 1.1  1994/08/29  23:42:31  zappo
 * Initial revision
 *
 * ::Header:: gtalklib.h
 */

#include "gtalklib.h"

/* Select.h is an aix thing, and includes the FD_* routines needed
 * to do selecting on that system
 */
#if HAVE_SELECT_H == 1
#include <select.h>
#endif /* HAVE_SELECT_H */

#if HAVE_SYS_SELECT_H == 1
#include <sys/select.h>
#endif /* HAVE_SYS_SELECT_H */

#if HAVE_ERRNO_H == 1
#include <errno.h>
#endif /* HAVE_ERRNO_H */
/* Not all errno's declair the integer errno!  Hrumph. */
extern int errno;

#if HAVE_FCNTL_H == 1
#include <fcntl.h>
#else
/* What to put here if I do not have fcntl?? */
#endif

#if HAVE_SYS_IOCTL_H
#include "sys/ioctl.h"
#endif

/* check if this system has FD_SET defined (a macro for setting 
 * descriptors in a selection mask.  These values were taken from
 * "process.c" in the emacs-19.28 distribution.
 */
#ifndef FD_SET
#define fd_set unsigned long
#define FD_ZERO(p) (*(p) = 0)
#define FD_SET(n, p) (*(p) |= (1 << (n)))
#define FD_ISSET(n, p) (*(p) & (1 << (n)))
#endif /* not FD_SET */

static MakeList(list);

/* If a reader mucks with the main list of devices, return DEV_Reset,
 * otherwise DEV_Ok.  On a DEV_Reset, stop input processing, and
 * reinitialize the selection loop.
 */
enum DEV_ReadStatus { DEV_Reset, DEV_Ok };

enum DEV_ReadStatus status;	/* return status of dynamic readers      */

static int recursive_exit = FALSE;

static void (*quicktimer)() = NULL;
static int quickpause = 0;
static struct InputDevice *quickio = NULL;


/*
 * Function: GT_select_all
 *
 * Main loop to the whole program.  Simply wait for input on any of
 * our great list of sockets, and when something shows up, run the
 * associated readme function.
 * Timeouts are handled on a one second granularity basis, and the IO
 * struct is modified over time untill a timeout occurs.
 * Return value simply is Success if something comes in, and Fail if not.
 * 
 * Parameters:  Ctxt  - context of the program to pass to READMEs Must
 *                      be NULL because lib fns don't include etalk.
 *              reton - return on a match of this value.  Good for UDP stuff.
 * History:
 * eml     3/1/94
 * eml     4/6/94    Decrementing timeouts added, and reton match.
 * zappo   9/14/94   Added check for no devices waiting to auto-return.
 */
int GT_select_all(Ctxt, reton)
     void               *Ctxt;
     struct InputDevice *reton;
{
  struct InputDevice *Q;	/* Q loop variable                       */
  fd_set              mask;	/* mask used in select function          */
  int                 rval;	/* return value of some functions        */
  int                 tocnt;	/* count of timeouts encountered         */
  struct timeval      timeout;	/* timeout passed into select            */
  int                 returnflag = FALSE; /* flag for returning on match */

  while(1) {			/* loop forever on devices   */

#ifdef DEBUG
    LIST_verify();		/* Sanity Check for lists    */
#endif

    if(reton && recursive_exit)	/* check recursive exit flag */
      return Fail;		/* return that instant!      */
    else
      recursive_exit = FALSE;

    status = DEV_Ok;		/* Reset status as we enter the loop */
    FD_ZERO(&mask);		/* init mask                         */
    rval = 0;			/* init return val/ fd #             */
    tocnt = 0;			/* init timeout count                */
    Q = FirstElement(list);	/* setup for loop                    */
    while(Q) {
      /* Only add to mask if dev has a reader function and is not dead.
       */
      if((Q->readme) && (Q->state != DEAD)) {
	FD_SET(Q->fd, &mask);
	/* get greatest file descriptor for select.
	 */
	if(rval <= Q->fd) 
	  rval = Q->fd + 1;
	if(Q->timeout > 0)
	  tocnt++;
      }
      Q = NextElement(Q);
    }
    
    /* check that we have something to do!
     */
    if(rval == 0)
      return Fail;

    if(quickpause)
      {
	timeout.tv_sec = 0;
	timeout.tv_usec = 10;
      }
    else if(tocnt)
      {
	timeout.tv_sec = 1;
	timeout.tv_usec = 0;
      }
    else
      {
	timeout.tv_sec = 60;
	timeout.tv_usec = 0;
      }
    /* Select with one second delay...
     */
    rval = select(rval, &mask, NULL, NULL, &timeout);

    if(rval < 0) {
      /* Don't abandon hope if we are interrupted. */
      if(errno == EINTR)
	continue;

      DISP_message(NULL, "select() main loop failure.", LOG_CRIT);
      return Fail;
    }
    if(rval == 0) {
      if(quickpause)
	{
	  quickpause = 0;
	  quicktimer(Ctxt, quickio);
	  /* This was a quickie... do it again... */
	  continue;
	}
      /* Loop on each item, and decrement their timeout counters...
       */
      Q = FirstElement(list);
      while(Q) {
	/* Cache Q into QI and advance Q now.  This is because callback fns
	 * can actually delete the instance QI, removing it from the loop.
	 * and this protects against referencing unused memory.
	 */
	struct InputDevice *QI = Q;
	Q = NextElement(Q);

	if((QI->timeout > 0) && (QI->state != DEAD))
	  {

	    QI->timeout--;
	    if(!QI->timeout) 
	      {
		if(QI->timefn)
		  {
		    QI->timefn(Ctxt, QI);
		  } 
		/* Check for match on reton variable
		 */
		if(QI == reton) returnflag = TRUE;
	      }
	    /* No timeout, see if there is something to do each sec.
	     */
	    else
	      {
		if(QI->timemsg)
		  {
		    QI->timemsg(Ctxt, QI);
		    /* Reset if the timemsg messed with us */
		  }
	      }
	  }
	if(status == DEV_Reset) break;
	QI = NextElement(QI);
      }
      if(returnflag) return Fail;
    } else {
      Q = FirstElement(list);
      while(Q) {
	/* Cache Q into QI and advance Q now.  This is because callback fns
	 * can actually delete the instance QI, removing it from the loop.
	 * and this protects against referencing unused memory.
	 */
	struct InputDevice *QI = Q;
	Q = NextElement(Q);

	if((QI->readme) && (QI->state != DEAD))
	  {	    
	    if(FD_ISSET(QI->fd, &mask)) {/* check for each socket w/ io avail*/
	      QI->readme(Ctxt, QI); /* and run the catch program.   */
	      if(QI == reton) returnflag = TRUE;/* return if waiting...      */
	      if(status == DEV_Reset) break; /* break if devices are deleted */
	    } 
	  }
      }
      if(returnflag) return Success;
    }
  }
}


/*
 * Function: GT_select_one
 *
 *   Waits on _only one_ io device for some response.  Loops forever
 * on this one device until RECURSIVE_EXIT is non-0.  If you look on
 * these functions are providing primitive threads, this could be
 * considered as stopping other thread execution to prevent conflict.
 *
 * Returns:     int  - 
 * Parameters:  Ctxt - Context
 *              dev  - Pointer to device
 * History:
 * zappo   11/18/95   Created
 */
int GT_select_one(Ctxt, dev)
     void *Ctxt;
     struct InputDevice *dev;
{
  fd_set              mask;	/* mask used in select function          */
  int                 rval;	/* return value of some functions        */
  int                 tocnt;	/* count of timeouts encountered         */
  struct timeval      timeout;	/* timeout passed into select            */
  int                 returnflag = FALSE; /* flag for returning on match */

  recursive_exit = FALSE;	/* make sure we go through this loop
				 * at least once */

  while(1) {			/* loop forever on devices   */

    if(recursive_exit)		/* check recursive exit flag */
      {
	recursive_exit = FALSE;	/* In these cases, only back out this level */
	return returnflag;	/* return that instant!      */
      }
    returnflag = FALSE;		/* Reset this each time.     */

    FD_ZERO(&mask);		/* init mask               */
    rval = 0;			/* init return val/ fd #   */
    tocnt = 0;			/* init timeout count      */
    if((dev->readme) && (dev->state != DEAD)) {
      FD_SET(dev->fd, &mask);
      /* get greatest file descriptor for select.
       */
      if(rval <= dev->fd) 
	rval = dev->fd + 1;
      if(dev->timeout > 0)
	tocnt++;
    }
    
    /* check that we have something to do!
     */
    if(rval == 0)
      {
	return Fail;
      }

    if(tocnt)
      {
	timeout.tv_sec = 1;
	timeout.tv_usec = 0;
      }
    else
      {
	timeout.tv_sec = 60;
	timeout.tv_usec = 0;
      }
    /* Select with one second delay...
     */
    rval = select(rval, &mask, NULL, NULL, &timeout);

    if(rval < 0) {
      /* Don't abandon hope if we are interrupted. */
      if(errno == EINTR)
	continue;
      DISP_message(NULL, "select() one main loop failure", LOG_CRIT);
      return Fail;
    }
    if(rval == 0) {
      /* Loop on each item, and decrement their timeout counters...
       */
      if((dev->timeout > 0) && (dev->state != DEAD))
	{
	  dev->timeout--;
	  if(!dev->timeout) 
	    {
	      if(dev->timefn)
		{
		  dev->timefn(Ctxt, dev);
		} 
	      returnflag = TRUE;
	    }
	  /* No timeout, see if there is something to do each sec.
	   */
	  else
	    {
	      if(dev->timemsg)
		dev->timemsg(Ctxt, dev);
	    }
	}
      /* Return Fail because we had a timeout waiting for something to
       * happen. */
      if(returnflag) return Fail;
    } else {
      if((dev->readme) && (dev->state != DEAD))
	  {
	    if(FD_ISSET(dev->fd, &mask)) {
	      dev->readme(Ctxt, dev);
	      returnflag = TRUE;
	    } 
	  }
      /* other loop returns on match, we dont return until we get
       * an explicit request to exit via recursion! */
      /* if(returnflag) return Success; */
    }
  }
}


/*
 * Function: GT_end_recursion
 *
 * Set a local variable so that any recursive calls to select_all
 * are canceled right away upon return.
 * 
 * Parameters: none
 *
 * History:
 * eml 4/15/94
 */
void GT_end_recursion()
{
  recursive_exit = TRUE;
}


/*
 * Function: GT_set_quicktimer
 *
 *   Add for one itteration a fast timer that then calls the
 * specified pointer with Ctxt and the specified IO device.
 *
 * Returns:     Nothing
 * Parameters:  fn -  The function to call.
 *              dev - Pointer  device
 * History:
 * zappo   4/28/98    Created
 */
void GT_set_quicktimer(fn, dev)
     void (*fn)();
     struct InputDevice *dev;
{
  quicktimer = fn;
  quickpause = 1;
  quickio = dev;
}


/*
 * Function: GT_clean_dev
 *
 * Do bookeeping on a device when it is shut down.
 * 
 * Parameters: io    - the device
 *             clean - Not 0 if we also cal GT_clean
 *
 * History:
 * eml 4/15/94
 */
void GT_clean_dev(io, clean)
     struct InputDevice *io;
     int                 clean;	/* 1 if we should also call GT_clean */
{
  /* ignore any nulls, make other procs much simpler */
  if(!io) return;
  /* never close the TTY parts by accident */
  if(io->fd <= 2) return;
  /* We should check here for file, and <maybe> delete it. */

  io->state = DEAD;
  close(io->fd);
  io->fd = 1;			/* if anyone write, goes to stdout   */
  io->readme = NULL;		/* and remove any functions on it.   */

  if(clean != 0) GT_clean();	/* auto-clean if requested           */
}

/*
 * Function: GT_close_all
 *
 *   Close all ports in the Q list.  There should be several
 * duplicates, but close doesn't care too much.  Never close type TTY
 * as we may need it when either shutting down, or as a sub-process
 * PIPEd out.  Maybe save a device specified in SAVE if we are forking
 * children inheriting devices.
 * 
 * Parameters: Ctxt - unused context field.
 *             save - input device to not close
 * History:
 * eml 3/1/94 */
void GT_close_all(Ctxt, save)
     void *Ctxt;
     struct InputDevice *save;
{
  struct InputDevice *Q;

  Q = FirstElement(list);
  while(Q) {
    if((Q->state != DEAD) && (Q != save) && (Q->type != IO_TTY))
      if(Q->type == IO_X) {
	/* Do nothing? */
      } else {
	close(Q->fd);
      }
    Q = NextElement(Q);
  }
}

/*
 * Function: GT_clean, IsDead
 *
 * Go through all devices and free all ports labelled as DEAD
 * 
 * IsDead is a predicate used for the List Management code.
 *
 * Parameters:
 *
 * History:
 * eml 4/21/94
 */
static unsigned char IsDead(io, criteria)
     struct InputDevice *io;
     void *criteria;
{
  return (io->state == DEAD);
}

void GT_clean()
{
  struct InputDevice *io;

  while((io = (struct InputDevice *)LIST_find(&list, IsDead, NULL)) != NULL)
    {
      /* Free the name if there is one. */
      if(io->name != NULL) free(io->name);
      LIST_deallocate(&list, io);
      status = DEV_Reset;	/* Signal the calling loop the reset */
    }
}

/*
 * Function: GT_gen_iodev
 *
 * Malloc space for one InputDevice, and fill in the fields.
 * 
 * Returns:     struct InputDevice * - New IO device
 * Parameters:  type  - udp,tcp,tty
 *              fd    - Number of fd
 *              raddr - remote address (to be sent to.)
 * History:
 * eml 3/1/93 
 * zappo   9/14/94         Devices with empty parts are give ANY
 *                         address for it's remote address.
 */
struct InputDevice *GT_gen_iodev(type, fd, raddr)
     enum InputDeviceType type;
     int                  fd;
     struct sockaddr_in  *raddr;
{
  struct InputDevice *new;
  
  /* first, don't keep allocating new structures.  Only for UDP for now
   */
  if(type == IO_UDP)
    {
      new = FirstElement(list);
      while(new)
	{
	  if((new->type == type) && (new->fd == fd) && raddr &&
	     (new->raddr.sin_family == raddr->sin_family) &&
	     (new->raddr.sin_port == raddr->sin_port) &&
	     (new->raddr.sin_addr.s_addr == raddr->sin_addr.s_addr))
	    break;
	  new = NextElement(new);
	}
      if(new) 
	{
	  return new;
	}
    }


  status = DEV_Reset;		/* Signal the calling loop the reset */

  new = (struct InputDevice *)LIST_alloc(&list, sizeof(struct InputDevice));

  if(!new)
    return Fail;

  new->type     =  type;
  new->state    =  UNUSED;
  new->name     =  NULL;
  new->sendc    =  0;
  new->recvc    =  0;
  if(raddr)
    new->raddr  = *raddr;
  else
    {
      /* modify address to accept from anybody.
       */
      new->raddr.sin_family = AF_INET;
      new->raddr.sin_port = 0;
      new->raddr.sin_addr.s_addr = INADDR_ANY;
    }
  new->fd       =  fd;
  new->host     =  NULL;
  new->readme   =  NULL;
  new->timeout  =  0;
  new->timefn   =  NULL;
  new->timemsg  =  NULL;

  return new;
}     


/*
 * Function: GT_portable_address
 *
 * Return a network sendable address struct from a file descriptor.
 * 
 * Parameters: dev - The input device with the desciptor attached.
 *
 * History:
 * eml 4/1/94
 */
struct sockaddr_in GT_portable_address(dev)
     struct InputDevice *dev;
{
  int                t;
  struct sockaddr_in addr;
  /*
   * and do the get name thing.
   */
  t = sizeof(addr);
  if(getsockname(dev->fd, (struct sockaddr *)&addr, &t) != 0) 
    {
      DISP_message(NULL, "getsockname", LOG_ERR);
      exit(1);
    }
  return addr;
}


/*
 * Function: GT_tty
 *
 * Simply generate an Input struct from the TTY. No addressing and
 * the like should be used.
 * 
 * Parameters:
 *
 * History:
 * eml 3/1/94
 */
struct InputDevice *GT_tty()
{
  static struct InputDevice *tty = NULL;

  if(tty) return tty;

  tty = GT_gen_iodev(IO_TTY, 0, NULL);
  
  tty->state = CONNECTED;
  tty->name = strdup("TTY");

  return tty;
}


/*
 * Function: GT_pipe
 *
 *   Creates an InputDevice based on a file descriptor which contains
 * a unix pipe.  The name of the app started under this pipe is also
 * passed in to create the name of the device.
 *
 * Returns:     struct InputDevice * - 
 * Parameters:  pipefd  - Number of pipefd
 *              appname - Name of the app being piped
 *              direct  - Direction of the pipe
 * History:
 * zappo   1/7/96     Created
 */
struct InputDevice *GT_pipe(pipefd, appname, direct)
     int pipefd;
     char *appname;
     char *direct;
{
  static char *namefmt = "Pipe %s %s";
  struct InputDevice *newpipe;

  newpipe = GT_gen_iodev(IO_PIPE, pipefd, NULL);

  newpipe->state = CONNECTED;
  newpipe->name = (char *)malloc(strlen(appname) + strlen(namefmt) + strlen(direct));
  sprintf(newpipe->name, namefmt, direct, appname);

  return newpipe;
}


/*
 * Function: GT_open_output_file
 *
 *   Opens a file of name NAME.  Return new file descriptor, or NULL
 * on failure.
 *
 * Returns:     struct InputDevice * - 
 * Parameters:  name - Name of
 *
 * History:
 * zappo   11/6/95    Created
 */
struct InputDevice *GT_open_output_file(name)
     char *name;
{
  static struct InputDevice *file = NULL;
  int f_fd;

  if(!name || (strlen(name) < 1))
    {
      return NULL;
    }

  /* First, we don't want to overwrite a file, so open read to see if
     it's there */
  f_fd = open(name, O_RDONLY);

  if(f_fd != -1)
    {
      DISP_message(NULL, "gt_open_output_file: file %s already exists.",
		   LOG_ERR);
      close(f_fd);
      return NULL;
    }

  /* On linux, last perameters is modes for created file.  Not certain
     of it's fields on other systems yet. */
  f_fd = open(name, O_WRONLY | O_CREAT, 0644);

  if(f_fd == -1)
    {
      if(verbose)
	perror(name);
      return NULL;
    }

  file = GT_gen_iodev(IO_FILE, f_fd, NULL);
  
  file->state = CONNECTED;
  file->name = strdup(name);

  if(!file->name)
    DISP_message(NULL, "GT_open_output_file: strdup(name)", LOG_ERR);
  
  return file;
}


/*
 * Function: GT_send
 *
 * Take an Input device, which contains a type, and send something to
 * them based on the type (udp, tcp etc).
 * 
 * Parameters: dev - Input device
 *             buffer - the buffer to send.
 *             size - the size of the buffer to send.
 * History:
 * eml 3/1/94
 */
int GT_send(dev, buffer, size)
     struct InputDevice *dev;
     void *buffer;
     int size;
{
  switch(dev->type) {
  case IO_TTY:
    if(write(1, buffer, size) < 0) {
      DISP_message(NULL, "GT_send: tty write", LOG_CRIT);
      return Fail;
    }
    dev->sendc += size;
    return Success;
  case IO_PIPE:
  case IO_TCP:
    {
      int rin = 0;		/* chars read in */
      
      /* Loop until all buffer parts have been sent.  This fixes old
       * talk programs and slow networks who can't keep up.
       */
      do {
	if((rin = write(dev->fd, (char *)buffer + rin, size - rin)) < 0) {
	  DISP_message(NULL, "GT_send: tcp write", LOG_CRIT);
	  return Fail;
	}
      } while (rin < size);

      dev->sendc += size;
      return Success;
    }
  case IO_UDP:
    if(sendto(dev->fd, buffer, size, 0, (struct sockaddr *)&dev->raddr, 
	      sizeof(struct sockaddr_in)) < 0) {
      DISP_message(NULL, "GT_send: udp sendto", LOG_CRIT);
      return Fail;
    }
    dev->sendc += 1;		/* count in terms of packets */
    return Success;
  case IO_FILE:
    if(write(dev->fd, buffer, size) < 0) {
      DISP_message(NULL, "GT_send: file write", LOG_CRIT);
      return Fail;
    }
    dev->sendc += size;
    return Success;
  default:
    DISP_message(NULL, "GT_send: device has wrong type", LOG_ERR);
  }
  return Fail;
}

/*
 * Function: GT_recv
 *
 * Universal read function for input devices.  Allows some special
 * dealins with UDP which can receive from someone other than the
 * selected individual.
 * 
 * Parameters:  dev - the input device
 *              buffer - the buffer to receive into
 *              size - size of said buffer
 * History:
 * eml     3/1/94   commented
 * zappo   9/17/94  UDP requests now store last sender's address
 */
int GT_recv(dev, buffer, size)
     struct InputDevice *dev;
     void *buffer;
     int size;
{
  int sinsize;
  int retsize;

  switch(dev->type) {
    /* TTY and PIPEs are buffered io, and will have CRs in them */
  case IO_TTY:
    {
      char *cr;
      if((buffer = fgets(buffer, size, stdin)) == 0) {
	DISP_message(NULL, "GT_recv: tty fgets:", LOG_CRIT);
	return Fail;
      }
      /* buffer is NULL terminated safe: */
      cr = strchr(buffer, '\n');
      if(cr) *cr = '\0';
      dev->recvc += strlen(buffer);
      return strlen(buffer);
    }
  case IO_TCP:
  case IO_PIPE:
    if((retsize = read(dev->fd, buffer, size)) < 0) {
      DISP_message(NULL, "GT_recv: tcp read:", LOG_CRIT);
      return Fail;
    }
    dev->recvc += retsize;
    return retsize;
  case IO_UDP:
    {
      struct sockaddr recvaddr;
      sinsize = sizeof(struct sockaddr_in);

      recvaddr = *(struct sockaddr *)&dev->raddr;

      if((retsize = recvfrom(dev->fd, buffer, size, 0, 
			     &recvaddr, &sinsize)) < 0) {
	DISP_message(NULL, "GT_recv: udp recvfrom:", LOG_CRIT);
	return Fail;
      }
      /* store last received general addrss into the structure.
       */
      dev->lraddr = *(struct sockaddr_in *)&recvaddr;
    }
    dev->recvc += 1;		/* in terms of packets. */
    return retsize;
  case IO_FILE:
    DISP_message(NULL, "GT_recv: Type file not allowed to recv", LOG_ERR);
    break;
  default:
    DISP_message(NULL, "GT_recv: device has wrong type", LOG_ERR);
  }
  return Fail;
}


/*
 * Function: GT_dev_name
 *
 * Take an input device, and return the appropriate name based on info
 * available.
 * 
 * Parameters: io - the input device.
 *
 * History:
 * eml 4/12/94
 */
char *GT_dev_name(io)
     struct InputDevice *io;
{
  static char *typestr[] = { "TTY", "TCP", "UDP", "X", "FILE", "PIPE" };
  static char short_term_buffer[180];

  if(!io) return "<NULL>";

  if(io->name) return io->name;

  if(io->host && io->host->name) 
    {
      sprintf(short_term_buffer, "%s to %s",
	      typestr[io->type], io->host->name);
      return short_term_buffer;
    }
  
  sprintf(short_term_buffer, "%s on descriptor %d",
	  typestr[io->type], io->fd);
  return short_term_buffer;
}


/*
 * Function: GT_dev_num
 *
 *   Returns the number of managed devices.
 *
 * Returns:     int  - 
 * Parameters:  None
 *
 * History:
 * zappo   12/7/95    Created
 */
int GT_dev_num()
{
  return LIST_map(&list, NULL, NULL);
}


/*
 * Function: GT_print_device
 *
 * Prints out the given input device with printer functions.
 *
 * Returns:     void  - 
 * Parameters:  io - the input device to print
 *
 * History:
 * eml	May 19, 1994	Created
 */
void GT_print_device(io)
     struct InputDevice *io;
{
  static char *typestr[] = { "TTY", "TCP", "UDP", "X", "FILE", "PIPE" };
  static char *state[] = { "EMPTY", "CONNECT", "LISTEN", "WAIT", "IDLE",
			     "DEAD" };
  char buff[200];

  sprintf(buff, "%-8.15s\t%s\t%s\t%d\t%s\t%ld\t%ld\t%s",
	 GT_dev_name(io),
	 typestr[io->type],
	 state[io->state],
	 io->fd, io->readme?"TRUE":"FALSE",
	 io->sendc, io->recvc,
	 (io->host && io->host->name)?
	 io->host->name:"<NULL>");  
  DISP_message(NULL, buff, LOG_DEBUG);
}


/*
 * Function: GT_print_q_list
 * 
 * print a formatted list of all open input devices with thier names.
 * 
 * Parameters: None
 *
 * History:
 * eml 4/15/94
 */
void GT_print_q_list()
{
  DISP_message(NULL, 
	       "Name\t\tType\tState\tDescptr\tActive\tSent\tRecv\tHostname",
	       LOG_DEBUG);

  LIST_map(&list, GT_print_device, NULL);
}

/*
 * Function: print_sockaddr
 *
 * print out a socket address.  For debugging purposes.
 * 
 * Parameters: addr - the addr to print
 *
 * History:
 * eml 3/1/94
 */
void print_sockaddr(addr)
     struct sockaddr *addr;
{
  printf("F:(%d) P:(%d) IN:(%d)", 
	 ((struct sockaddr_in *)addr)->sin_family, 
	 /* ports are alwas in net order...
	  */
	 ntohs(((struct sockaddr_in *)addr)->sin_port),
	 ((struct sockaddr_in *)addr)->sin_addr.s_addr);
}

/*
 * Function: print_swapped_sockaddr
 *
 * Prints out the sockaddr which is in network order by rotating the ints.
 *
 * Returns:     void  - 
 * Parameters:  addr - address to print
 *
 * History:
 * eml	May 19, 1994	Created
 */
void print_swapped_sockaddr(addr)
     struct sockaddr *addr;
{
  printf("F:(%d) P:(%d) IN:(%d)", 
	 ntohs(((struct sockaddr_in *)addr)->sin_family),
	 ntohs(((struct sockaddr_in *)addr)->sin_port),
	 ((struct sockaddr_in *)addr)->sin_addr.s_addr);
}
