/* gtl_x.c - Basic X routines to interface with libetalk
 *
 * Copyright (C) 1994, 1995, 1996 Eric M. Ludlam
 * Copyright (C) 1997 Free Software Foundation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, you can either send email to this
 * program's author (see below) or write to:
 *
 *              The Free Software Foundation, Inc.
 *              675 Mass Ave.
 *              Cambridge, MA 02139, USA. 
 *
 * Please send bug reports, etc. to zappo@gnu.org.
 *
 * Purpose:
 *   This file contains some functions used to link X into the device handling
 * structures managed by the etalklibrary.
 *
 * $Log: gtl_x.c,v $
 * Revision 1.11  1998/01/04 13:34:29  zappo
 * Fixed warnings
 *
 * Revision 1.10  1997/12/14 19:22:02  zappo
 * Renamed package to gtalk, renamed symbols and files apropriately
 * Fixed copyright and email address.
 *
 * Revision 1.9  1997/10/18 03:07:40  zappo
 * Fixed unix_socket identification to be more robust.
 *
 * Revision 1.8  1997/10/15 02:10:57  zappo
 * Made sure X IO name is malloced space.
 *
 * Revision 1.7  1997/01/15 23:16:26  zappo
 * Switched from using X private struct parts to XConnectionNumber
 *
 * Revision 1.6  1996/03/02  03:29:59  zappo
 * Fixed some warnings
 *
 * Revision 1.5  1995/12/10  03:49:03  zappo
 * smegged a comment
 *
 * Revision 1.4  1995/12/10  03:48:26  zappo
 * Changed all error messages to use DISP_message
 *
 * Revision 1.3  1995/06/13  00:03:56  zappo
 * Removed a rogue simicolon
 *
 * Revision 1.2  1995/05/27  01:38:17  zappo
 * Added GTL_ggt_resource for reading random resources from any X app.
 *
 * Revision 1.1  1995/04/19  01:28:06  zappo
 * Initial revision
 *
 * ::Header:: gtalklib.h
 */
/* In order to get at display parts, we need to announce that we are
 * trying to get at it's private parts.
 */
#include <X11/Xlib.h>
#include <X11/Xresource.h>
#include "gtalklib.h"


/*
 * Function: GTL_x
 *
 *   This routine will an X display pointer, and return an etalk IO
 * device which can be used to poll the X connection in the central
 * loop function.
 *
 * Returns:     struct InputDevice * - 
 * Parameters:  display - Pointer to display
 *
 * History:
 * zappo   5/26/95    Created
 */
struct InputDevice *GTL_x(display)
     Display *display;
{
  static struct InputDevice *X;
  struct sockaddr raddr;
  int unix_socket = False;
  int t;

  if(X) return X;

  /* Find address of our display */
  t = sizeof(struct sockaddr_in);		/* how big is it */

  if(getsockname(XConnectionNumber(display), &raddr, &t) != 0) 
    {
      unix_socket = True;
    }

  if(unix_socket == False)
    /* We have an address, so use that. */
    X = GT_gen_iodev(IO_X, XConnectionNumber(display),
		     (struct sockaddr_in *)&raddr);
  else
    /* We are attached locally, and not through unix port. */
    X = GT_gen_iodev(IO_X, XConnectionNumber(display), NULL);

  X->state = CONNECTED;
  X->name  = strdup("X_display");
  if(unix_socket == False)
    X->host = HOST_gen_host_by_addr((struct sockaddr *)&X->raddr, t);
  else
    X->host = HOST_gen_local_host();

  return X;
}


/*
 * Function: GTL_get_resource
 *
 *   Looks for the given resources in the X resource database.
 *
 * Returns:     int  - 
 * Parameters:  display  - Pointer to display
 *              resource - Pointer toCharacter of resource
 *              value    -  value
 * History:
 * zappo   5/26/95    Created
 */
int GTL_get_resource(display, name, class, value)
     Display *display;
     char *name, *class;
     XrmValue *value;
{
  static at_least_once = 0;
  static XrmDatabase homeDB;

  char  *str_type[20];

  if(! at_least_once)
    {
      XrmDatabase serverDB;
      char         fnamebuf[1024];

      if(XResourceManagerString(display) != NULL)
	{
	  serverDB = XrmGetStringDatabase(XResourceManagerString(display));
	} else {
	  /* Grab Xdefaults instead */
	  if(getenv("HOME"))
	    strcpy(fnamebuf, getenv("HOME"));
	  else
	    /* No hope of getting any resources now! */
	    return FALSE;

	  strcat(fnamebuf, "/.Xdefaults");

	  serverDB = XrmGetFileDatabase(fnamebuf);
	}

      if(getenv("XENVIRONMENT"))
	{
	  strcpy(fnamebuf, getenv("XENVIRONMENT"));
	}
      else
	{
	  int len;

	  /* Grab Xdefaults-machine instead */
	  if(getenv("HOME"))
	    strcpy(fnamebuf, getenv("HOME"));
	  else
	    /* No hope of getting any resources now! */
	    return FALSE;

	  strcat(fnamebuf, "/.Xdefaults-");
	  len = strlen(fnamebuf);
	  gethostname(fnamebuf+len, sizeof(fnamebuf)-len);
	}	  

      homeDB = XrmGetFileDatabase(fnamebuf);
      XrmMergeDatabases(serverDB, &homeDB);

      at_least_once = 1;
    }

  if(verbose)
    printf("Trying to read %s from environment.\n", name);

  /*
   * Now extract the asked for string!
   */
  return XrmGetResource(homeDB, name, class, str_type, value);
  
}
