/* doodlenet.c - Network management for doodle app
 *
 * Copyright (C) 1996 Eric M. Ludlam
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, you can either send email to this
 * program's author (see below) or write to:
 * 
 *              The Free Software Foundation, Inc.
 *              675 Mass Ave.
 *              Cambridge, MA 02139, USA. 
 * 
 * Please send bug reports, etc. to zappo@gnu.ai.mit.edu.
 * 
 * 
 * Description:
 *   Handles network connection via passed in port.  This code will
 * send commands or parse incomming connads based on XtInput.
 *   
 * $Log: doodlenet.c,v $
 * Revision 1.3  1997/03/23 15:59:18  zappo
 * Standardized on eraser/dot size, and converted errors to print to
 * stdout.
 *
 * Revision 1.2  1996/03/17 15:22:01  zappo
 * Added comments to header
 *
 * History:
 * zappo   3/8/96     Created
 *
 * Tokens: ::Header:: doodle.h
 */
#include "config.h"
#include "headers.h"

/* check if this system has FD_SET defined (a macro for setting 
 * descriptors in a selection mask.  These values were taken from
 * "process.c" in the emacs-19.28 distribution.
 */
#ifndef FD_SET
#define fd_set unsigned long
#define FD_ZERO(p) (*(p) = 0)
#define FD_SET(n, p) (*(p) |= (1 << (n)))
#define FD_ISSET(n, p) (*(p) & (1 << (n)))
#endif /* not FD_SET */

#include <X11/Xlib.h>
#include <X11/keysym.h>

/* Some Athena widgets we need */
#include <X11/StringDefs.h>
#include <X11/IntrinsicP.h>

#include "doodle.h"

/* This contains the opened socket.  Commands passed between will
 * be command ending in CR, so fgets will be used.
 */
static FILE *sockopen;
static XtInputId xid;


/*
 * Function: dnet_init
 *
 *   Initializes the doodle net connection coming in on the port
 * defined by PORT.  This should be the char string passed in on the
 * command line.
 *
 * Returns:     Nothing
 * Parameters:  port  - Pointer toCharacter of port
 *              XCtxt - Context
 * History:
 * zappo   3/8/96     Created
 */
void dnet_init(port, XCtxt)
     char *port;
     struct Xcontext *XCtxt;
{
  int sock;

  sock = atoi(port);

  if(sock == 0)
    {
      fprintf(stderr, "Invalid socket number: %s\n", port);
      fprintf(stderr, "Usage: doodle <sock fd>\n");
      exit(1);
    }

  /* Verify that mysock really is an IO device */
  sockopen = fdopen(sock, "wr");

  if(!sockopen)
    {
      fprintf(stderr, "File Descriptor %d is not valid.\n", sock);
      exit(1);
    }

  /* Turn on this socket for X to give us pieces */
  xid = XtAppAddInput(XCtxt->app_context, sock, (XtPointer)XtInputReadMask,
		      (XtInputCallbackProc)dnet_read_sock, XCtxt);
  
  /* Write an initstring to our socket */
  fprintf(sockopen, "INIT BLACKWHITE\n");
}


/*
 * Function: dnet_send_draw_command
 *
 *   Sends a drawing command with coordinates.  Some shapes have more
 * coordinates than others, so they are passed in as unbounded arrays.
 *
 * Returns:     Nothing
 * Parameters:  tool      - Pointer toCharacter of tool
 *              color     - Pointer toCharacter of color
 *              numcoords - Number of number
 *              x         - Number of x
 *              y         - Number of y
 * History:
 * zappo   3/8/96     Created
 */
void dnet_send_draw_command(tool, color, numcoords, x, y)
     char *tool;
     char *color;
     int numcoords;
     int x[], y[];
{
  int i;

  fprintf(sockopen, "DRAW %s %s ", tool, color);
  for(i = 0; i<numcoords; i++)
    {
      fprintf(sockopen, "%d %d ", x[i], y[i]);
    }
  fprintf(sockopen, "\n");
  fflush(sockopen);
}


/*
 * Function: dnet_read_sock
 *
 *   Reads from a socket and creates the apropraite shapes in our
 * doodle area
 *
 * Returns:     Nothing
 * Parameters:  XCtxt - Context
 *              src   - Pointer toNumber of src
 *              lxid  - Pointer to lxid
 * History:
 * zappo   3/8/96     Created
 */
void dnet_read_sock(XCtxt, src, lxid)
     struct Xcontext *XCtxt;
     int *src;
     XtInputId *lxid;
{
  char buff[1000];
  char *parse, *parse2;
  int i, c, x[100], y[100];

  if(read(*src, buff, sizeof(buff)) == 0)
    {
      printf("Connection to remote closed!\n");
      XtRemoveInput(xid);
      return;
    }

  parse = strtok(buff, " \n");

  if(!parse)
    {
      printf("Failed to read first keyword!\n");
      return;
    }

  if(!strcmp("INIT", parse))
    {
      printf("Init recieved!\n");
      return;
    }
  if(!strcmp("DRAW", parse))
    {
      parse2 = strtok(NULL, "  \n");

      if(!parse2)
	{
	  printf("Failed to read tool name!\n");
	  return;
	}

      parse = strtok(NULL, " \n");
      
      if(!parse)
	{
	  printf("Failed to read color name!\n");
	  return;
	}

      for(c = 0; (c < MAXCOLOR) && strcmp(parse, color_names[c]); c++);

      i = 0;
      while(parse)
	{
	  parse = strtok(NULL, " \n");
	  if(!parse) break;
	  x[i] = atoi(parse);
	  parse = strtok(NULL, " \n");
	  if(!parse && (x[i] == 0))
	    break;
	  else if(!parse)
	    {
	      printf("Read X, but no Y!\n");
	      break;
	    }
	  y[i] = atoi(parse);
	  i++;
	}
      
      if(!strcmp(tool_names[Point], parse2))
	{
	  XFillArc(XCtxt->display, XCtxt->doodlePix,  
		   XCtxt->gcs[c], x[0]-DOTSIZE/2, y[0]-DOTSIZE/2, 7, 7, 0, 360*64);
	}
      else if(!strcmp(tool_names[Line], parse2))
	{
	  XDrawLine(XCtxt->display, XCtxt->doodlePix,
		     XCtxt->gcs[c], x[0], y[0], x[1], y[1]);
	}
      else if(!strcmp(tool_names[Scribble], parse2))
	{
	  XPoint *xp = malloc(sizeof(XPoint) * i);
	  int j;

	  if(!xp)
	    {
	      fprintf(stderr, "Malloc failure!\n");
	      exit(1);
	    }

	  for(j = 0; j < i; j++)
	    {
	      xp[j].x = x[j];
	      xp[j].y = y[j];
	    }
	  XDrawLines(XCtxt->display, XCtxt->doodlePix,
		     XCtxt->gcs[c], xp, i, CoordModeOrigin);
	  free(xp);
	}
      else if(!strcmp(tool_names[Rectangle], parse2))
	{
	  XDrawRectangle(XCtxt->display, XCtxt->doodlePix, 
			 XCtxt->gcs[c], MIN(x[0],x[1]), MIN(y[0],y[1]),
			 ABS(x[1]-x[0]), ABS(y[1]-y[0]));
	}
      else if(!strcmp(tool_names[Elipse], parse2))
	{
	  XDrawArc(XCtxt->display, XCtxt->doodlePix, 
		   XCtxt->gcs[c], MIN(x[0],x[1]), MIN(y[0],y[1]),
		   ABS(x[1]-x[0]), ABS(y[1]-y[0]), 0, 64*360);
	}
      else if(!strcmp(tool_names[Eraser], parse2))
	{
	  XFillArc(XCtxt->display, XCtxt->doodlePix, XCtxt->white,
		   x[0]-ERASERSIZE/2, y[0]-ERASERSIZE/2, 21, 21, 0, 360*64);
	}
      else
	{
	  printf("Unrecognized tool command %s\n", parse);
	}
      X_expose_doodle(XCtxt->doodleCanvas, XCtxt, NULL);
     
    }
  else
    {
      printf("Unrecognized command %s\n", parse);
      return;
    }

}
