/*
 * group.c
 * - group authentication file stuff
 * 
 * Copyright (c) 1999 Jack Moffitt, Barath Raghavan, and Alexander Havng
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *  
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *  
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#ifdef HAVE_CONFIG_H
#ifdef _WIN32
#include <win32config.h>
#else
#include <config.h>
#endif
#endif

#include "definitions.h"

#include <stdio.h>
#include "definitions.h"

#ifdef HAVE_SIGNAL_H
#include <signal.h>
#endif

#include <string.h>
#include <sys/types.h>
#include <time.h>
#include <stdlib.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifndef _WIN32
#include <sys/socket.h>
#include <netinet/in.h>
#endif

/* Can someone verify this? */
#ifndef W_OK
#define W_OK 2
#endif

#include "avl.h"
#include "threads.h"
#include "icetypes.h"
#include "icecast.h"
#include "utility.h"
#include "ice_string.h"
#include "connection.h"
#include "log.h"
#include "sock.h"
#include "avl_functions.h"
#include "restrict.h"
#include "match.h"
#include "memory.h"
#include "commands.h"
#include "admin.h"
#include "basic.h"
#include "group.h"
#include "user.h"

extern server_info_t info;

extern mutex_t authentication_mutex;
extern mounttree_t *mounttree;
extern usertree_t *usertree;
extern grouptree_t *grouptree;

void parse_group_authentication_file()
{
	int fd;
	char *groupfile = get_icecast_file(info.groupfile, conf_file_e, R_OK);
	group_t *group;
	char line[BUFSIZE];

	if (!groupfile || ((fd = open_for_reading(groupfile)) == -1)) {
		if (groupfile)
			nfree(groupfile);
		xa_debug(1, "WARNING: Could not open group authentication file");
		return;
	}
	while (fd_read_line(fd, line, BUFSIZE)) {
		if (line[0] == '#' || line[0] == ' ')
			continue;

		group = create_group_from_line(line);

		if (group)
			add_authentication_group(group);
	}

	if (groupfile)
		nfree(groupfile);
	fd_close(fd);
}

group_t *
 create_group_from_line(char *line)
{
	group_t *group;
	ice_user_t *user;
	char name[BUFSIZE], cuser[BUFSIZE];
	int go_on = 1;

	if (!line) {
		xa_debug(1, "WARNING: create_group_from_line() called with NULL pointer");
		return NULL;
	}
	if (!splitc(name, line, ':')) {
		xa_debug(1, "ERROR: Syntax error in group file, with line [%s]", line);
		return NULL;
	}
	group = create_group();

	group->name = nstrdup(clean_string(name));

	do {
		if (!splitc(cuser, line, ',')) {
			strcpy(cuser, line);
			go_on = 0;
		}
		user = find_user_from_tree(usertree, clean_string(cuser));

		if (!user) {
			write_log(LOG_DEFAULT, "WARNING: Unrecognized user [%s] specified for group [%s]",
				  cuser, name);
		} else {
			avl_insert(group->usertree, user);
		}
	} while (go_on);

	return group;
}

group_t *
 create_group()
{
	group_t *group = (group_t *) nmalloc(sizeof (group_t));

	group->usertree = create_user_tree();
	group->name = NULL;
	return group;
}

grouptree_t *
 create_group_tree()
{
	grouptree_t *gt = avl_create(compare_groups, &info);

	return gt;
}

void add_authentication_group(group_t * group)
{
	group_t *out;

	if (!group || !grouptree || !group->name || !group->usertree) {
		xa_debug(1, "ERROR: add_authentication_group() called with NULL pointers");
		return;
	}
	out = avl_replace(grouptree, group);

	if (out) {
		write_log(LOG_DEFAULT, "WARNING: Duplicate group record %s, using latter", group->name);
		nfree(out->name);
		free_user_tree(out->usertree);
		nfree(out);
	}
	xa_debug(1, "DEBUG: add_authentication_group(): Inserted group [%s]", group->name);
}

void free_group_tree(grouptree_t * gt)
{
	avl_traverser trav =
	{0};
	group_t *group, *out;

	if (!gt)
		return;

	while ((group = avl_traverse(gt, &trav))) {
		out = avl_delete(gt, group);

		if (!out) {
			xa_debug(1, "WARNING: Weird ass things going on in grouptree!");
			continue;
		}
		nfree(group->name);
		avl_destroy(group->usertree, NULL);
		nfree(group);
	}
}

int is_member_of(char *user, group_t * group)
{
	ice_user_t *up;
	ice_user_t search;

	if (!user || !group || !group->usertree) {
		xa_debug(1, "WARNING: is_member_of() called with NULL pointers");
		return 0;
	}
	search.name = user;

	up = avl_find(group->usertree, &search);

	return up ? 1 : 0;
}

group_t *
 find_group_from_tree(grouptree_t * gt, const char *name)
{
	group_t search;

	search.name = strchr(name, name[0]);

	if (!gt || !name) {
		xa_debug(1, "WARNING: find_group_from_tree() called with NULL pointers");
		return NULL;
	}
	return avl_find(gt, &search);
}

void con_display_groups(com_request_t * req)
{
	avl_traverser trav =
	{0};
	avl_traverser usertrav =
	{0};
	group_t *group;
	ice_user_t *user;
	int listed = 0;

	admin_write_line(req, ADMIN_SHOW_AUTH_GROUP_START, "Listing groups in the authentication module:");

	thread_mutex_lock(&authentication_mutex);

	while ((group = avl_traverse(grouptree, &trav))) {
		zero_trav(&usertrav);

		admin_write(req, ADMIN_SHOW_AUTH_GROUP_ENTRY, "%s: ", group->name ? group->name : "(null)");

		while ((user = avl_traverse(group->usertree, &usertrav)))
			admin_write(req, -1, "%s ", user->name);
		admin_write_line(req, -1, "");
		listed++;
	}

	thread_mutex_unlock(&authentication_mutex);

	admin_write_line(req, ADMIN_SHOW_AUTH_GROUP_END, "End of group listing (%d listed)", listed);
}


int runtime_add_group(const char *name)
{
	char line[BUFSIZE];
	char *groupfile;
	int fd;

	if (!name || !name[0])
		return ICE_ERROR_INVALID_SYNTAX;
#ifdef _WIN32
	sprintf(line, "%s:\r\n", name);
#else
	sprintf(line, "%s:\n", name);
#endif

	thread_mutex_lock(&authentication_mutex);

	if (find_group_from_tree(grouptree, name)) {
		thread_mutex_unlock(&authentication_mutex);
		return ICE_ERROR_DUPLICATE;
	}
	groupfile = get_icecast_file(info.groupfile, conf_file_e, W_OK);

	if (!groupfile || ((fd = open_for_append(groupfile)) == -1)) {
		if (groupfile)
			nfree(groupfile);
		xa_debug(1, "WARNING: Could not open group authentication file for writing");
		thread_mutex_unlock(&authentication_mutex);
		return ICE_ERROR_FILE;
	}
	fd_write_line(fd, "%s", line);
	fd_close(fd);
	if (groupfile)
		nfree(groupfile);
	thread_mutex_unlock(&authentication_mutex);
	return 1;
}


int runtime_add_group_with_user(const char *name, char *users)
{
	char line[BUFSIZE];
	char *groupfile, *s;
	int fd;

	if (!name || !users || !name[0] || !users[0])
		return ICE_ERROR_INVALID_SYNTAX;

	while ((s = strchr(users, ' ')))
		*s = ',';

#ifdef _WIN32
	sprintf(line, "%s:%s\r\n", name, users);
#else
	sprintf(line, "%s:%s\n", name, users);
#endif

	thread_mutex_lock(&authentication_mutex);

	if (find_group_from_tree(grouptree, name)) {
		thread_mutex_unlock(&authentication_mutex);
		return ICE_ERROR_DUPLICATE;
	}
	groupfile = get_icecast_file(info.groupfile, conf_file_e, W_OK);

	if (!groupfile || ((fd = open_for_append(groupfile)) == -1)) {
		if (groupfile)
			nfree(groupfile);
		xa_debug(1, "WARNING: Could not open group authentication file for writing");
		thread_mutex_unlock(&authentication_mutex);
		return ICE_ERROR_FILE;
	}
	fd_write_line(fd, "%s", line);
	fd_close(fd);
	if (groupfile)
		nfree(groupfile);
	thread_mutex_unlock(&authentication_mutex);
	return 1;
}
