/*
Copyright 1990-2003 Sun Microsystems, Inc. All Rights Reserved.

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions: The above copyright notice and this
permission notice shall be included in all copies or substantial
portions of the Software.


THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE OPEN GROUP OR SUN MICROSYSTEMS, INC. BE LIABLE
FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH
THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE EVEN IF
ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH DAMAGES.


Except as contained in this notice, the names of The Open Group and/or
Sun Microsystems, Inc. shall not be used in advertising or otherwise to
promote the sale, use or other dealings in this Software without prior
written authorization from The Open Group and/or Sun Microsystems,
Inc., as applicable.


X Window System is a trademark of The Open Group

OSF/1, OSF/Motif and Motif are registered trademarks, and OSF, the OSF
logo, LBX, X Window System, and Xinerama are trademarks of the Open
Group. All other trademarks and registered trademarks mentioned herein
are the property of their respective owners. No right, title or
interest in or to any trademark, service mark, logo or trade name of
Sun Microsystems, Inc. or its licensors is granted.

*/

#include <stdio.h>
#include "dayi_data.h"

#define MAX_LINE_LEN    256

#define BASE_DIR_STRING         "base_dir"
#define DATA_FILE_STRING        "data_file"
#define SYMBOL_LIST_STRING      "symbol_list"

#define BASE_DIR_NAME_STRING    "path"
#define DATA_FILE_NAME_STRING   "file"

enum {
	DAYI_DEFAULT_SECTION = 0,
	DAYI_BASE_DIR_SECTION,
	DAYI_DATA_FILE_SECTION,
	DAYI_SYMBOL_LIST_SECTION,
};

/**************************************************************
The format of Dayi configure file as follow:

[ base_dir ]
path=/usr/lib/im/locale/zh_TW

[ data_file ]
file=data/dayi.data

[ symbol_list ]
key  DayiSymbolList

**************************************************************/

char *skip_space(char *s)
{
        while (*s && (*s == ' ' || *s == '\t')) s++;
        return s;
}

char *to_space(char *s)
{
	while (*s && !(*s == ' ' || *s == '\t')) s++;
        return s;
}

char *trim_string(char *s)
{
	char *head_ptr, *tail_ptr;

	if (!s || !*s) return(s);

	head_ptr = s;

	/* trim left space */
	while(*head_ptr && (*head_ptr == ' ' || *head_ptr == '\t' || *head_ptr == '\n'))
		head_ptr ++;

	if (!*head_ptr) return(head_ptr);

	/* trim right space */
	tail_ptr = head_ptr + strlen(head_ptr) - 1;
	
	while(*tail_ptr && (*tail_ptr == ' ' || *tail_ptr == '\t' || *tail_ptr == '\n')) {
		*tail_ptr = '\0';
		tail_ptr --;
	}

	return(head_ptr);
}

int parse_line_for_pair(char *line_buf, char **key_ptr, char **value_ptr)
{
	char *ptr, *kptr, *vptr;

	ptr = skip_space(line_buf);
	if (*ptr == '\\') ptr++;
	*key_ptr = ptr;

	while(*ptr && !(*ptr==' ' || *ptr=='\t' || *ptr=='\n' || *ptr=='=')) ptr++;

	if (*ptr=='=') {
		*ptr = '\0';
		ptr = skip_space(ptr+1);
	} else if (*ptr==' ' || *ptr=='\t') {
		*ptr = '\0';
		ptr = skip_space(ptr+1);
		if (*ptr=='=') {
			ptr = skip_space(ptr+1);
		}
	} else if (*ptr=='\n') {
		*ptr = '\0';
	}

	/* skip " */
	if (*ptr == '\"') ptr++;
	*value_ptr = ptr;
	
	ptr = to_space(ptr);
	if (*(ptr-1) == '\"') ptr--;
	*ptr = '\0';

	return(0);
}

int parse_line_for_section_flag(TDayiData *pDayiData, char *line_buf)
{
	char *flag_section_str;

	int flag_section = DAYI_DEFAULT_SECTION;

	/* Extract the name of section */
	line_buf[strlen(line_buf) - 1] = '\0';
	flag_section_str = trim_string(line_buf+1);

	if (!*flag_section_str) return(DAYI_DEFAULT_SECTION);

	/* get flag_section */
	if (!(strncasecmp(flag_section_str, BASE_DIR_STRING, strlen(BASE_DIR_STRING)))) {
		flag_section = DAYI_BASE_DIR_SECTION;
	} else if (!(strncasecmp(flag_section_str, DATA_FILE_STRING, strlen(DATA_FILE_STRING)))) {
		flag_section = DAYI_DATA_FILE_SECTION;
	} else if (!(strncasecmp(flag_section_str, SYMBOL_LIST_STRING, strlen(SYMBOL_LIST_STRING)))) {
		flag_section = DAYI_SYMBOL_LIST_SECTION;
	}

	return(flag_section);
}

int parse_line_for_basedir_section(TDayiData *pDayiData, char *line_buf, char **pBaseDir)
{
	char *key_ptr, *value_ptr;

	parse_line_for_pair(line_buf, &key_ptr, &value_ptr);

	if (!strcasecmp(key_ptr, BASE_DIR_NAME_STRING)) {
		if (*pBaseDir != NULL) free((char *)*pBaseDir);
		*pBaseDir = (char *)strdup(value_ptr);
	}
	return(DAYI_OK);
}

int parse_line_for_datafile_section(TDayiData *pDayiData, char *line_buf, char **pDataFile)
{
	char *key_ptr, *value_ptr;

	parse_line_for_pair(line_buf, &key_ptr, &value_ptr);

	if (!strcasecmp(key_ptr, DATA_FILE_NAME_STRING)) {
		if (*pDataFile != NULL) free((char *)*pDataFile);
		*pDataFile = (char *)strdup(value_ptr);
	}
	return(DAYI_OK);
}

int parse_line_for_symbollist_section(TDayiData *pDayiData, char *line_buf)
{
	char *key_ptr, *value_ptr;
	int nSymbol_ID;

	parse_line_for_pair(line_buf, &key_ptr, &value_ptr);

	if (!*key_ptr || !*value_ptr)
		return(DAYI_ERROR);

	if (pDayiData->pDayiSymbolList->nNum_Symbols == 0) {
		pDayiData->pDayiSymbolList->nNum_Symbols ++;
		DayiSymbolList_Item_Prepare(pDayiData->pDayiSymbolList, 0);
		DayiSymbolList_Item_Set_SymbolList(pDayiData->pDayiSymbolList, 0, 0x20, "　");
	}

	pDayiData->pDayiSymbolList->nNum_Symbols ++;
	nSymbol_ID = pDayiData->pDayiSymbolList->nNum_Symbols - 1;
	DayiSymbolList_Item_Prepare(pDayiData->pDayiSymbolList, nSymbol_ID);
	DayiSymbolList_Item_Set_SymbolList(pDayiData->pDayiSymbolList, nSymbol_ID, *key_ptr, value_ptr);

	return(DAYI_OK);
}

DayiData_Read_Config(char *file_name, TDayiData *pDayiData)
{
	FILE *ifile;
	char line_buf[MAX_LINE_LEN], *line_ptr;
	int flag_section;
	char *pBaseDir = NULL;
	char *pDataFile = NULL;

	ifile = fopen(file_name, "r");
	if (ifile == NULL) {
		fprintf(stderr, "Error: open file %s\n", file_name);
		return(DAYI_ERROR);
	}

	while(fgets(line_buf, MAX_LINE_LEN, ifile) != NULL) {
		line_ptr = trim_string(line_buf);

		if (*line_ptr && line_ptr[0] == '[' && line_ptr[strlen(line_ptr)-1] == ']') {
			flag_section = parse_line_for_section_flag(pDayiData, line_ptr);
			continue;
		}

		/* If space line or comment line */
		if (*line_ptr == '\0' || *line_ptr == '#') continue;

		switch (flag_section) {
			case DAYI_DEFAULT_SECTION:
				break;

			case DAYI_BASE_DIR_SECTION:
				parse_line_for_basedir_section(pDayiData, line_ptr, &pBaseDir);
				break;

			case DAYI_DATA_FILE_SECTION:
				parse_line_for_datafile_section(pDayiData, line_ptr, &pDataFile);
				break;

			case DAYI_SYMBOL_LIST_SECTION:
				parse_line_for_symbollist_section(pDayiData, line_ptr);
				break;
		}
	}

	pDayiData->data_file = NULL;
	if (pBaseDir && pDataFile) {
		char file_name[MAX_LINE_LEN];
		sprintf(file_name, "%s/%s", pBaseDir, pDataFile);
		pDayiData->data_file = (char *)strdup(file_name);
	}

	if (pBaseDir) free((char *)pBaseDir);
	if (pDataFile) free((char *)pDataFile);

	fclose(ifile);

	return(DAYI_OK);
}
