/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Takuro Ashie
 *  Copyright (C) 2004 Hiroyuki Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "prefs_history.h"

#include <string.h>
#include <glib/gi18n.h>
#include "kazehakase.h"
#include "utils/gtk-utils.h"
#include "utils/estsearch.h"


#define DATA_KEY "KzPrefsHistory::info"


static GtkWidget     *prefs_history_create      (void);
static void           prefs_history_response    (GtkWidget *widget,
						 gint       response);


static KzPrefsWinPageEntry prefs_entry =
{
	path:            N_("/History"),
	priority_hint:   0,
	ui_level:        ~KZ_UI_LEVEL_BEGINNER,
	create:          prefs_history_create,
	response:        prefs_history_response,
};


typedef struct _KzPrefsHistory
{
	GtkWidget     *main_vbox;
	GtkWidget     *store_cache_check;
	GtkWidget     *limit_days_check;
	GtkWidget     *limit_days_spin;
	GtkWidget     *optimize_label;
	GtkWidget     *optimize_button;
	GtkWidget     *max_results;
	GtkWidget     *num_summary;
	gboolean       changed;
} KzPrefsHistory;


static void
cb_store_cache_toggled(GtkToggleButton *button, KzPrefsHistory *prefsui)
{
	gtk_widget_set_sensitive(prefsui->limit_days_check,
				 gtk_toggle_button_get_active(button));
	gtk_widget_set_sensitive(prefsui->limit_days_spin,
				 gtk_toggle_button_get_active(button));
	prefsui->changed = TRUE;
}

static void
cb_limit_days_toggled(GtkToggleButton *button, KzPrefsHistory *prefsui)
{
	gtk_widget_set_sensitive(prefsui->limit_days_spin,
				 gtk_toggle_button_get_active(button));
	prefsui->changed = TRUE;
}

static void
cb_changed (GtkWidget *widget, KzPrefsHistory *prefsui)
{
	prefsui->changed = TRUE;
}

static void
cb_optimize_exit (GPid pid, gint status, gpointer data)
{
	g_spawn_close_pid(pid);

	gtk_widget_set_sensitive(GTK_WIDGET(data), TRUE);
}

static void
cb_optimize_button_clicked (GtkButton *button, KzPrefsHistory *prefsui)
{
	GPid pid;
	
	gtk_widget_set_sensitive(GTK_WIDGET(button), FALSE);

	pid = estsearch_optimize_index();
	g_child_watch_add(pid,
			  cb_optimize_exit,
			  button);
}


static void
prefs_history_destroy (gpointer data)
{
	KzPrefsHistory *prefsui = data;

	g_free(prefsui);
}


static GtkWidget *
prefs_history_create (void)
{
	KzPrefsHistory *prefsui = g_new0(KzPrefsHistory, 1);
	GtkWidget *main_vbox, *vbox, *hbox, *frame;
	GtkWidget *label, *check, *spin, *button;
	GtkAdjustment *adj;
	gboolean limit, store_cache = TRUE;
	gint limit_days, max_results, num_summary;

	main_vbox = gtk_vbox_new(FALSE, 0);
	prefsui->main_vbox = main_vbox;
	g_object_set_data_full(G_OBJECT(main_vbox), DATA_KEY,
			       prefsui, prefs_history_destroy);

	label = kz_prefs_ui_utils_create_title(_("History"));
	gtk_box_pack_start(GTK_BOX(main_vbox), label,
			   FALSE, FALSE, 0);
	gtk_widget_show(label);


	/*
	 *  History Cache 
	 */
	KZ_CONF_GET("History", "store_cache", store_cache, BOOL);
	prefsui->store_cache_check = check =
		gtk_check_button_new_with_label(_("Store history cache for search"));
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check),
				     store_cache);
	g_signal_connect(check, "toggled",
			 G_CALLBACK(cb_store_cache_toggled), prefsui);
	frame = gtk_frame_new("");
	gtk_frame_set_label_widget(GTK_FRAME(frame), check);
	gtk_frame_set_label_align(GTK_FRAME(frame), 0.03, 0.5);
	gtk_container_set_border_width(GTK_CONTAINER(frame), 5);
	gtk_box_pack_start(GTK_BOX(main_vbox), frame, FALSE, FALSE, 2);
	gtk_widget_show(frame);
	gtk_widget_show(check);

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 4);
	gtk_container_add(GTK_CONTAINER(frame), vbox);
	gtk_widget_show(vbox);

	limit = TRUE;
	KZ_CONF_GET("History", "limit_cache", limit, BOOL);

	limit_days = 30;
	KZ_CONF_GET("History", "limit_days", limit_days, INT);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(hbox), 4);
	gtk_container_add(GTK_CONTAINER(vbox), hbox);
	gtk_widget_show(hbox);

	check = gtk_check_button_new_with_mnemonic(_("_Limit days of keeping the history cache"));
	prefsui->limit_days_check = check;
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check), limit);
	gtk_widget_set_sensitive(check, store_cache);
	gtk_box_pack_start(GTK_BOX(hbox), check, FALSE, FALSE, 2);
	g_signal_connect(check, "toggled",
			 G_CALLBACK(cb_limit_days_toggled), prefsui);
	gtk_widget_show(check);

	prefsui->limit_days_spin = spin
		= gtk_spin_button_new_with_range(1, 1000, 1);
	adj = gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(spin));
	gtk_adjustment_set_value(adj, limit_days);
	gtk_box_pack_start(GTK_BOX(hbox), spin, FALSE, FALSE, 2);
	gtk_widget_set_sensitive(spin, limit && store_cache);
	g_signal_connect(spin, "value-changed",
			 G_CALLBACK(cb_changed), prefsui);
	gtk_widget_show(spin);

	/*
	 * optimize index
	 */
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(hbox), 4);
	gtk_container_add(GTK_CONTAINER(vbox), hbox);
	gtk_widget_show(hbox);

	prefsui->optimize_label = label
		= gtk_label_new(_("Optimize history cache"));
	gtk_widget_set_sensitive(label, store_cache);
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 2);
	gtk_widget_show(label);

	prefsui->optimize_button = button
		= gtk_button_new_with_label (_("start"));
	gtk_widget_set_sensitive(button, store_cache);
	gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 4);
	g_signal_connect(button, "clicked",
			 G_CALLBACK(cb_optimize_button_clicked), prefsui);
	gtk_widget_show(button);

	/*
	 * History Search
	 */
	frame = gtk_frame_new(_("Full-text search in history"));
	gtk_frame_set_label_align(GTK_FRAME(frame), 0.03, 0.5);
	gtk_container_set_border_width(GTK_CONTAINER(frame), 5);
	gtk_box_pack_start(GTK_BOX(main_vbox), frame, FALSE, FALSE, 2);
	gtk_widget_show(frame);

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 4);
	gtk_container_add(GTK_CONTAINER(frame), vbox);
	gtk_widget_show(vbox);
	
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(hbox), 4);
	gtk_container_add(GTK_CONTAINER(vbox), hbox);
	gtk_widget_show(hbox);
	max_results = 20;
	KZ_CONF_GET("History", "max_results", max_results, INT);
	
	label = gtk_label_new_with_mnemonic(_("The _maximum number of shown results"));
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 2);
	gtk_widget_show(label);
	prefsui->max_results = spin
		= gtk_spin_button_new_with_range(-1, 1000, 1);
	adj = gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(spin));
	gtk_adjustment_set_value(adj, max_results);
	gtk_box_pack_start(GTK_BOX(hbox), spin, FALSE, FALSE, 2);
	g_signal_connect(spin, "value-changed",
			 G_CALLBACK(cb_changed), prefsui);
	gtk_widget_show(spin);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(hbox), 4);
	gtk_container_add(GTK_CONTAINER(vbox), hbox);
	gtk_widget_show(hbox);
	num_summary = 128;
	KZ_CONF_GET("History", "num_summary", num_summary, INT);
	
	label = gtk_label_new_with_mnemonic(_("The number of words in _summary"));
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 2);
	gtk_widget_show(label);
	prefsui->num_summary = spin
		= gtk_spin_button_new_with_range(-1, 1000, 1);
	adj = gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(spin));
	gtk_adjustment_set_value(adj, num_summary);
	gtk_box_pack_start(GTK_BOX(hbox), spin, FALSE, FALSE, 2);
	g_signal_connect(spin, "value-changed",
			 G_CALLBACK(cb_changed), prefsui);
	gtk_widget_show(spin);

	prefsui->changed = FALSE;

	return main_vbox;
}


static void
prefs_history_response (GtkWidget *widget, gint response)
{
	KzPrefsHistory *prefsui = g_object_get_data(G_OBJECT(widget), DATA_KEY);

	g_return_if_fail(prefsui);

	switch (response) {
	case GTK_RESPONSE_ACCEPT:
	case GTK_RESPONSE_APPLY:
	{
		GtkAdjustment *adj;
		gint limit_days, num_summary, max_results;
		gboolean limit, store_cache;

		if (!prefsui->changed)
			break;

		store_cache = gtk_toggle_button_get_active
				(GTK_TOGGLE_BUTTON(prefsui->store_cache_check));
		KZ_CONF_SET("History", "store_cache", store_cache, BOOL);
		
		limit = gtk_toggle_button_get_active
				(GTK_TOGGLE_BUTTON(prefsui->limit_days_check));
		KZ_CONF_SET("History", "limit_cache", limit, BOOL);

		adj = gtk_spin_button_get_adjustment
			(GTK_SPIN_BUTTON(prefsui->limit_days_spin));
		limit_days = gtk_adjustment_get_value(adj);
		KZ_CONF_SET("History", "limit_days", limit_days, INT);

		adj = gtk_spin_button_get_adjustment
			(GTK_SPIN_BUTTON(prefsui->max_results));
		max_results = gtk_adjustment_get_value(adj);
		KZ_CONF_SET("History", "max_results", max_results, INT);
		
		adj = gtk_spin_button_get_adjustment
			(GTK_SPIN_BUTTON(prefsui->num_summary));
		num_summary = gtk_adjustment_get_value(adj);
		KZ_CONF_SET("History", "num_summary", num_summary, INT);
		
		prefsui->changed = FALSE;

		break;
	}
	case GTK_RESPONSE_REJECT:
		break;
	case KZ_RESPONSE_UI_LEVEL_MEDIUM:
		break;
	case KZ_RESPONSE_UI_LEVEL_EXPERT:
		break;
	case KZ_RESPONSE_UI_LEVEL_CUSTOM:
		break;
	default:
		break;
	}
}


KzPrefsWinPageEntry *
prefs_history_get_entry (gint idx)
{
	if (idx == 0)
		return &prefs_entry;
	else
		return NULL;
}
