/* ****************************************************************************
  This file is part of KMathTool

  Copyright (C) 2003 by Marco Wegner <mail@marcowegner.de>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

**************************************************************************** */


#include <qlayout.h>
#include <qsplitter.h>
#include <qwidgetstack.h>

#include <kconfig.h>
#include <kglobal.h>
#include <khtml_part.h>
#include <kstandarddirs.h>

#include "kmathtoolview.h"
#include "kmtmessagewidget.h"
#include "kmtmodule.h"
#include "kmttreeview.h"


KMathToolView::KMathToolView(QWidget* parent, const char* name)
  : QWidget(parent, name)
{
  setupView( );
  readConfig( );

  intropart = 0;
  slotShowIntroduction( );
}

void KMathToolView::setupView( )
{
  QHBoxLayout* topLayout = new QHBoxLayout(this);

  m_hsplitter = new QSplitter(this);
  m_hsplitter->setOrientation(Qt::Horizontal);
  topLayout->addWidget(m_hsplitter);

  treeView = new ModuleTreeView(m_hsplitter);

  connect(treeView, SIGNAL(signalShowView(KMathToolModule*, ModuleView*)),
    this, SLOT(slotShowView(KMathToolModule*, ModuleView*)));

  mainstack = new QWidgetStack(m_hsplitter);

  QWidget* w = new QWidget(this);
  QVBoxLayout* wl = new QVBoxLayout(w);
  mainstack->addWidget(w, MODULES);

  m_vsplitter = new QSplitter(w);
  m_vsplitter->setOrientation(Qt::Vertical);
  wl->addWidget(m_vsplitter);

  modstack = new QWidgetStack(m_vsplitter);

  connect(modstack, SIGNAL(aboutToShow(QWidget*)),
    this, SLOT(slotAboutToShow(QWidget*)));

  messages = new MessageWidget(m_vsplitter);
}

void KMathToolView::readConfig( )
{
  KConfig* config = KGlobal::config( );
  config->setGroup("MainWindow");

  // Restoring the main splitter's sizes.
  QValueList<int> sizes;
  sizes.append(config->readNumEntry("MainSplitterLeft", 1));
  sizes.append(config->readNumEntry("MainSplitterRight", 3));
  m_hsplitter->setSizes(sizes);

  // Restoring the tool splitter's sizes.
  sizes.clear( );
  sizes.append(config->readNumEntry("ToolSplitterTop", 2));
  sizes.append(config->readNumEntry("ToolSplitterBottom", 1));
  m_vsplitter->setSizes(sizes);
}

void KMathToolView::writeConfig( )
{
  KConfig* config = KGlobal::config( );
  config->setGroup("MainWindow");

  // Saving the main splitter's sizes.
  QValueList<int> sizes = m_hsplitter->sizes( );
  config->writeEntry("MainSplitterLeft", sizes[0]);
  config->writeEntry("MainSplitterRight", sizes[1]);

  // Saving the tool splitter's sizes.
  sizes = m_vsplitter->sizes( );
  config->writeEntry("ToolSplitterTop", sizes[0]);
  config->writeEntry("ToolSplitterBottom", sizes[1]);
}

// *******************************
// ***** SLOT implementation *****
// *******************************
void KMathToolView::slotIntegrateModule(KMathToolModule* module)
{
  modstack->addWidget(module->widget( ));
  treeView->createTreeItems(module);
}

void KMathToolView::slotShowView(KMathToolModule* module, ModuleView* view)
{
  module->showView(view);

  // QWidgetStack doesn't emit the signal aboutToShow( ) if the current
  // visibleWidget( ) is null (eg. when raising a widget for the first time);
  // therefore the slot needs to be called manually
  if (!modstack->visibleWidget( ))
    slotAboutToShow(module->widget( ));

  modstack->raiseWidget(module->widget( ));
  mainstack->raiseWidget(MODULES);
}

void KMathToolView::slotAboutToShow(QWidget* w)
{
  QWidget* widget = modstack->visibleWidget( );
  if (widget)
    disconnect(widget, 0, messages, 0);

  ModuleViewStack* mvstack = static_cast<ModuleViewStack*>(w);

  connect(mvstack, SIGNAL(signalMessage(const QString&)),
    messages, SLOT(slotMessage(const QString&)));
  connect(mvstack, SIGNAL(signalError(const QString&)),
    messages, SLOT(slotError(const QString&)));
  connect(mvstack, SIGNAL(signalNewCalculation( )),
    messages, SLOT(clear( )));

  messages->clear( );
}

void KMathToolView::slotShowIntroduction( )
{
  if (!intropart) {
    intropart = new KHTMLPart(this);
    mainstack->addWidget(intropart->widget( ), INTROPART);

    connect(intropart->browserExtension( ),
      SIGNAL(openURLRequest(const KURL&, const KParts::URLArgs&)),
      SLOT(slotOpenURL(const KURL&)));
  }

  mainstack->raiseWidget(INTROPART);
  slotOpenURL(KURL(locate("appdata", "intro/index.html")));
}

void KMathToolView::slotOpenURL(const KURL& url)
{
  intropart->openURL(url);
}

#include "kmathtoolview.moc"
