/* lightlab, Copyright (c) 2002 Jamie Zawinski <jwz@jwz.org>
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.  No representations are made about the suitability of this
 * software for any purpose.  It is provided "as is" without express or 
 * implied warranty.
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <stdlib.h>
#include <stdio.h>

#include <X11/Xlib.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/glx.h>

#include "lightlab.h"
#include "scene.h"
#include "teapot.h"
#include "sphere.h"
#include "trackball.h"

struct scene_data {
  Display *dpy;
  Window window;
  XWindowAttributes xgwa;
  GLXContext glx_context;
  GLfloat rotx, roty, rotz;
  GLfloat dx, dy, dz;
};


Visual *
get_gl_visual (Display *dpy, int screen_num)
{
# define R GLX_RED_SIZE
# define G GLX_GREEN_SIZE
# define B GLX_BLUE_SIZE
# define D GLX_DEPTH_SIZE
# define I GLX_BUFFER_SIZE
# define DB GLX_DOUBLEBUFFER

  int attrs[][20] = {
    { GLX_RGBA, R, 8, G, 8, B, 8, D, 8, DB, 0 }, /* rgb double */
    { GLX_RGBA, R, 4, G, 4, B, 4, D, 4, DB, 0 },
    { GLX_RGBA, R, 2, G, 2, B, 2, D, 2, DB, 0 },
    { GLX_RGBA, R, 8, G, 8, B, 8, D, 8,     0 }, /* rgb single */
    { GLX_RGBA, R, 4, G, 4, B, 4, D, 4,     0 },
    { GLX_RGBA, R, 2, G, 2, B, 2, D, 2,     0 },
    { I, 8,                       D, 8, DB, 0 }, /* cmap double */
    { I, 4,                       D, 4, DB, 0 },
    { I, 8,                       D, 8,     0 }, /* cmap single */
    { I, 4,                       D, 4,     0 },
    { GLX_RGBA, R, 1, G, 1, B, 1, D, 1,     0 }  /* monochrome */
  };

  int i;
  for (i = 0; i < sizeof(attrs)/sizeof(*attrs); i++)
    {
      XVisualInfo *vi = glXChooseVisual (dpy, screen_num, attrs[i]);
      if (vi)
        {
          Visual *v = vi->visual;
          XFree (vi);
          return v;
        }
    }
  return 0;
}


static void
check_gl_error (const char *type)
{
  char buf[100];
  GLenum i;
  const char *e;
  switch ((i = glGetError())) {
  case GL_NO_ERROR: return;
  case GL_INVALID_ENUM:          e = "invalid enum";      break;
  case GL_INVALID_VALUE:         e = "invalid value";     break;
  case GL_INVALID_OPERATION:     e = "invalid operation"; break;
  case GL_STACK_OVERFLOW:        e = "stack overflow";    break;
  case GL_STACK_UNDERFLOW:       e = "stack underflow";   break;
  case GL_OUT_OF_MEMORY:         e = "out of memory";     break;
# ifdef GL_TABLE_TOO_LARGE_EXT
  case GL_TABLE_TOO_LARGE_EXT:   e = "table too large";   break;
# endif
# ifdef GL_TEXTURE_TOO_LARGE_EXT
  case GL_TEXTURE_TOO_LARGE_EXT: e = "texture too large"; break;
# endif
  default:
    e = buf; sprintf (buf, "unknown error %d", (int) i); break;
  }
  fprintf (stderr, "%s: %s error: %s\n", progname, type, e);
  exit (1);
}



scene_data *
init_scene (Display *dpy, Window window)
{
  scene_data *sd = (scene_data *) calloc (sizeof(*sd), 1);
  int screen_num = DefaultScreen (dpy);
  XVisualInfo vi_in, *vi_out;
  int out_count;

  sd->dpy = dpy;
  sd->window = window;

  XGetWindowAttributes (dpy, window, &sd->xgwa);

  vi_in.screen = screen_num;
  vi_in.visualid = XVisualIDFromVisual (sd->xgwa.visual);
  vi_out = XGetVisualInfo (dpy, VisualScreenMask|VisualIDMask,
			   &vi_in, &out_count);
  if (! vi_out) abort ();

  sd->glx_context = glXCreateContext (dpy, vi_out, 0, GL_TRUE);
  XSync (sd->dpy, False);

  if (!sd->glx_context)
    {
      fprintf(stderr, "%s: couldn't create GL context for visual 0x%x.\n",
	      progname, (unsigned int) XVisualIDFromVisual (sd->xgwa.visual));
      exit(1);
    }

  glXMakeCurrent (dpy, window, sd->glx_context);
  check_gl_error ("glXMakeCurrent");

  {
    GLboolean rgba_mode = 0;
    glGetBooleanv(GL_RGBA_MODE, &rgba_mode);
    if (!rgba_mode)
      {
	glIndexi (WhitePixel (dpy, screen_num));
	glClearIndex (BlackPixel (dpy, screen_num));
      }
  }

  sd->rotx = 0;
  sd->roty = 30;
  sd->rotz = 0;

  sd->dy = -2;

  glDrawBuffer (GL_BACK);

  reshape_scene (sd);
  return sd;
}


void
reshape_scene (scene_data *sd)
{
  int width, height;
  GLfloat h;

  XGetWindowAttributes (sd->dpy, sd->window, &sd->xgwa);

  width = sd->xgwa.width;
  height = sd->xgwa.height;
  h = (GLfloat) height / (GLfloat) width;
  glViewport (0, 0, (GLint) width, (GLint) height);
  check_gl_error ("glViewport");

  glMatrixMode (GL_PROJECTION);
  glLoadIdentity ();
  gluPerspective (30, 1/h, 1, 100);
  check_gl_error ("gluPerspective");

  glMatrixMode (GL_MODELVIEW);
  glLoadIdentity ();
  gluLookAt (0, 0, 30,
             0, 0, 0,
             0, 1, 0);
  check_gl_error ("gluLookAt");

  glClear (GL_COLOR_BUFFER_BIT);
}


void
init_texture (scene_data *sd, int width, int height, unsigned char *data)
{
  check_gl_error ("unknown");
  glPixelStorei (GL_UNPACK_ALIGNMENT, 1);
  glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
  glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
  glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
  glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
  glTexGeni (GL_S, GL_TEXTURE_GEN_MODE, GL_OBJECT_LINEAR);
  glTexGeni (GL_T, GL_TEXTURE_GEN_MODE, GL_OBJECT_LINEAR);
  glTexEnvf (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

  glTexImage2D (GL_TEXTURE_2D, 0, GL_RGBA, width, height, 0,
                GL_RGBA, GL_UNSIGNED_BYTE, data);
  check_gl_error ("glTexImage2D");
}


static void
make_floor (int wire_p)
{
  GLfloat z = -1.0;
  GLfloat outer = 3;
  int steps = 7;
  GLfloat size = outer/steps;
  int xx, yy;

  glFrontFace (GL_CW);
  glNormal3f (0, 1, 0);
  for (yy = 0; yy < steps; yy++)
    for (xx = 0; xx < steps; xx++)
      {
        GLfloat t_scale = 4;
        GLfloat x = xx * size - outer/2;
        GLfloat y = yy * size - outer/2;

        GLfloat tx = (GLfloat)xx / steps * t_scale;
        GLfloat ty = (GLfloat)yy / steps * t_scale;
        GLfloat tx2 = tx + size/outer * t_scale;
        GLfloat ty2 = ty + size/outer * t_scale;

        if ((xx&1) ^ (yy&1)) continue;  /* xor checkerboard */

        glBegin (wire_p ? GL_LINE_LOOP : GL_QUADS);
        glTexCoord2f(tx,  ty);  glVertex3f (x,      z, y);
        glTexCoord2f(tx2, ty);  glVertex3f (x+size, z, y);
        glTexCoord2f(tx2, ty2); glVertex3f (x+size, z, y+size);
        glTexCoord2f(tx,  ty2); glVertex3f (x,      z, y+size);
        glEnd();
      }
  check_gl_error ("make_floor");
}


static void
make_axes (void)
{
  int size1 = 10;
  int size2 = 10;
  GLfloat w1 = 0.20;
  GLfloat w2 = 0.05;
  int i, j;

  glDisable (GL_LIGHTING);
  glDisable (GL_TEXTURE_2D);

  glColor3f (1, 1, 1);

  glBegin (GL_LINES);
  glVertex3f (-size1, 0, 0); glVertex3f (size1, 0, 0);
  glVertex3f (0, -size1, 0); glVertex3f (0, size1, 0);
  glVertex3f (0, 0, -size1); glVertex3f (0, 0, size1);

  for (i = 0; i <= size1; i++)
    {
      glVertex3f (i, -w1, 0); glVertex3f (i, w1, 0);
      glVertex3f (i, 0, -w1); glVertex3f (i, 0, w1);

      glVertex3f (-w1, i, 0); glVertex3f (w1, i, 0);
      glVertex3f (0, i, -w1); glVertex3f (0, i, w1);

      glVertex3f (-w1, 0, i); glVertex3f (w1, 0, i);
      glVertex3f (0, -w1, i); glVertex3f (0, w1, i);

   /* glVertex3f (-i, -w1, 0); glVertex3f (-i, w1, 0);*/
      glVertex3f (-i, 0, -w1); glVertex3f (-i, 0, w1);

      glVertex3f (-w1, -i, 0); glVertex3f (w1, -i, 0);
   /* glVertex3f (0, -i, -w1); glVertex3f (0, -i, w1);*/

      glVertex3f (-w1, 0, -i); glVertex3f (w1, 0, -i);
   /* glVertex3f (0, -w1, -i); glVertex3f (0, w1, -i);*/

      if (i == size1) break;

      for (j = 1; j < size2; j++)
        {
          GLfloat k = i + (j / (GLfloat) size2);

          glVertex3f (k, -w2, 0); glVertex3f (k, w2, 0);
          glVertex3f (k, 0, -w2); glVertex3f (k, 0, w2);

          glVertex3f (-w2, k, 0); glVertex3f (w2, k, 0);
          glVertex3f (0, k, -w2); glVertex3f (0, k, w2);

          glVertex3f (-w2, 0, k); glVertex3f (w2, 0, k);
          glVertex3f (0, -w2, k); glVertex3f (0, w2, k);

       /* glVertex3f (-k, -w2, 0); glVertex3f (-k, w2, 0); */
          glVertex3f (-k, 0, -w2); glVertex3f (-k, 0, w2);

          glVertex3f (-w2, -k, 0); glVertex3f (w2, -k, 0);
       /* glVertex3f (0, -k, -w2); glVertex3f (0, -k, w2); */

          glVertex3f (-w2, 0, -k); glVertex3f (w2, 0, -k);
       /* glVertex3f (0, -w2, -k); glVertex3f (0, w2, -k); */
        }
    }

  glEnd();
  check_gl_error ("make_axes");
}



static void
generate_scene (scene_data *sd, int object, int smooth_p, int wire_p)
{
  glColor3f (1, 1, 1);

  make_floor (wire_p);

  switch (object)
    {
    case 0:
      glFrontFace (GL_CW);
      if (wire_p)
        wireTeapot (1);
      else
        solidTeapot (1);
      break;

    case 1:
      glScalef (1.4, 1.4, 1.4);
      unit_cube (wire_p);
      break;

    case 2:
      unit_sphere (32, 32, wire_p);
      break;

    case 3:
      glScalef (0.8, 1.5, 0.8);
      glTranslatef (0, -0.4, 0);
      unit_tube (32, smooth_p, wire_p);
      break;

    case 4:
      glScalef (0.8, 2.5, 0.8);
      glTranslatef (0, -0.2, 0);
      unit_cone (32, smooth_p, wire_p);
      break;

    default:
      abort();
      break;
    }
  check_gl_error ("generate_scene");
}


static void
position_light (int which, GLboolean z_up_p, light_info *L)
{
  GLfloat a[4];
  a[0] = L->x;
  a[1] = (z_up_p ? L->z : L->y);
  a[2] = (z_up_p ? L->y : L->z);
  a[3] = L->w;

  which += GL_LIGHT0;

  glLightfv (which, GL_POSITION, a);

  if (L->vector_p)
    {
      /* Set the color of the vector to be the brighest of the three
         possible colors of the light.
       */
      GLfloat c=0, n;
      if ((n = L->ambient.r + L->ambient.g + L->ambient.b) > c)
        c = n, glColor3f (L->ambient.r, L->ambient.g, L->ambient.b);

      if ((n = L->diffuse.r + L->diffuse.g + L->diffuse.b) > c)
        c = n, glColor3f (L->diffuse.r, L->diffuse.g, L->diffuse.b);

      if ((n = L->specular.r + L->specular.g + L->specular.b) > c)
        c = n, glColor3f (L->specular.r, L->specular.g, L->specular.b);

      /* No lights or textures when rendering the vector itself. */
      glDisable (GL_LIGHTING);
      glDisable (GL_TEXTURE_2D);

      if (a[3] == 0) a[3] = 0.02; /* infinity, where infinity == 20 */
      a[0] /= a[3];
      a[1] /= a[3];
      a[2] /= a[3];

      glBegin(GL_LINES);
      glVertex3f (0, 0, 0);
      glVertex3f (a[0], a[1], a[2]);
      glEnd();
      glPointSize (16);
      glBegin(GL_POINTS);
      glVertex3f (a[0], a[1], a[2]);
      glEnd();

    }
  check_gl_error ("position_light");
}


static void
color_light (int which, light_info *L)
{
  GLfloat a[4];

  which += GL_LIGHT0;

  if (L->enabled_p) glEnable (which);
  else              glDisable(which);

  a[3] = 1.0;
  a[0] = L->ambient.r; a[1] = L->ambient.g; a[2] = L->ambient.b;
  glLightfv (which, GL_AMBIENT, a);

  a[0] = L->diffuse.r; a[1] = L->diffuse.g; a[2] = L->diffuse.b;
  glLightfv (which, GL_DIFFUSE, a);

  a[0] = L->specular.r; a[1] = L->specular.g; a[2] = L->specular.b;
  glLightfv (which, GL_SPECULAR, a);

  check_gl_error ("color_light");
}


static void
color_object (object_info *M)
{
  GLfloat a[4];

  a[3] = 1.0;

  a[0] = M->ambient.r; a[1] = M->ambient.g; a[2] = M->ambient.b;
  glMaterialfv (GL_FRONT_AND_BACK, GL_AMBIENT, a);

  a[0] = M->diffuse.r; a[1] = M->diffuse.g; a[2] = M->diffuse.b;
  glMaterialfv (GL_FRONT_AND_BACK, GL_DIFFUSE, a);

  a[0] = M->specular.r; a[1] = M->specular.g; a[2] = M->specular.b;
  glMaterialfv (GL_FRONT_AND_BACK, GL_SPECULAR, a);

  glMaterialf (GL_FRONT_AND_BACK, GL_SHININESS, M->shininess);

  check_gl_error ("color_object");
}



void
update_scene (scene_data *sd)
{
  light_info L0, L1, L2;
  object_info M;
  int wire_p = 0, smooth_p = 0;
  GLfloat m[4][4];

  check_gl_error ("unknown");

  /* --------------------------------------------------------------------
     Collect information from the UI widgets
     --------------------------------------------------------------------
   */

  get_light_info (0, &L0);
  get_light_info (1, &L1);
  get_light_info (2, &L2);

  get_object_info (0, &M);

  build_rotmatrix (m, M.mouse_quaternion);

  switch (M.render)
    {
    case 0:  smooth_p = 1; break;  /* smooth */
    case 1:                break;  /* faceted */
    case 2:  wire_p = 1;   break;  /* wireframe */
    default: abort();      break;
    }

  if (M.spin_p)   /* if the object is to be rotating, tick the counters. */
    {
#     define INC(a,b) a += b; if (a < 0) a += 360; else if (a >= 360) a -= 360
      INC (sd->rotx, sd->dx);
      INC (sd->roty, sd->dy);
      INC (sd->rotz, sd->dz);
#     undef INC
    }


  /* --------------------------------------------------------------------
     Set global GL environment parameters
     --------------------------------------------------------------------
   */
  glClear (GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
  glEnable (GL_DEPTH_TEST);
  glEnable (GL_NORMALIZE);
  glEnable (GL_CULL_FACE);
  glEnable (GL_POINT_SMOOTH);

  color_light (0, &L0);
  color_light (1, &L1);
  color_light (2, &L2);

  color_object (&M);

  /* --------------------------------------------------------------------
     Rotations and transformations
     --------------------------------------------------------------------
   */
  glPushMatrix ();

  glScalef (3, 3, 3);     /* Scale everything up. */

  /* First do the rotation due to the positioning of the observer
     with the mouse...  This rotation occurs before lights are placed,
     because whether lights are relative to the scene or to the object,
     they are never relative to the camera.
   */
  glMultMatrixf (&m[0][0]);

  if (! M.rotate_lights_p)
    {
      position_light (0, M.z_up_p, &L0);
      position_light (1, M.z_up_p, &L1);
      position_light (2, M.z_up_p, &L2);
    }

  if (! M.spin_p)    /* draw coordinate system when mouse button down. */
    make_axes ();

  /* Then do the auto-rotation (and hardcoded initial tilt) of the
     objects in the scene.
   */
  glRotatef (sd->rotx, 1, 0, 0);
  glRotatef (sd->roty, 0, 1, 0);
  glRotatef (sd->rotz, 0, 0, 1);


  /* Now position the lights, if they are rotating with the objects.
   */
  if (M.rotate_lights_p)
    {
      position_light (0, M.z_up_p, &L0);
      position_light (1, M.z_up_p, &L1);
      position_light (2, M.z_up_p, &L2);
    }


  /* Set the object's desired appearance flags.
   */
  if (wire_p)    glDisable(GL_LIGHTING);
  else           glEnable (GL_LIGHTING);

  if (M.solid_p) glDisable(GL_TEXTURE_2D);
  else           glEnable (GL_TEXTURE_2D);

  if (smooth_p)  glShadeModel (GL_SMOOTH);
  else           glShadeModel (GL_FLAT);


  /* Draw the objects.
   */
  generate_scene (sd, M.type, smooth_p, wire_p);

  glPopMatrix();

  glFinish();
  glXSwapBuffers (sd->dpy, sd->window);
  glFlush();
  check_gl_error ("update_scene");
}
