/*
 *   Copyright (C) 2002,2003 by Jonathan Naylor G4KLX/HB9DRD
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "SunMoonPanel.h"

#include "common/Locator.h"

#include <cmath>

BEGIN_EVENT_TABLE(CSunMoonPanel, wxPanel)
	EVT_PAINT(CSunMoonPanel::onPaint)
END_EVENT_TABLE()

CSunMoonPanel::CSunMoonPanel(wxWindow* parent, int id, const wxPoint& pos, const wxSize& size) :
wxPanel(parent, id, pos, size),
wxTimer(),
m_sun(),
m_moon(),
m_bitmap(NULL),
m_width(0),
m_height(0)
{
	m_width  = size.GetWidth();
	m_height = size.GetHeight();

	m_bitmap = new wxBitmap(m_width, m_height);

	wxMemoryDC dc;
	dc.SelectObject(*m_bitmap);

	dc.BeginDrawing();
	dc.SetBackground(*wxBLACK_BRUSH);
	dc.Clear();
	dc.EndDrawing();

	Start(1000);
}


CSunMoonPanel::~CSunMoonPanel()
{
	delete m_bitmap;
}

void CSunMoonPanel::onPaint(const wxPaintEvent& event)
{
	wxPaintDC dc(this);

	dc.BeginDrawing();
	dc.DrawBitmap(*m_bitmap, 0, 0, false);
	dc.EndDrawing();
}

void CSunMoonPanel::Notify()
{
	wxDateTime now = wxDateTime::UNow();

	double dayNum = m_moon.getDayNum(now);
	m_moon.findMoon(dayNum);
	m_sun.findSun(dayNum);

	wxString moonAz;
	moonAz.Printf(wxT("Az: %3.1f"), m_moon.getAzimuth());

	wxString moonEl;
	moonEl.Printf(wxT("El: %2.1f"), m_moon.getElevation());

	wxString moonRA;
	moonRA.Printf(wxT("%3.2f"), m_moon.getRA());

	wxString moonDec;
	moonDec.Printf(wxT("%2.2f"), m_moon.getDec());

	wxString moonLHA;
	moonLHA.Printf(wxT("%3.2f"), m_moon.getLHA());

	wxString moonGHA;
	moonGHA.Printf(wxT("%3.2f"), m_moon.getGHA());

	wxString moonRange;
	moonRange.Printf(wxT("%3.0f"), m_moon.getRange() / 1000.0);

	wxString delay;
	delay.Printf(wxT("%.0f"), 2.0 * m_moon.getRange() / 299810.0);

	wxString doppler;
	double shift = m_moon.getDoppler();
	doppler.Printf(wxT("%c%3.1f"), (shift >= 0.0) ? wxT('+') : wxT('-'), ::fabs(shift));

	wxString echo;
	shift = 2.0 * m_moon.getDoppler();
	echo.Printf(wxT("%c%3.1f"), (shift >= 0.0) ? wxT('+') : wxT('-'), ::fabs(shift));

	wxString loss;
	double ploss = m_moon.getPathLoss();
	loss.Printf(wxT("%c%2.1f"), (ploss >= 0.0) ? wxT('+') : wxT('-'), ::fabs(ploss));

	wxString sunAz;
	sunAz.Printf(wxT("%3.1f"), m_sun.getAzimuth());

	wxString sunEl;
	sunEl.Printf(wxT("%2.1f"), m_sun.getElevation());

	wxString sunRA;
	sunRA.Printf(wxT("%3.2f"), m_sun.getRA());

	wxString sunDec;
	sunDec.Printf(wxT("%2.2f"), m_sun.getDec());

	wxString sunRange;
	sunRange.Printf(wxT("%.5f"), m_sun.getRange());

	wxMemoryDC mDC;
	mDC.SelectObject(*m_bitmap);
	mDC.BeginDrawing();

	mDC.SetBackground(*wxBLACK_BRUSH);
	mDC.Clear(),

	mDC.SetTextForeground(*wxWHITE);

	wxFont font1(50, wxDEFAULT, wxNORMAL, wxNORMAL);
	mDC.SetFont(font1);

	writeText(mDC, moonAz, m_width / 2, 10, Text_Centred);
	writeText(mDC, moonEl, m_width / 2, 70, Text_Centred);

	wxFont font2(20, wxDEFAULT, wxNORMAL, wxNORMAL);
	mDC.SetFont(font2);

	writeText(mDC, wxT("Sun"),  m_width / 2 - 200, m_height / 2, Text_Centred);
	writeText(mDC, wxT("Moon"), m_width / 2 + 100, m_height / 2, Text_Centred);

	wxBrush brush(wxT("BLACK"), wxTRANSPARENT);
	mDC.SetBrush(brush);

	wxPen pen("WHITE", 5, wxSOLID);
	mDC.SetPen(pen);

	mDC.DrawRectangle(m_width / 2 - 280, m_height / 2 - 10, 560, 150);
	mDC.DrawRectangle(m_width / 2 - 280, m_height / 2 - 10, 180, 150);

	wxFont font3(14, wxDEFAULT, wxNORMAL, wxNORMAL);
	mDC.SetFont(font3);

	writeText(mDC, wxT("Az:"),    m_width / 2 - 270, m_height / 2 + 30,  Text_Left);
	writeText(mDC, sunAz,         m_width / 2 - 150, m_height / 2 + 30,  Text_Right);
	writeText(mDC, wxT(""),      m_width / 2 - 145, m_height / 2 + 30,  Text_Left);
	writeText(mDC, wxT("El:"),    m_width / 2 - 270, m_height / 2 + 50,  Text_Left);
	writeText(mDC, sunEl,         m_width / 2 - 150, m_height / 2 + 50,  Text_Right);
	writeText(mDC, wxT(""),      m_width / 2 - 145, m_height / 2 + 50,  Text_Left);
	writeText(mDC, wxT("RA:"),    m_width / 2 - 270, m_height / 2 + 70,  Text_Left);
	writeText(mDC, sunRA,         m_width / 2 - 150, m_height / 2 + 70,  Text_Right);
	writeText(mDC, wxT(""),      m_width / 2 - 145, m_height / 2 + 70,  Text_Left);
	writeText(mDC, wxT("Dec:"),   m_width / 2 - 270, m_height / 2 + 90,  Text_Left);
	writeText(mDC, sunDec,        m_width / 2 - 150, m_height / 2 + 90,  Text_Right);
	writeText(mDC, wxT(""),      m_width / 2 - 145, m_height / 2 + 90,  Text_Left);
	writeText(mDC, wxT("Range:"), m_width / 2 - 270, m_height / 2 + 110, Text_Left);
	writeText(mDC, sunRange,      m_width / 2 - 150, m_height / 2 + 110, Text_Right);
	writeText(mDC, wxT("AU"),     m_width / 2 - 145, m_height / 2 + 110, Text_Left);

	writeText(mDC, wxT("RA:"),    m_width / 2 - 70, m_height / 2 + 30,  Text_Left);
	writeText(mDC, moonRA,        m_width / 2 + 40, m_height / 2 + 30,  Text_Right);
	writeText(mDC, wxT(""),      m_width / 2 + 45, m_height / 2 + 30,  Text_Left);
	writeText(mDC, wxT("Dec:"),   m_width / 2 - 70, m_height / 2 + 50,  Text_Left);
	writeText(mDC, moonDec,       m_width / 2 + 40, m_height / 2 + 50,  Text_Right);
	writeText(mDC, wxT(""),      m_width / 2 + 45, m_height / 2 + 50,  Text_Left);
	writeText(mDC, wxT("LHA:"),   m_width / 2 - 70, m_height / 2 + 70,  Text_Left);
	writeText(mDC, moonLHA,       m_width / 2 + 40, m_height / 2 + 70,  Text_Right);
	writeText(mDC, wxT(""),      m_width / 2 + 45, m_height / 2 + 70,  Text_Left);
	writeText(mDC, wxT("GHA:"),   m_width / 2 - 70, m_height / 2 + 90,  Text_Left);
	writeText(mDC, moonGHA,       m_width / 2 + 40, m_height / 2 + 90,  Text_Right);
	writeText(mDC, wxT(""),      m_width / 2 + 45, m_height / 2 + 90,  Text_Left);
	writeText(mDC, wxT("Range:"), m_width / 2 - 70, m_height / 2 + 110, Text_Left);
	writeText(mDC, moonRange,     m_width / 2 + 40, m_height / 2 + 110, Text_Right);
	writeText(mDC, wxT("km"),     m_width / 2 + 45, m_height / 2 + 110, Text_Left);

	writeText(mDC, wxT("Doppler:"),    m_width / 2 + 100, m_height / 2 + 30,  Text_Left);
	writeText(mDC, doppler,            m_width / 2 + 230, m_height / 2 + 30,  Text_Right);
	writeText(mDC, wxT("Hz"),          m_width / 2 + 235, m_height / 2 + 30,  Text_Left);
	writeText(mDC, wxT("Echo:"),       m_width / 2 + 100, m_height / 2 + 50,  Text_Left);
	writeText(mDC, echo,               m_width / 2 + 230, m_height / 2 + 50,  Text_Right);
	writeText(mDC, wxT("Hz"),          m_width / 2 + 235, m_height / 2 + 50,  Text_Left);
	writeText(mDC, wxT("Path Loss:"),  m_width / 2 + 100, m_height / 2 + 70,  Text_Left);
	writeText(mDC, loss,               m_width / 2 + 230, m_height / 2 + 70,  Text_Right);
	writeText(mDC, wxT("dB"),          m_width / 2 + 235, m_height / 2 + 70,  Text_Left);
	writeText(mDC, wxT("Echo Delay:"), m_width / 2 + 100, m_height / 2 + 90,  Text_Left);
	writeText(mDC, delay,              m_width / 2 + 230, m_height / 2 + 90,  Text_Right);
	writeText(mDC, wxT("ms"),          m_width / 2 + 235, m_height / 2 + 90,  Text_Left);

	mDC.EndDrawing();
	mDC.SelectObject(wxNullBitmap);

	wxClientDC cDC(this);
	cDC.BeginDrawing();
	cDC.DrawBitmap(*m_bitmap, 0, 0, false);
	cDC.EndDrawing();
}

void CSunMoonPanel::writeText(wxDC& dc, const wxString& text, int x, int y, enum textPos pos) const
{
	wxCoord width, height;
	dc.GetTextExtent(text, &width, &height);

	switch (pos) {
		case Text_Left:
			break;
		case Text_Right:
			x -= width;
			break;
		case Text_Centred:
			x -= width / 2;
			break;
		default:
			wxASSERT(false);
			break;
	}

	wxASSERT(x >= 0 && x < m_width);
	wxASSERT(y >= 0 && y < m_height);

	dc.DrawText(text, x, y);
}

bool CSunMoonPanel::setLocator(const wxString& locator)
{
	CLocator loc(locator);

	if (!loc.isValid())
		return false;

	m_moon.setLocation(loc.getLatitude(), loc.getLongitude());
	m_sun.setLocation(loc.getLatitude(), loc.getLongitude(), 150.0);

	return true;
}

void CSunMoonPanel::setFrequency(double frequency)
{
	wxASSERT(frequency > 0.0);

	m_moon.setFrequency(frequency);
}
