#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <stdint.h>

#include <sys/socket.h>
#include <sys/types.h>
#include <sys/time.h>

#include <arpa/inet.h>

#include "gnet_lib.h"
#include "gnet_engine.h"
#include "gnet_channel.h"
#include "gnet_msg.h"
#include "gnet_host.h"
#include "gnet_search.h"
#include "gnet_defaults.h"

static void
rescan_sets(struct gnet *gnet){
    struct list_head *p;
    struct channel *chan;

    FD_ZERO(&gnet->g_rdset);
    FD_ZERO(&gnet->g_wrset);
    FD_ZERO(&gnet->g_exset);

    FD_SET(gnet->g_signal_fd[0], &gnet->g_rdset);
    FD_SET(gnet->g_signal_fd[0], &gnet->g_exset);
    gnet->g_maxfd = gnet->g_signal_fd[0];

    pthread_mutex_lock(&gnet->g_channels_lock);

    list_for_each(p, &gnet->g_channels){
	chan = list_entry(p, struct channel, c_list);
//	TRACE("channel state: %d", chan->c_state);

	if(chan->c_state != CHANNEL_STATE_DISCONNECTED){
	    if(chan->c_testrd){
//		TRACE("adding fd for reading...");

		FD_SET(chan->c_fd, &gnet->g_rdset);
		FD_SET(chan->c_fd, &gnet->g_exset);
		if(chan->c_fd > gnet->g_maxfd)
		    gnet->g_maxfd = chan->c_fd;
	    }

	    if(chan->c_testwr){
//		TRACE("adding fd for writing...");

		FD_SET(chan->c_fd, &gnet->g_wrset);
		FD_SET(chan->c_fd, &gnet->g_exset);
		if(chan->c_fd > gnet->g_maxfd)
		    gnet->g_maxfd = chan->c_fd;
	    }
	}
    }

    pthread_mutex_unlock(&gnet->g_channels_lock);
}

static int
check_to(struct gnet *gnet, struct channel *chan, time_t now){
    if(now - chan->c_stamp >= (chan->c_state == CHANNEL_STATE_CONNECTING ? gnet->g_cfg->connect_to : gnet->g_cfg->rw_to)){
	TRACE("channel timed out %s ", chan->c_state == CHANNEL_STATE_CONNECTING? "connecting":"on rw");

	gnet_drop_channel(gnet, chan);
	return 1;
    }

    return 0;
}

static void
engine_loop(struct gnet *gnet){
    struct list_head *p, *tmp;
    struct timeval tv;
    fd_set rd_set, wr_set, ex_set;
    int res, done = 0;
    time_t now;

    TRACE("entering I/O loop...");

    rescan_sets(gnet);

    do {
	rd_set = gnet->g_rdset;
	wr_set = gnet->g_wrset;
	ex_set = gnet->g_exset;

	tv.tv_sec = gnet->g_cfg->check_to;
	tv.tv_usec = 0;

	if((res = select(gnet->g_maxfd + 1, &rd_set, &wr_set, &ex_set, &tv)) < 0){
	    if(errno == EINTR){
		TRACE("interrupted...");
		continue;
	    }else{
		WARN("select failed: %s", strerror(errno));
		done = 1;
		break;
	    }
	}

//	TRACE("status: %d connected channels, %d connecting channels", gnet->g_connected_peers, gnet->g_connecting_peers);

	if(FD_ISSET(gnet->g_signal_fd[0], &rd_set)){
	    while((res = read(gnet->g_signal_fd[0], gnet->g_buf, MAX_BUFFER)) == MAX_BUFFER);
	    if(res <= 0){
		TRACE("signal pipe closed");
		break;
	    }
	    
	    TRACE("got a rescan signal");

	    rescan_sets(gnet);
	    continue;
	}

	pthread_mutex_lock(&gnet->g_channels_lock);

	now = time(NULL);

	list_for_each_safe(p, tmp, &gnet->g_channels){
	    struct channel *chan = list_entry(p, struct channel, c_list);

	    if(check_to(gnet, chan, now))
		continue;
	    
	    if(FD_ISSET(chan->c_fd, &rd_set))
		gnet_channel_io(gnet, chan, GNET_OP_READ);
	    else if(FD_ISSET(chan->c_fd, &wr_set))
		gnet_channel_io(gnet, chan, GNET_OP_WRITE);
	    else if(FD_ISSET(chan->c_fd, &ex_set))
		gnet_channel_io(gnet, chan, GNET_OP_EX);

	}

	gnet_check_peers(gnet);

	pthread_mutex_unlock(&gnet->g_channels_lock);

    }while(!done);

    TRACE("exiting I/O loop...");

}

static void*
thread_launcher(void *param){
    struct gnet *gnet = (struct gnet*)param;

    TRACE("engine thread created...");

    engine_loop(gnet);
    
    TRACE("engine thread exiting...");

    return NULL;
}

void
gnet_shutdown(struct gnet *gnet){
    struct list_head *p, *tmp;
    struct query *q;
    struct host *h;
    int res;

    TRACE("shutting down gnet engine...");    

    close(gnet->g_signal_fd[1]);

    if((res = pthread_join(gnet->g_thread, NULL)))
	TRACE("join failed: %s", strerror(res));
    
    TRACE("engine thread stopped...");

    list_for_each_safe(p, tmp, &gnet->g_channels)
	gnet_channel_destroy(gnet, list_entry(p, struct channel, c_list));

    list_for_each_safe(p, tmp, &gnet->g_queries){
	q = list_entry(p, struct query, list);
	list_del(&q->list);
	free(q);
    }

    list_for_each_safe(p, tmp, &gnet->g_hosts){
	h = list_entry(p, struct host, list);
	list_del(&h->list);
	free(h);
    }

    close(gnet->g_signal_fd[0]);

    free(gnet->g_cfg);
    free(gnet);
}

struct gnet*
gnet_init(struct gnet_config *conf){
    struct gnet *gnet;
    struct gnet_config *cfg;

    TRACE("initializing gnet engine...");

    srandom(time(NULL));

    if(!(cfg = malloc(sizeof(struct gnet_config)))){
	WARN("out of mem!");
	return NULL;
    }
    
    memcpy(cfg, conf, sizeof(struct gnet_config));

    if(!(gnet = malloc(sizeof(struct gnet)))){
	WARN("out of memory!");
	goto fail_gnet;
    }

    memset(gnet, 0, sizeof(struct gnet));

    if(pipe(gnet->g_signal_fd) < 0){
	WARN("could not create pipe: %s", strerror(errno));
	goto fail_thread;
    }

    gnet_make_guid(gnet->g_guid);

    gnet->g_cfg = cfg;

    INIT_LIST_HEAD(&gnet->g_channels);
    INIT_LIST_HEAD(&gnet->g_queries);
    INIT_LIST_HEAD(&gnet->g_hosts);
    pthread_mutex_init(&gnet->g_channels_lock, NULL);

    gnet->g_guids_root.guid[0] = 0x80;

    if(pthread_create(&gnet->g_thread, NULL, thread_launcher, gnet)){
	WARN("could not create thread!");
	goto fail_thread;
    }

    return gnet;

  fail_thread:
    free(gnet);
  fail_gnet:
    free(cfg);

    return NULL;
}

void
gnet_engine_signal(struct gnet *gnet, char sig){
    
    write(gnet->g_signal_fd[1], &sig, 1);
}

void
gnet_set_defaults(struct gnet_config *cfg){

    cfg->shared_files 	= DEF_SHARED_FILES;
    cfg->shared_kb	= DEF_SHARED_KB;
    cfg->listen_port	= DEF_LISTEN_PORT;
    cfg->query_ttl 	= DEF_QUERY_TTL;
    cfg->query_wait	= DEF_QUERY_WAIT;
    cfg->min_speed	= DEF_MIN_SPEED;
    cfg->keep_peers 	= DEF_KEEP_PEERS;
    cfg->keep_connecting= DEF_KEEP_CONNECTING;
    cfg->min_hosts	= DEF_MIN_HOSTS;
    cfg->max_hosts	= DEF_MAX_HOSTS;
    cfg->check_to	= DEF_CHECK_TO;
    cfg->rw_to		= DEF_RW_TO;
    cfg->connect_to	= DEF_CONNECT_TO;
}

