/***************************************************************************
 $RCSfile: tagfile.cpp,v $
 -------------------
 cvs         : $Id: tagfile.cpp,v 1.2 2003/02/09 01:59:29 aquamaniac Exp $
 begin       : Sat Feb 08 2003
 copyright   : (C) 2003 by Martin Preuss
 email       : martin@libchipcard.de

 ****************************************************************************
 * This program is free software; you can redistribute it and/or modify     *
 * it under the terms of the GNU General Public License as published by     *
 * the Free Software Foundation; either version 2 of the License, or        *
 * (at your option) any later version.                                      *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA *
 ****************************************************************************/


#include "tagfile.h"
#include <errno.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/stat.h>


TagFile::Tag::Tag(const string &t, unsigned long o)
  :_type(0)
  ,_length(0)
  ,_offset(o)
{
  if (t.length()>5) {
    _type=((unsigned char)(t[0]))<<8;
    _type+=((unsigned char)(t[1]));

    _length=((unsigned char)(t[2]))<<24;
    _length+=((unsigned char)(t[3]))<<16;
    _length+=((unsigned char)(t[4]))<<8;
    _length+=((unsigned char)(t[5]));

    if (t.length()>6)
      _data=t.substr(6);
  }
}


TagFile::Tag::Tag(int t, unsigned long l, unsigned long o, const string &d)
:_type(t)
,_length(l)
,_offset(o)
,_data(d)
{
}


TagFile::Tag::Tag()
:_type(0)
,_length(0)
,_offset(0)
{
}


TagFile::Tag::~Tag(){
}


string TagFile::Tag::toString() const{
  unsigned char buffer[6];

  buffer[0]=(_type>>8)&0xff;
  buffer[1]=(_type>>8);
  buffer[2]=(_length>>24)&0xff;
  buffer[3]=(_length>>16)&0xff;
  buffer[4]=(_length>>8)&0xff;
  buffer[5]=(_length)&0xff;

  return string((char*)buffer, 6);
}









TagFile::TagFile()
:_file(0)
,_nextTag(0)
{
}


TagFile::TagFile(const string &filename)
:_filename(filename)
,_file(0)
,_nextTag(0)
{
}


bool TagFile::openFile(){
  struct stat st;

  if (_file) {
    fprintf(stderr, "File already open.\n");
    return false;
  }

  // get filesize and store it internally
  if (stat(_filename.c_str(),&st)) {
    fprintf(stderr,
	    "Could not stat file \"%s\": %s\n",
	    _filename.c_str(),strerror(errno));
    return false;
  }
  _filesize=st.st_size;

  // open the file
  _file=fopen(_filename.c_str(),"r");
  if (!_file) {
    fprintf(stderr,
	    "Could not open file \"%s\": %s\n",
	    _filename.c_str(),strerror(errno));
    return false;
  }
  return true;
}


bool TagFile::createFile(){
  if (_file) {
    fprintf(stderr, "File already open.\n");
    return false;
  }
  _file=fopen(_filename.c_str(),"w+");
  if (!_file) {
    fprintf(stderr,
	    "Could not open file \"%s\": %s\n",
	    _filename.c_str(),strerror(errno));
    return false;
  }

  return true;
}


bool TagFile::closeFile(){
  if (!_file) {
    fprintf(stderr, "File not open.\n");
    return false;
  }
  if (fclose(_file)) {
    _file=0;
    fprintf(stderr,
	    "Could not close file \"%s\": %s\n",
	    _filename.c_str(),strerror(errno));
    return false;
  }
  _file=0;
  return true;
}


bool TagFile::scanFile(){
  Tag tag;
  char buffer[6];
  string tmp;

  if (!_file) {
    fprintf(stderr,"File not open.\n");
    return false;
  }

  _nextTag=0;
  while(_nextTag<_filesize) {
    // seek position of next tag
    if (fseek(_file,_nextTag, SEEK_SET)) {
      fprintf(stderr,
	      "Could not seek file \"%s\":%lu: %s\n",
	      _filename.c_str(),_nextTag,strerror(errno));
      return false;
    }
    // read tag header
    if (fread(buffer, 1, 6, _file)!=6) {
      fprintf(stderr,
	      "Could not read from file \"%s\":%lu: %s\n",
	      _filename.c_str(),_nextTag,strerror(errno));
      return false;
    }
    // create Tag from it
    tmp=string(buffer,6);
    tag=Tag(tmp, _nextTag);
    if ((tag.length()+_nextTag+6)>_filesize) {
      fprintf(stderr,
	      "Bad tag size in file \"%s\":%lu: %s\n",
	      _filename.c_str(),_nextTag,strerror(errno));
    }
    // add tag
    _tags.push_back(tag);
    _nextTag+=tag.length()+6;
  } // while

  return true;
}


bool TagFile::formatFile(){
  list<Tag>::iterator it;
  string data;

  if (!_file) {
    fprintf(stderr,"File not open.\n");
    return false;
  }
  if (_tags.empty()) {
    fprintf(stderr,"No tags to write.\n");
    return false;
  }

  for (it=_tags.begin();
       it!=_tags.end();
       it++) {
    // seek position
    if (fseek(_file,(*it).offset(), SEEK_SET)) {
      fprintf(stderr,
	      "Could not seek file \"%s\":%lu: %s\n",
	      _filename.c_str(),(*it).offset(),strerror(errno));
      return false;
    }
    // create tag header string
    data=(*it).toString();
    // write tag header
    if (fwrite(data.data(),1,data.length(),_file)!=data.length()) {
      fprintf(stderr,
	      "Could not write to file \"%s\":%lu: %s\n",
	      _filename.c_str(),(*it).offset(),strerror(errno));
      return false;
    }
  } // for
  return true;
}


list<TagFile::Tag> &TagFile::tagList(){
  return _tags;
}


void TagFile::addTag(TagFile::Tag &t){
  t.setOffset(_nextTag);
  _nextTag+=(t.length()+6);
  _tags.push_back(t);
}


string TagFile::readTagData(const TagFile::Tag &t,
			    unsigned long pos,
			    unsigned long len){
  unsigned long offs;
  char *buffer;
  int i;
  string data;

  if (!_file) {
    fprintf(stderr,"File not open.\n");
    return "";
  }
  offs=t.dataOffset()+pos;
  if (fseek(_file,offs, SEEK_SET)) {
    fprintf(stderr,
	    "Could not seek file \"%s\":%lu: %s\n",
	    _filename.c_str(),offs,strerror(errno));
    return "";
  }
  buffer=(char*)malloc(len);
  if (!buffer) {
    fprintf(stderr,"Could not allocate enough memory.\n");
    return "";
  }
  i=fread(buffer, 1,len, _file);
  if (ferror(_file)) {
    fprintf(stderr,
	    "Could not read from file \"%s\":%lu: %s\n",
	    _filename.c_str(),offs,strerror(errno));
    free(buffer);
    return "";
  }
  data=string(buffer,i);
  free(buffer);
  return data;
}


bool TagFile::writeTagData(const TagFile::Tag &t,
			   unsigned long pos,
			   const string &data){
  unsigned long offs;

  if (!_file) {
    fprintf(stderr,"File not open.\n");
    return false;
  }
  offs=t.dataOffset()+pos;
  if (fseek(_file,offs, SEEK_SET)) {
    fprintf(stderr,
	    "Could not seek file \"%s\":%lu: %s\n",
	    _filename.c_str(),offs,strerror(errno));
    return false;
  }

  if (fwrite(data.data(), 1,data.length(), _file)!=data.length()) {
    fprintf(stderr,
	    "Error reading from file \"%s\":%lu: %s\n",
	    _filename.c_str(),offs,strerror(errno));
    return false;
  }

  return true;
}





