/*****************************************************************************
 *                                                                           *
 * Widget:    gtkimmov                                                       *
 * Uses:      GTK+1.2, GdkImlib                                              *
 * Purpose:   Cut images at coordites this way, that they match each other.  *
 *            The matching can be controlled via certain view modes          *
 * Author:    Andreas Tille <tille@debian.org>                                  *
 * Date:      May 15, 2001                                                   *
 * Copyright: Andreas Tille, 1998, 1999, 2000, 2001                          *
 * License:   LGPL                                                           *
 *                                                                           *
 *****************************************************************************/

#include <string.h>
#include <gdk_imlib.h>

#include "gtk/gtkbutton.h"
#include "gtk/gtkcontainer.h"
#include "gtk/gtkdialog.h"
#include "gtk/gtkdrawingarea.h"
#include "gtk/gtkeventbox.h"
#include "gtk/gtkframe.h"
#include "gtk/gtkhbbox.h"
#include "gtk/gtkhbox.h"
#include "gtk/gtkhruler.h"
#include "gtk/gtklabel.h"
#include "gtk/gtkmain.h"
#include "gtk/gtkmenu.h"
#include "gtk/gtknotebook.h"
#include "gtk/gtkoptionmenu.h"
#include "gtk/gtkradiomenuitem.h"
#include "gtk/gtksignal.h"
#include "gtk/gtkspinbutton.h"
#include "gtk/gtktable.h"
#include "gtk/gtkvbox.h"
#include "gtk/gtkvruler.h"
#include "gtkimmov.h"

#include "gtkintl.h"

static void gtk_immov_class_init     (GtkImMovClass *class);
static void gtk_immov_init           (GtkImMov *immov);
static void gtk_immov_destroy        (GtkObject *object);

static void gtk_immov_configure      (GtkWidget *widget, GdkEventConfigure *event, GtkImMov *immov);
static void gtk_immov_expose         (GtkWidget *widget, GdkEventExpose *event, GtkImMov *immov);
static void gtk_immov_value_changed  (GtkAdjustment *adj, GtkWidget *spinner);

/* internal functions */
static gint _difference_move_picture (GtkWidget *immov);

/* internal helpers   */
static gint _gtk_immov_get_move_dimensions (GtkImMovPic *target, GtkImMovPic *src, gint *width, gint *height,
					    guchar **tptr, guchar **sptr);
static void _gtk_immov_clear_target        (GtkImMovPic *im, gint red, gint green, gint blue);
static gint _gtk_immov_insert_picture      (GtkImMovPic *target, GtkImMovPic *src, gint colshift);
static void _gtk_immov_accept_images       (GtkWidget *widget, GtkImMov *immov);
static GtkWidget *_gtk_immov_init_drawing_area (GtkImMov *immov) ;
static void _gtk_immov_page_switch         (GtkWidget *widget, GtkNotebookPage *page, gint page_num);
static void _gtk_immov_update_struct       (GtkImMov *immov);

static GtkWidget      *_build_option_menu  (GtkImMov *immov);
static GtkAdjustment  *_create_adjustment  (GtkWidget *table, GtkImMov *immov, gchar *name, 
                                            gint pos, gint *value, gint max);
static gint            _set_parameter_if_valid (gint *val2set, gint val, gint cond, 
						GtkAdjustment *adj);
static void            _destroy_changing_imlibs ( void ) ;

#define gtk_object_remove_user_data(object)  gtk_object_remove_data(object, "user_data")
#define _gtk_immov_iswap(x, y)  {register int zw = x; x = y; y = zw;}
#define CHANGE_TIME  500    /* delay time for changing images */
static  int          TIMEOUT_TAG = 0;

typedef enum {
   DISPLAY,
   NO_DISPLAY,
   DESTROY
} DisplayFlag;

static DisplayFlag DELAY_SHOW = DISPLAY;  /* sometimes we don't want to show the image immediately */

/* This macro is shamelessly stolen from testgtk.c.                                           *
 * It is necessary to depress doubling of display calculation when switching display methods. */
#define RADIOMENUTOGGLED(_rmi_, __i) { \
  GSList * __g; \
  __i = 0; \
  __g = gtk_radio_menu_item_group(_rmi_); \
  while( __g  && !((GtkCheckMenuItem *)(__g->data))->active) { \
    __g = __g->next; \
    __i++; \
  }\
}

static GtkWindowClass *parent_class = NULL;

GtkType
gtk_immov_get_type (void)
{
  static GtkType IMMOV_type = 0;

  if (!IMMOV_type)
    {
      GtkTypeInfo immov_info =
      {
	"GtkImMov",
	sizeof (GtkImMov),
	sizeof (GtkImMovClass),
	(GtkClassInitFunc) gtk_immov_class_init,
	(GtkObjectInitFunc) gtk_immov_init,
	/* reserved_1 */ NULL,
	/* reserved_2 */ NULL,
        (GtkClassInitFunc) NULL,
      };

      IMMOV_type = gtk_type_unique (GTK_TYPE_WINDOW, &immov_info);
    }

  return IMMOV_type;
}

static void
gtk_immov_class_init (GtkImMovClass *class)
{
  GtkObjectClass *object_class;

  object_class = (GtkObjectClass*) class;

  parent_class = gtk_type_class (GTK_TYPE_DIALOG);

  object_class->destroy = gtk_immov_destroy;
}

static void
gtk_immov_init (GtkImMov *immov)
{
  GtkWidget *confirm_area, *move_type_menu, *hbox, *vbox, *automatch;

  /* The dialog-sized vertical box  */
  immov->main_vbox = gtk_vbox_new (FALSE, 10);
  gtk_container_set_border_width (GTK_CONTAINER (immov), 10);
  gtk_container_add (GTK_CONTAINER (immov), immov->main_vbox);
  gtk_widget_show (immov->main_vbox);

  /**********************************************************************************
   * the moving image                                                               *
   **********************************************************************************/
  immov->nb = gtk_notebook_new();
  gtk_container_add(GTK_CONTAINER(immov->main_vbox), immov->nb);
  gtk_container_set_border_width(GTK_CONTAINER(immov->nb), 10);
  /* Ensure, that immov will be found in the page_switch function */
  gtk_object_set_user_data( GTK_OBJECT(immov->nb), immov );
  gtk_widget_show (immov->nb);

  /**********************************************************************************
   * area for several parameters in the control part                                *
   **********************************************************************************/

  hbox = gtk_hbox_new(FALSE, 1);
  gtk_box_pack_start(GTK_BOX(immov->main_vbox), hbox, TRUE, TRUE, 0);

  immov->parms = gtk_hbox_new(FALSE, 1);
  gtk_box_pack_start(GTK_BOX(hbox), immov->parms, TRUE, FALSE, 0);

  vbox = gtk_vbox_new(FALSE, 5);
  gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, FALSE, 0);
  move_type_menu = _build_option_menu (immov);
  gtk_box_pack_start (GTK_BOX (vbox), move_type_menu, TRUE, FALSE, 0);

  vbox = gtk_vbox_new(FALSE, 5);
  gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, FALSE, 0);
  automatch = gtk_button_new_with_label (_("Try auto-matching"));
  gtk_signal_connect (GTK_OBJECT(automatch), "clicked", GTK_SIGNAL_FUNC(gtk_immov_automatch), immov);
  gtk_box_pack_start (GTK_BOX (vbox), automatch, TRUE, FALSE, 0);

  gtk_widget_show_all (hbox);
 
  /**********************************************************************************
   * The OK/Cancel button area                                                      *
   **********************************************************************************/
   
  confirm_area = gtk_hbutton_box_new ();
  gtk_button_box_set_layout(GTK_BUTTON_BOX(confirm_area), GTK_BUTTONBOX_END);
  gtk_button_box_set_spacing(GTK_BUTTON_BOX(confirm_area), 5);
  gtk_box_pack_end (GTK_BOX (immov->main_vbox), confirm_area, FALSE, FALSE, 0);
  gtk_widget_show (confirm_area);

  /**********************************************************************************
   * The OK button                                                                  *
   **********************************************************************************/
  immov->ok_button = gtk_button_new_with_label ( _("OK") );
  GTK_WIDGET_SET_FLAGS (immov->ok_button, GTK_CAN_DEFAULT);
  gtk_box_pack_start (GTK_BOX (confirm_area), immov->ok_button, TRUE, TRUE, 0);
  gtk_widget_grab_default (immov->ok_button);
  gtk_signal_connect(GTK_OBJECT(immov->ok_button), "clicked", 
                     GTK_SIGNAL_FUNC(_gtk_immov_accept_images), immov);
  gtk_widget_show (immov->ok_button);

  /**********************************************************************************
   * The Cancel button                                                              *
   **********************************************************************************/
  immov->cancel_button = gtk_button_new_with_label (_("Cancel"));
  GTK_WIDGET_SET_FLAGS (immov->cancel_button, GTK_CAN_DEFAULT);
  gtk_box_pack_start (GTK_BOX (confirm_area), immov->cancel_button, TRUE, TRUE, 0);
  gtk_widget_show (immov->cancel_button);

  /**********************************************************************************
   * Initialize structure value                                                     *
   **********************************************************************************/
  immov->iml     = NULL;
  immov->vert    = immov->hori    = NULL;
  /*  immov->curvert = immov->curhori = 0; */
  immov->maxvert = immov->maxhori = 0;
}

GtkWidget*
gtk_immov_new (const gchar *title, GtkImMovPic *fix, GList *movl, gint automatch)
/* --- Parameters: ---
 * const gchar *title     : title of widget window
 * GtkImMovPic *fix       : fix image which is used as reference for all images
 * GList       *movl      : list of images to move against fix
 * gint         automatch : try auto matching
 */
{
  GtkImMov    *immov;
  GtkImMovPic *cur, *im;
  GtkWidget   *table;
  GList       *l;
  gint         w_max = fix->width, h_max = fix->height;

  g_return_val_if_fail ( fix  , NULL ) ;
  g_return_val_if_fail ( fix->width > 0, NULL ) ;
  g_return_val_if_fail ( fix->height > 0, NULL ) ;
  g_return_val_if_fail ( fix->data , NULL ) ;
  g_return_val_if_fail ( fix->id , NULL ) ;
  g_return_val_if_fail ( movl , NULL ) ;
  g_return_val_if_fail ( (cur = movl->data), NULL ) ;

  /* check for valid data structures */
  for ( im = (l = movl)->data; l && (im = l->data); l = l->next ) {
    g_return_val_if_fail ( im->width > 0, NULL ) ;
    if ( im->width  > w_max ) w_max = im->width ;
    g_return_val_if_fail ( im->height > 0, NULL ) ;
    if ( im->height > h_max ) h_max = im->height ;
    g_return_val_if_fail ( im->data , NULL ) ;
    g_return_val_if_fail ( im->id , NULL ) ;
  }

  /*  gdk_rgb_set_verbose (TRUE);
      gdk_rgb_init ();  */

  immov = gtk_type_new (GTK_TYPE_IMMOV);
  g_return_val_if_fail ( immov != NULL, NULL );

  if ( title ) gtk_window_set_title (GTK_WINDOW (immov), title);
  else {
    gchar *ptr;
    
    if ( (ptr = strrchr(fix->id, '/')) && *(ptr+1) ) ptr++;
    else                                             ptr = fix->id;
    ptr = g_strdup_printf (_("Move set of images relative to %s"), ptr);
    gtk_window_set_title (GTK_WINDOW (immov), ptr);
    g_free (ptr) ;
  }

  immov->fix = fix;
  immov->cur = cur;
  immov->im  = gtk_immov_set_struct (w_max, h_max, NULL, "internal image");
  _gtk_immov_update_struct (immov);

  immov->show    = _difference_move_picture ;
  immov->iml     = movl;

  for ( im = (l = movl)->data; l && (im = l->data); l = l->next ) {
    GtkWidget       *child, *label_box, *label, *vbox, *viewport, *draw;
    gchar           *buf;
    
    label_box = gtk_hbox_new(FALSE, 0);
    gtk_widget_show(label_box);
    label     = gtk_label_new (im->id);
    gtk_box_pack_start (GTK_BOX(label_box), label, FALSE, TRUE, 0);
    gtk_widget_show (label);
    
    buf   = g_strdup_printf (_("%s moved relativ to %s"), im->id, fix->id);
    child = gtk_frame_new (buf);
    gtk_container_set_border_width (GTK_CONTAINER(child), 10);
    g_free (buf);
    gtk_notebook_append_page (GTK_NOTEBOOK(immov->nb), child, label_box);

    vbox = gtk_vbox_new (FALSE, 10);
    gtk_container_add (GTK_CONTAINER(child), vbox);
    
    viewport = gtk_frame_new(NULL);
    gtk_frame_set_shadow_type(GTK_FRAME(viewport), GTK_SHADOW_IN);
    gtk_box_pack_start (GTK_BOX(vbox), viewport, FALSE, TRUE, 0);
    gtk_container_set_border_width ( GTK_CONTAINER(viewport), 10);

    draw = _gtk_immov_init_drawing_area(immov);

    gtk_container_add (GTK_CONTAINER(viewport), draw->parent);
    gtk_object_set_user_data( GTK_OBJECT(child), draw);
    gtk_object_set_user_data( GTK_OBJECT(draw), im);
    gtk_widget_show_all (vbox);
  }
  gtk_notebook_set_tab_pos(GTK_NOTEBOOK(immov->nb), GTK_POS_RIGHT);
  gtk_notebook_set_show_tabs(GTK_NOTEBOOK(immov->nb), TRUE);
  gtk_notebook_set_scrollable(GTK_NOTEBOOK(immov->nb), TRUE);
  gtk_signal_connect(GTK_OBJECT(immov->nb), "switch_page", GTK_SIGNAL_FUNC(_gtk_immov_page_switch), immov);
  
  DELAY_SHOW = DISPLAY;
  gtk_drawing_area_size(GTK_DRAWING_AREA(immov->draw), immov->im->width, immov->im->height);
  gtk_widget_show_all (immov->nb);
  
  g_return_val_if_fail ( (immov->show)(GTK_WIDGET(immov)) == 0, NULL ) ;
  
  /**********************************************************************************
   * editables for horizontal and vertical movement                                 *
   **********************************************************************************/
  table = gtk_table_new (2, 2, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_box_pack_start(GTK_BOX(immov->parms), table, TRUE, FALSE, 0);
  
  immov->vert = _create_adjustment(table, immov, _("horizontal:"), 0, &(immov->im->xo), immov->maxvert);
  immov->hori = _create_adjustment(table, immov, _("vertical:"), 1,  &(immov->im->yo), immov->maxhori);
  gtk_widget_show(table);

  if ( automatch ) gtk_immov_automatch(NULL, immov);

  return GTK_WIDGET (immov);
}

void
gtk_immov_free_struct (GtkImMovPic *pic)
{
  if ( !pic ) return;

  if ( (pic->free & 1) && pic->data ) g_free(pic->data);
  if ( (pic->free & 2) && pic->id   ) g_free(pic->id);

  g_free(pic);
}

GtkImMovPic* 
gtk_immov_copy_struct (gint width, gint height, guchar *data, gchar *id)
/* 
 */
{
  GtkImMovPic *pic;
  gchar       *ptr;
   
  g_return_val_if_fail ( width > 0 && height > 0 && id != NULL, NULL ) ;
  
  pic             = g_new ( GtkImMovPic, 1 ) ;
  pic->width      = width;
  pic->height     = height;
  pic->xo         = 0;
  pic->yo         = 0;
  pic->x1         = 0;
  pic->y1         = 0;
  pic->x2         = 0;
  pic->y2         = 0;
  pic->data       = g_new ( guchar, width*height*3 );
  if ( data ) memcpy ( pic->data, data, width*height*3 );
  else        memset ( pic->data, 0, width*height*3 );
  if ( id ) {
    if ( (ptr = strrchr(id, '/')) ) pic->id     = g_strdup ( ptr+1 ) ;
    else                            pic->id     = g_strdup ( id ) ;
  } else 
    pic->id = NULL;
  pic->free       = 3;
  
  return pic ;
}

GtkImMovPic* 
gtk_immov_set_struct (gint width, gint height, guchar *data, gchar *id)
{
  GtkImMovPic *pic;
   
  g_return_val_if_fail ( width > 0 && height > 0 && id != NULL, NULL ) ;
  
  pic             = g_new ( GtkImMovPic, 1 ) ;
  pic->width      = width;
  pic->height     = height;
  pic->xo         = 0;
  pic->yo         = 0;
  pic->x1         = 0;
  pic->y1         = 0;
  pic->x2         = 0;
  pic->y2         = 0;
  pic->free       = 0;
  if ( data ) 
    pic->data     = data;
  else {
    pic->data     = g_new ( guchar, width*height*3 );
    memset ( pic->data, 0, width*height*3 );
    pic->free    |= 1;
  }
  pic->id         = id;
  
  return pic ;
}

/**********************************************************************
 *	   Funktions for automatically moving images                  *
 *                                                                    *
 * This is a first simple attempt to move images automatically.       *
 **********************************************************************/

static gulong
_gtk_immov_automatch_single_bright(GtkImMovPic *fix, GtkImMovPic *mov, gint rx, gint ry, gulong *match)
/* Write image data into movable image
 * --- Parameter: ---
 * GtkImMovPic *fix                            : reference image
 * GtkImMOvPic *mov                            : image to move
 * gint         rx                             : border in x-direction
 * gint         ry                             : border in x-direction
 * gulong      *match                          : number of exact matches
 * --- Return: ---
 * gulong _gtk_immov_automatch_single_bright() : brightness of the difference of the images
 */
{
  register gint    d, m = 0;
  register gulong  b = 0;
  register guchar *rap, *map, *mfip;
  guchar          *rs, *ms, *fsp;
  gint             width, height;
  
  g_return_val_if_fail ( fix, 0 );
  g_return_val_if_fail ( mov, 0 );

  if ( (d = _gtk_immov_get_move_dimensions (fix, mov, &width, &height, &rs, &ms)) )
    return 0;
  if ( (width  -= (rx<<1)) < 2 || (height -= (ry<<1)) < 2 ) {
    g_warning(_("Moving image to small to include %s"), mov->id);
    return 0;
  }

  rs += 3*(rx + ry*fix->width);
  ms += 3*(rx + ry*mov->width);

  for ( fsp = ms + 3 * mov->width * height;
        ms < fsp; 
        rs += 3 * fix->width,
        ms += 3 * mov->width ) {
    for ( rap = rs, mfip = (map = ms) + 3*width; map < mfip; 
          rap += 25, map += 25 ) { /* take account to ALL bytes, not only one color!!, but NOT all pixels */
      if ( !(d  = (int)*rap - (int)*map) ) m++;
      else if ( d < 0 ) d = -d;
      b += d;
    }
  }
  *match = (unsigned long)m;
  return b;
}

static gint
_gtk_immov_automatch_check_min(gulong *buf, gint bufstep, gint abs, gint min)
/* Check for valid minimum
 * --- Parameter: ---
 * gulong *buf       : vector of brightnesses
 * gint    bufstep   : step between the each brightness
 * gint    abs       : min has to be between <-abs,abs>
 * gint    min       : index of maximum brightness
 * --- Return: ---
 * gint    _gtk_immov_automatch_check_min(): -1 in case of error
 */
{
  register int            i;
  register unsigned long *ap;
  register long           delta;
   
  if ( !abs || min == -abs || min == abs ) return -1;
  delta = (*(buf + (min+abs)*bufstep)) >> 7;  /* 0.8% vom Minimum */
  for ( i = 1 - abs, ap = buf + bufstep; i < abs; i++, ap += bufstep )
    if ( (long)*(ap-bufstep) - (long)*ap < delta ) break;
  if ( (i-1) == min ) {
    for ( ; i < abs; i++, ap += bufstep )
      if ( (long)*ap - (long)*(ap-bufstep) < delta ) break;
    if ( i == abs ) return 0;  /* upto this point for monotone brightness of differences */
  }
  return -1;
}

static gint
_gtk_immov_automatch_check_max(gulong *buf, gint bufstep, gint abs, gint max)
/* Check for valid maximum
 * --- Parameter: ---
 * gulong *buf       : buffer with number of equal pixels
 * gint    bufstep   : step between the numbers of equal pixels to test
 * gint    abs       : max has to be between <-abs,abs>
 * gint    max       : index of maximam number of equal pixels 
 * --- Return: ---
 * gint    _gtk_immov_automatch_check_max(): -1 in case of error
 */
{
  register int            i;
  register unsigned long *ap;
  register long           delta;

  if ( !abs || max == -abs || max == abs ) return -1;
  delta = (*(buf + (max+abs)*bufstep)) >> 7;  /* 0.8% vom Maximum */ 
  for ( i = 1 - abs, ap = buf + bufstep; i < abs; i++, ap += bufstep )
    if ( (long)*ap - (long)*(ap-bufstep) < delta ) break;
  if ( (i-1) == max ) {
    for ( ; i < abs; i++, ap += bufstep )
      if ( (long)*(ap-bufstep) - (long)*ap < delta ) break;
    if ( i == abs ) return 0;  /* upto this point for monotone matches */
  }
  return -1;
}

static gint
_gtk_immov_automatch_single(GtkImMovPic *fix, GtkImMovPic *mov, gint x, gint y)
/* Try to move images automatically by minimizing the difference between the
 * fix image (reference) and each other image
 * Todo: May be it is useful to calculate the number of equal pixelvalues
 *       to have an additional criterium if the difference method fails
 *
 * <Private remark for Andreas Tille>
 * Eventuell sollte "exakte" bereinstimmung durch
 * " < eps"-bereinstimmung ersetzt werden, vielleicht sogar derart, da exakte bereinstimmung
 * hher bewertet wird, also:
 * if ( d == 0 ) ++m; if ( d < 2 ) ++m; if ( d < 3 ) ++m; ... if ( d < eps ) ++m;
 * Aber ob das was hilft??  Auerdem sind Videobilder zur Zeit nicht aktuell.
 * Das Problem bei Videobildern ist, da die Folge der Differenzen nicht monoton ist.
 * <End private remark>
 * --- Parameter: ---
 * GtkImMovPic *fix                        : referenze image (first one in list)
 * GtkImMovPic *mov                        : image to move
 * gint      x                             : maximum movement in x-direction
 * gint      y                             : maximum movement in y-direction
 * --- Return: ---
 * gint      _gtk_immov_automatch_single() : -1 in case or error
 */
{ 
  gulong        bmin = 0xFFFFFFFF,    /* minimum difference                 */
               *xsum,                 /* sum over all differences           */
               *xmatch,               /* all exact matches                  */
               *xb, 
              **yb,                   /* rows of the matrix of differences  */
              **ayb, **fyb, *xm;
  gint          xa, ya,               /* current position                   */
                xmin = -x, ymin = -y, /* position of minimum difference     */
                mmax = 0,             /* maximum match                      */
                xmax = -x, ymax = -y; /* position of maximum match          */
  GtkImMovPic  *int_fix,              /* internal reference picture         */
               *int_mov;              /* internal picture to move           */
  register gint y2;
   
  g_return_val_if_fail ( fix, -1 );
  g_return_val_if_fail ( mov, -1 );

  y2 = (y<<2) + 2;
  xsum   = g_new0(unsigned long, ((x<<1)+1)*y2);
  xmatch = xsum + ((x<<1)+1)*((y<<1)+1);
  yb     = g_new(unsigned long *, y2);

  for ( fyb = (ayb = yb) + y2, xb = xsum; ayb < fyb; ayb++, xb += (x<<1)+1 ) {
    *ayb = xb;
  }
  
  if ( ( fix->width < mov->width ) || ( fix->height < mov->height ) ) {
    if ( ( fix->width < mov->width ) && ( fix->height < mov->height ) ) {
      int_fix = mov;
      int_mov = fix;
      fix->xo = mov->xo;
      fix->yo = mov->yo;	 
    } else {
      g_warning(_("Unable to automove pictures with strange size."));
      return -1;
    }
  } else {
    int_fix = fix;
    int_mov = mov;
  }
   
  for ( ya = -y, xb = xsum, xm = xmatch; ya <= y; ya++ ) 
    for ( xa = -x; xa <= x; xa++, xb++, xm++ ) {
      int_fix->xo = int_mov->xo + xa;
      int_fix->yo = int_mov->yo + ya;
      if ( (*xb = _gtk_immov_automatch_single_bright(int_fix, int_mov, x, y, xm)) < bmin ) {
        if ( !*xb ) {
          g_free(xsum);
          return -1;
        }
        bmin = *xb;
        xmin = xa;
        ymin = ya;
      }
      if ( *xm > mmax ) {
        mmax = *xm;
        xmax = xa;
        ymax = ya;
      }
    }
  if ( !_gtk_immov_automatch_check_min(xsum+(x*ymin), 1, x, xmin) ) int_mov->xo = xmin;
  else if ( !_gtk_immov_automatch_check_max(xmatch+(x*ymax), 1, x, xmax) ) int_mov->xo = xmax;
  if ( !_gtk_immov_automatch_check_min(xsum+xmin, x, y, ymin)     ) int_mov->yo = ymin;
  else if ( !_gtk_immov_automatch_check_max(xmatch+xmax, x, y, ymax) ) int_mov->yo = ymax;
  if ( fix != int_fix ) { /* if pictures where exchanged due to bigger mov change back now */
    _gtk_immov_iswap(fix->xo, mov->xo);
    _gtk_immov_iswap(fix->yo, mov->yo);
  }
  fix->xo = fix->yo = 0;
  g_free (yb);
  g_free (xsum);
  return 0;
}

void
immov_automatch(GList *l, GtkImMovPic *fix)
/* This function enables silent access to the automatic movement algorithm without
 * necessarily initialising GTK
 * --- Parameter: ---
 * GList       *l   : list of GtkImMov pictures to move
 * GtkImMovPic *fix : fix image
 */
{
  GtkImMovPic *im;

  for ( im = l->data; l && (im = l->data); l = l->next ) {
    if ( _gtk_immov_automatch_single (fix, im, 10,  0) ||
         _gtk_immov_automatch_single (fix, im,  0, 10) ||
         (!im->xo && !im->yo) ) {
      g_warning(_("No automatic movement of %s against %s."), im->id, fix->id);
      continue; 
    }
  }
}

gint
gtk_immov_automatch(GtkWidget *button, GtkImMov *immov)
/* Try automatic movement by minimizing the difference between the images
 * --- Parameter: ---
 * GtkWidget *button
 * GtkImMov  *immov  : structure with the images to move
 */
{
  GtkImMovPic *fix;
  GList       *l;
  /*  gint         flag;   ... set if any image was successfully moved */
   
  if ( button ) g_return_val_if_fail ( GTK_IS_BUTTON(button), -1 ) ;
  g_return_val_if_fail ( immov, -1 );
  g_return_val_if_fail ( GTK_IS_IMMOV(immov), -1 );
  g_return_val_if_fail ( GTK_IS_ADJUSTMENT(immov->vert), -1 );
  g_return_val_if_fail ( GTK_IS_ADJUSTMENT(immov->hori), -1 );
  g_return_val_if_fail ( (fix = immov->fix), -1 );
  g_return_val_if_fail ( fix->width > 0 && fix->height > 0 && fix->data , -1 );
  g_return_val_if_fail ( (l = immov->iml), -1 );
  g_return_val_if_fail ( l->data, -1 );
  
  immov_automatch(l, fix);
  
  gtk_adjustment_set_value(immov->vert, immov->cur->xo);
  gtk_adjustment_set_value(immov->hori, immov->cur->yo);
  /* if ( flag ) _do_move_images(immov); */
  return 0;
}


/**********************************************************************
 *	   internal functions                                         *
 **********************************************************************/

#define LEFT(im)    ( im->xo - (im->width >> 1) - ((im->width)&1) )
#define RIGHT(im)   ( im->xo + (im->width >> 1) )
#define TOP(im)     ( im->yo - (im->height >> 1) - ((im->height)&1) )
#define BOTTOM(im)  ( im->yo + (im->height >> 1) )

gint
immov_accept_images(GList *list, GtkImMovPic *fix)
/* This function enables silent access to the algorithm which calculates the
 * cropping coordinates without necessarily initialising GTK
 * --- Parameter: ---
 * GList       *list : list of GtkImMov pictures to move
 * GtkImMovPic *fix  : fix image
 */
{
  GtkImMovPic *im;
  gint         min_xo, max_xo, min_yo, max_yo,
               checkmove = 0, samesize = 1, eps_x, eps_y;
  GList       *movl;

  min_xo = LEFT(fix);
  max_xo = RIGHT(fix);
  min_yo = TOP(fix);
  max_yo = BOTTOM(fix);

  for ( im = (movl = list)->data; movl && (im = movl->data); movl = movl->next ) {
    if ( (im->x1 = LEFT(im))   > min_xo ) min_xo = im->x1;
    if ( (im->x2 = RIGHT(im))  < max_xo ) max_xo = im->x2;
    if ( (im->y1 = TOP(im))    > min_yo ) min_yo = im->y1;
    if ( (im->y2 = BOTTOM(im)) < max_yo ) max_yo = im->y2;
    if ( fix->width != im->width || fix->width != im->width ) samesize = 0;
    checkmove = abs(im->xo) + abs(im->yo);
  }
  if ( min_xo > max_xo || min_yo > max_yo ) {
    g_warning(_("Can not match all these images."));
    return -1 ;
  }
  
  if ( !checkmove && samesize ) return -1;
      /* There was not done any moving and all images have the same size */

  eps_x = max_xo - min_xo;
  eps_y = max_yo - min_yo;
  
  for ( im = (movl = list)->data; movl && (im = movl->data); movl = movl->next ) {
    im->x2 = (im->x1  = min_xo - im->x1) + eps_x;
    im->y2 = (im->y1  = min_yo - im->y1) + eps_y;
  }

/* moving the fix image */
  fix->x2 = (fix->x1 = min_xo + (fix->width >> 1) + ((fix->width)&1)) + eps_x;
  fix->y2 = (fix->y1 = min_yo + (fix->height >> 1) + ((fix->height)&1)) + eps_y;

  return 0;
}


static void
_gtk_immov_accept_images(GtkWidget *widget, GtkImMov *immov)
/* Calculate the coordinate each single image has to be cropped so that the
 * resulting images will match each other
 * --- Parameter: ---
 * GtkWidget *widget : OK-button inside GtkImReg widget
 * GtkImMov  *immov  : immov widget
 */
{
  g_return_if_fail ( immov ) ;
  g_return_if_fail ( GTK_IS_IMMOV(immov) ) ;
  g_return_if_fail ( immov->fix ) ;
  g_return_if_fail ( immov->iml ) ;

  if ( immov_accept_images(immov->iml, immov->fix) ) {
    DELAY_SHOW = NO_DISPLAY;
    gtk_widget_destroy(GTK_WIDGET(immov));     
    return ;
  }
}

static void
gtk_immov_destroy (GtkObject *object)
{
  GtkImMov *immov;

  g_return_if_fail (object != NULL);
  g_return_if_fail (GTK_IS_IMMOV (object));

  DELAY_SHOW = DESTROY;
  
  immov = GTK_IMMOV (object);
  if ( TIMEOUT_TAG ) _destroy_changing_imlibs();

    /* #ifdef _hupsi_ */
  if ( immov->pm ) gdk_imlib_free_pixmap(immov->pm);
  immov->pm = NULL;
    /* #endif */

  gtk_immov_free_struct (immov->im) ;
  immov->im = NULL;
  
  if (GTK_OBJECT_CLASS (parent_class)->destroy)
    (* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

static void
gtk_immov_configure(GtkWidget *widget, GdkEventConfigure *event, GtkImMov *immov)
/* Create a new backing pixmap of the appropriate size 
 * --- Parameter: ---
 * GtkWidget         *widget : drawing area with image to cut and rubber box
 * GdkEventConfigure *event  : configure_event
 * GtkImMov          *immov  : image move structure
 */
{
    /* #ifdef _hupsi_ */
  GdkImlibImage *iml;
    /* #endif */
  gint w, h;

  if ( DELAY_SHOW == NO_DISPLAY ) return ;
  g_return_if_fail ( widget );
  g_return_if_fail ( immov );
  g_return_if_fail ( GTK_IS_DRAWING_AREA (widget) );
  g_return_if_fail ( GTK_IS_IMMOV (immov) );
  g_return_if_fail ( immov->im ) ;
  if ( event ) g_return_if_fail ( event->type == GDK_CONFIGURE );

  if ( !immov->draw || !immov->draw->window ) return ;
  DELAY_SHOW = NO_DISPLAY;

  gdk_window_get_size (immov->draw->parent->window, &w, &h) ;
  if ( w != immov->im->width || h != immov->im->height ) {
    gdk_window_resize(immov->draw->parent->window, immov->im->width, immov->im->height);
    gdk_window_resize(immov->draw->window, immov->im->width, immov->im->height);
  }
    gtk_widget_show_all (immov->draw->parent) ;

  /*
  gdk_draw_rgb_image (immov->draw->window, immov->draw->style->white_gc,
                      0, 0, immov->im->width, immov->im->height, GDK_RGB_DITHER_NONE,
                      immov->im->data, immov->im->width * 3);

      #ifdef _hupsi_ */

  if ( immov->pm ) gdk_imlib_free_pixmap(immov->pm); /* free the pixmap, not sure if this is necessary */
  g_return_if_fail ( (iml = gdk_imlib_create_image_from_data(immov->im->data, NULL, 
                                                             immov->im->width, immov->im->height)) );

  g_return_if_fail ( gdk_imlib_render(iml, immov->im->width, immov->im->height) ) ;
  immov->pm = gdk_imlib_copy_image(iml);

  gdk_window_set_back_pixmap(immov->draw->window, immov->pm, 0);
  gdk_window_clear(immov->draw->window); 

  gdk_imlib_kill_image (iml) ;
  gdk_window_show(immov->draw->window);
  gdk_flush();
  /* #endif */
  
  DELAY_SHOW = DISPLAY;
  return;
}

static void
gtk_immov_expose(GtkWidget *widget, GdkEventExpose *event, GtkImMov *immov)
/* Redraw the screen from the backing pixmap 
 * --- Parameter: ---
 * GtkWidget      *widget : drawing area with image to cut and rubber box
 * GdkEventExpose *event  : expose_event
 * GtkImMov       *immov  : image move structure
 */
{
  if ( DELAY_SHOW == NO_DISPLAY ) return ;
  g_return_if_fail ( widget );
  g_return_if_fail ( immov );
  g_return_if_fail ( GTK_IS_DRAWING_AREA (widget) );
  g_return_if_fail ( GTK_IS_IMMOV (immov) );
  if  ( !immov->pm ) return ;
  g_return_if_fail ( event );
  g_return_if_fail ( event->type == GDK_EXPOSE );

  gdk_draw_pixmap(widget->window, widget->style->fg_gc[GTK_WIDGET_STATE(widget)], immov->pm,
		  event->area.x, event->area.y, event->area.x, event->area.y,
		  event->area.width, event->area.height);
}

static GtkWidget *
_gtk_immov_init_drawing_area(GtkImMov *immov)
/* create drawing area inside event box and set up signal handlers
 * --- Parameter: ---
 * GtkImMov  *immov                          : immov widget to set up signal functions correctly
 * --- Return: ---
 * GtkWidget *_gtk_immov_init_drawing_area() : new drawing area
 */
{
  GtkWidget *event_box, *draw;
   
  draw = gtk_drawing_area_new();

  /**********************************************************************************
   * all the signals and events of the pixmap area                                  *
   **********************************************************************************/
  gtk_widget_set_events(draw, GDK_EXPOSURE_MASK | GDK_LEAVE_NOTIFY_MASK | GDK_BUTTON_PRESS_MASK 
			 | GDK_BUTTON_RELEASE_MASK | GDK_POINTER_MOTION_MASK | GDK_ENTER_NOTIFY_MASK );
  /* Signals used to handle backing pixmap */
  gtk_signal_connect(GTK_OBJECT(draw), "expose_event", GTK_SIGNAL_FUNC(gtk_immov_expose), immov);
  gtk_signal_connect(GTK_OBJECT(draw), "configure_event", GTK_SIGNAL_FUNC(gtk_immov_configure), immov);

  event_box = gtk_event_box_new ();
  gtk_container_add ( GTK_CONTAINER(event_box), draw );
  if ( !(immov->draw) ) immov->draw = draw;
  return draw;
}

static void
_gtk_immov_page_switch(GtkWidget *widget, GtkNotebookPage *page, gint page_num)
/* switch notebook pages and link drawing area to the current page
 * --- Parameter: ---
 * GtkWidget       *widget   : immov->nb, immov set as userdata
 * GtkNotebookPage *page     : current notebook page, event_box set as user data
 * gint             page_num : current page number
 */
{
  GtkImMov    *immov;
  GtkWidget   *draw;
  GtkImMovPic *im;
  static gint  last_displayed_page = -1;

  if ( DELAY_SHOW == DESTROY ) return ;
  if ( last_displayed_page == page_num ) return ;
  last_displayed_page = page_num ;
  
  g_return_if_fail ( widget ) ;
  g_return_if_fail ( GTK_IS_NOTEBOOK (widget) );
  g_return_if_fail ( (immov = gtk_object_get_user_data ( GTK_OBJECT(widget) )) );
  g_return_if_fail ( GTK_IS_IMMOV (immov) );
  g_return_if_fail ( page ) ;
  g_return_if_fail ( page->child ) ; 
  g_return_if_fail ( GTK_IS_FRAME (page->child) );
  g_return_if_fail ( (draw = gtk_object_get_user_data ( GTK_OBJECT(page->child) )) );
  g_return_if_fail ( GTK_IS_DRAWING_AREA(draw) ) ;
  g_return_if_fail ( (im = gtk_object_get_user_data ( GTK_OBJECT(draw) )) );
  
  immov->draw = draw;
  gtk_immov_configure (GTK_IMMOV(immov)->draw, 0, GTK_IMMOV(immov));
  immov->cur = im;
  _gtk_immov_update_struct (immov);
  DELAY_SHOW = NO_DISPLAY;
  gtk_adjustment_set_value(immov->vert, immov->cur->xo);
  gtk_adjustment_set_value(immov->hori, immov->cur->yo);
  DELAY_SHOW = DISPLAY;

  immov->show(GTK_WIDGET(immov));
}


/**********************************************************************************
 * functions to create moving image                                               *
 **********************************************************************************/

static GdkImlibImage *ImlibImg = NULL, *ChangeImg = NULL;
static int            ChangeWhich = 0;
static GtkImMovPic   *ChangePic;

static void _destroy_changing_imlibs ( void )
{
  gtk_timeout_remove(TIMEOUT_TAG);
  if ( ChangeImg ) {
    gdk_imlib_kill_image( ChangeImg );
    ChangeImg = NULL;
  }
  if ( ImlibImg )  {
    gdk_imlib_kill_image( ImlibImg );
    ImlibImg = NULL;
  }
  gtk_immov_free_struct (ChangePic) ;
  ChangePic = NULL;
}

static int _timeout_change_move_picture(GtkWidget *immov)
/* timeout-function for animation of images to move
 * --- parameter: ---
 * GtkWidget *immov                        : moving images structure
 * --- return:    ---
 * int       _timeout_change_move_picture(): 0 in case of error
 */
{
  GdkImlibImage *im;
  GdkPixmap     *pm;
  GtkWidget     *draw;

  g_return_val_if_fail ( immov, 0 );
  g_return_val_if_fail ( GTK_IS_IMMOV(immov), 0 );

  if ( !(draw = GTK_IMMOV(immov)->draw) ) return 1;
  if ( !(draw->window) ) return 1;
  DELAY_SHOW = NO_DISPLAY;

  if ( !ChangeWhich ) im = ImlibImg;
  else                im = ChangeImg;
  ChangeWhich = ~ChangeWhich ;
  g_return_val_if_fail ( im, 0 );
  
  /*  if ( ANIFLAG ) return 1;  / * avoid displaying if timout is faster than display */
  g_return_val_if_fail ( gdk_imlib_render(im, im->rgb_width, im->rgb_height), 0 ) ;
  pm = gdk_imlib_copy_image(im);
  gdk_window_set_back_pixmap(draw->window, pm, 0);
  gdk_window_clear(draw->window); 

  gdk_window_show(draw->window);
  gdk_flush();
  /* #endif */
  
  DELAY_SHOW = DISPLAY;

  return 1;
}

static gint
_change_move_picture(GtkWidget *immov)
/* install timeout function which switches continousely between both images
 * --- Parameter: ---
 * GtkWidget     *immov        : moving images structure
 * --- Return: ---
 * gint _change_move_picture() : -1 in case of error
 */
{
  GtkImMovPic  *im;
  gint          xo, yo;
  GtkWidget    *draw;

  if ( DELAY_SHOW == DESTROY ) return 0;

  g_return_val_if_fail ( immov, -1 );
  g_return_val_if_fail ( GTK_IS_IMMOV(immov), -1 );
  g_return_val_if_fail ( GTK_IMMOV(immov)->fix, -1 );
  g_return_val_if_fail ( GTK_IMMOV(immov)->cur, -1 );
  g_return_val_if_fail ( (im = GTK_IMMOV(immov)->im), -1 );

  if ( !(draw = GTK_IMMOV(immov)->draw) ) return 1;
  if ( !(draw->window) ) return 1;

    /* put fix image into moving image */
  if ( !ImlibImg ) {
    xo = im->xo; yo = im->yo;
    im->xo = 0;  im->yo = 0;
    _gtk_immov_insert_picture (im, GTK_IMMOV(immov)->fix, 3); 
    im->xo = xo; im->yo = yo;
    g_return_val_if_fail ( (ImlibImg  = gdk_imlib_create_image_from_data(im->data, NULL, 
                                                                         im->width, im->height)), -1 );
  }

  if ( !ChangePic ) ChangePic  = gtk_immov_set_struct (im->width, im->height, NULL, "changing image");
    /* put current image into change which is only needed for this method */
  ChangePic->xo = im->xo;
  ChangePic->yo = im->yo; 
  _gtk_immov_insert_picture (ChangePic, GTK_IMMOV(immov)->cur, 3);

  gdk_window_get_size (draw->parent->window, &xo, &yo) ;
  if ( xo != im->width || yo != im->height ) {
    gdk_window_resize(draw->parent->window, im->width, im->height);
    gdk_window_resize(draw->window, im->width, im->height);
  }
    gtk_widget_show_all (draw->parent) ;

  if ( GTK_IMMOV(immov)->pm ) 
    gdk_imlib_free_pixmap ( GTK_IMMOV(immov)->pm ); /* free the pixmap, not sure if this is necessary */
  GTK_IMMOV(immov)->pm = NULL;
  
  if ( ChangeImg ) gdk_imlib_kill_image( ChangeImg );
  
  g_return_val_if_fail ( (ChangeImg = gdk_imlib_create_image_from_data(ChangePic->data, NULL, 
                                                            ChangePic->width, ChangePic->height)), -1 );
  if ( TIMEOUT_TAG ) gtk_timeout_remove(TIMEOUT_TAG);
  TIMEOUT_TAG = gtk_timeout_add(CHANGE_TIME, (GtkFunction)_timeout_change_move_picture, immov);
 
  return 0;
}

static gint
_color_move_picture(GtkWidget *immov)
/* write image data into color move image
 * --- Parameter: ---
 * GtkWidget     *immov       : moving images structure
 * --- Return: ---
 * gint _color_move_picture() : -1 in case of error
 */
{
  GtkImMovPic  *im;
  gint          xo, yo;
   
  if ( DELAY_SHOW == DESTROY ) return 0;

  g_return_val_if_fail ( immov, -1 );
  g_return_val_if_fail ( GTK_IS_IMMOV(immov), -1 );
  g_return_val_if_fail ( GTK_IMMOV(immov)->fix, -1 );
  g_return_val_if_fail ( GTK_IMMOV(immov)->cur, -1 );
  g_return_val_if_fail ( (im = GTK_IMMOV(immov)->im), -1 );

  if ( TIMEOUT_TAG ) _destroy_changing_imlibs();

    /* put fix image as green part into moving image */
  xo = im->xo; yo = im->yo;
  im->xo = 0;  im->yo = 0;
  _gtk_immov_insert_picture (im, GTK_IMMOV(immov)->fix, 0); 
  im->xo = xo; im->yo = yo;

    /* put fix image as blue part into moving image */
  _gtk_immov_insert_picture (im, GTK_IMMOV(immov)->cur, 1);   

  gtk_immov_configure (GTK_IMMOV(immov)->draw, 0, GTK_IMMOV(immov));
  
  return 0;
}

static gint
_difference_move_picture(GtkWidget *immov)
/* write image data into difference move image
 * --- Parameter: ---
 * GtkWidget     *immov            : moving images structure
 * --- Return: ---
 * gint _difference_move_picture() : -1 in case of error
 */
{
  register gint    d;
  register guchar *tap, *sap, *u, *v, *fap;
  guchar          *ts, *ss, *fsp,
                   offset=10, eps=0;
  gint             width, height, sxoff, syoff,
                   scale = 2;
  GtkImMovPic     *fix, *cur, *im;
  
  if ( DELAY_SHOW == DESTROY ) return 0;
  
  g_return_val_if_fail ( immov, -1 );
  g_return_val_if_fail ( GTK_IS_IMMOV(immov), -1 );
  g_return_val_if_fail ( (fix = GTK_IMMOV(immov)->fix), -1 );
  g_return_val_if_fail ( (cur = GTK_IMMOV(immov)->cur), -1 );
  g_return_val_if_fail ( (im  = GTK_IMMOV(immov)->im), -1 );

  if ( TIMEOUT_TAG ) _destroy_changing_imlibs();

  if ( ((im->width - fix->width) >> 1) < 0 ||
       ((im->height - fix->height) >> 1) < 0 ) {
    g_warning(_("Moving image to small to include %s"), fix->id);
    return -1;
  }

  if ( (d = _gtk_immov_get_move_dimensions(im, cur, &width, &height, &ts, &ss)) )
    return d;

  ss++;  /* +1 == green == real information */
  ts++;  /* +1 == green !! */

  _gtk_immov_clear_target (im, 1, 1, 1);
  sxoff  = im->xo; syoff  = im->yo;
  im->xo = 0;      im->yo = 0;
  _gtk_immov_insert_picture (im, fix, 0);   /* put fix image into moving image */
  im->xo = sxoff;  im->yo = syoff;
   
  for ( fsp = ss + 3 * cur->width * height;
        ss < fsp; 
        ts += 3 * im->width,
	ss += 3 * cur->width ) {
    for ( tap = ts, fap = (sap = ss) + 3*width; sap < fap; 
          sap += 3, tap += 3 ) {
      if ( (d  = (int)*tap - (int)*sap) < 0 ) {
        u = tap - 1;  /* bild1 < bild2 => store difference in red */
        v = tap + 1;  /*               => set blue 0              */
      } else {
        u = tap + 1;
        v = tap - 1;
      }
      if ( (d = abs(d)) < eps ) { /* suppress noise */
        *(tap-1) = *tap = *(tap+1) = 0xFF;  /* remainder white in favour of black */
        continue;   
      }
      d    = d * scale + offset;     /* enhance Contrast and brightness */
      *u   = (guchar)(d < 0x100 ? d : 0xFF);
      *tap = *v = 0;
    }
  }

  gtk_immov_configure(GTK_IMMOV(immov)->draw, 0, GTK_IMMOV(immov));
  
  return 0;
}

static gint 
_gtk_immov_get_move_dimensions(GtkImMovPic *target, GtkImMovPic *src, gint *width, gint *height,
                               guchar **tptr, guchar **sptr)
/* Obtain pointer to start of common area in each image and the dimensions of this common area
 * --- Parameter: ---
 * GtkImMovPic *target   : moving image
 * GtkImMovPic *src      : image to insert
 * gint        *width    : width of moving image
 * gint        *height   : height of moving image
 * guchar     **tptr     : pointer to start of pixels to write in the moving image
 * guchar     **sptr     : pointer to start of pixels to write from the source image
 * --- Return: ---
 * gint _gtk_immov_get_move_dimensions() : 0 if OK, else -1
 * gint        *width    : width of moving image
 * gint        *height   : height of moving image
 * guchar     **tptr     : pointer to start of pixels to write in the moving image
 * guchar     **sptr     : pointer to start of pixels to write from the source image
 */
{
  register gint xt0, yt0, /* left top corner */
               toffset;

  if ( (xt0 = (target->width - src->width) >> 1) < 0 ||
       (yt0 = (target->height - src->height) >> 1) < 0 ) {
    g_warning(_("Moving image to small to include %s"), src->id);
    return -1;
  }

  xt0 += target->xo;
  yt0 += target->yo;
  
  if ( xt0 > 0 ) *width  = MIN(target->width  - xt0, src->width);
  else           *width  = MIN(src->width     + xt0, target->width);
  if ( yt0 > 0 ) *height = MIN(target->height - yt0, src->height);
  else           *height = MIN(src->height    + yt0, target->height);

  if ( *width <= 0 || *height <= 0 ) {
    gdk_beep();
    return 1;
  }
      
  if ( (toffset = yt0*target->width + (xt0<0?0:xt0)) <= 0 ) 
    *tptr = target->data + 3*(xt0<0?0:xt0);
  else *tptr = target->data + 3*toffset;

  *sptr  = src->data;
  if ( yt0 < 0 ) *sptr -= 3 * src->width * yt0;
  if ( xt0 < 0 ) *sptr -= 3 * xt0;

  return 0;
}

static void
_gtk_immov_clear_target(GtkImMovPic *im, gint red, gint green, gint blue)
/* Delete Red/Green/Blue-part of a moving image
 * --- Parameter: ---
 * GtkImMovPic *im       : moving image
 * gint         red      : delete red ?
 * gint         green    : delete green ?
 * gint         blau     : delete blue ?
 */
{
  register guchar *ap, *fip;
  register gint    size;

  g_return_if_fail ( im ) ;
  size = 3 * im->width * im->height;
  
  if ( red && green && blue ) {
    memset(im->data, 0, size);
    return;
  }
  if ( red )
    for ( fip = (ap = im->data)   + size; 
          ap < fip; ap += 3 )
      *ap = 0;
  if ( green )
    for ( fip = (ap = im->data+1)   + size; 
          ap < fip; ap += 3 )
      *ap = 0;
  if ( blue )
    for ( fip = (ap = im->data+2) + size; 
          ap < fip; ap += 3 )
      *ap = 0;
}

static gint
_gtk_immov_insert_picture (GtkImMovPic *target, GtkImMovPic *src, gint colshift)
/* write image data into moving image
 * --- Parameter: ---
 * GtkImMovPic *target                      : moving image
 * GtkImMovPic *src                         : image to insert
 * gint         colshift                    : == 0 => reference image, use green
 *                                            == 1 => blue
 *                                            ==-1 => red
 * --- Return: ---
 * GtkImMovPic *target                      : moving image with included image
 * gint         _gtk_immov_insert_picture() : 0 if OK, else -1
 */
{
  register gint    step;
  register guchar *tap, *sap, *fap;
  guchar          *ts,  *ss,  *fsp;
  gint             width, height, i;

  g_return_val_if_fail ( target, -1 );
  g_return_val_if_fail ( src, -1 );
  
  if ( (i = _gtk_immov_get_move_dimensions(target, src, &width, &height, &ts, &ss)) )
    return i;
   
  if ( abs(colshift) <= 1 ) {
    ts += colshift + 1;  /* +1 == green !!                  */
    ss++;                /* +1 == green == true information */

    _gtk_immov_clear_target(target, colshift>0?0:1, !colshift, colshift>0?1:0);
    /* go foreward by 3 bytes so only one color is inserted into the target image */
    step = 3;
  } else {
    /* insert each byte so only all colors are inserted into the target image */
    _gtk_immov_clear_target(target, 1, 1, 1);
    step = 1;
  }
  
  for ( fsp = ss + 3 * src->width * height;
        ss < fsp; 
        ts += 3 * target->width,
        ss += 3 *    src->width ) {
    for ( tap = ts, fap = (sap = ss) + 3*width; sap < fap; 
          sap += step, tap += step ) {
      *tap = *sap;	 
    }
  }
  return 0;
}

/**********************************************************************
 *	   image move internal utilities                              *
 **********************************************************************/

static void
_gtk_immov_update_struct (GtkImMov *immov)
/* change parameters of GtkImMovPic *immov->im struct
 * VERY IMPORTANT:
 *    It is assumed, that the space required for the data pointer was set up
 *    big enough!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
 */
{
  gint   w, h;
  
   
  g_return_if_fail ( immov ) ;
  g_return_if_fail ( GTK_IS_IMMOV( immov ) ) ;  
  
  w    = immov->fix->width;
  h    = immov->fix->height;
  if ( w < immov->cur->width  ) w = immov->cur->width ;
  if ( h < immov->cur->height ) h = immov->cur->height ; 
  immov->maxvert = (w>>1) + 1;
  immov->maxhori = (h>>1) + 1;
  immov->im->width  = w;
  immov->im->height = h;
}

typedef struct _OptionMenuItem
{
  gchar           *name;
  GtkImMovShowFunc func;
  gint             index;
} OptionMenuItem;

static OptionMenuItem 
items[] = {{ NULL, _difference_move_picture, 2 },
           { NULL, _color_move_picture,      1 },
           { NULL, _change_move_picture,     0 },
           {NULL,                   NULL }
          };

static void 
_gtk_immov_set_move_method (GtkWidget *widget, GtkImMov *immov)
{
  GtkWidget      *omenu;
  OptionMenuItem *item;
  int             i;
   
  if (!GTK_WIDGET_MAPPED (widget)) return;
  g_return_if_fail ( immov ) ;
  g_return_if_fail ( GTK_IS_IMMOV(immov) ) ;
  g_return_if_fail ( immov->vert ) ;
  g_return_if_fail ( GTK_IS_ADJUSTMENT(immov->vert) ) ;
  g_return_if_fail ( (omenu = gtk_object_get_data (GTK_OBJECT(widget), "user_data")) );
  g_return_if_fail ( GTK_IS_OPTION_MENU(omenu) );
  g_return_if_fail ( (item = gtk_object_get_data (GTK_OBJECT(widget), "menu_item")) );

  RADIOMENUTOGGLED ((GtkRadioMenuItem *)(((GtkOptionMenu *)omenu)->menu_item), i);
  if ( item->index != i ) return ;
  immov->show    = item->func;
  gtk_signal_emit_by_name (GTK_OBJECT(immov->vert), "value_changed");
}

static GtkWidget *
_build_option_menu (GtkImMov *immov)
{
  OptionMenuItem *ip;
  GtkWidget      *omenu, *menu, *menu_item;
  GSList         *group  = NULL;

  omenu = gtk_option_menu_new ();
  menu  = gtk_menu_new ();
  
  items->name     = _("Difference");
  (items+1)->name = _("Different colors");
  (items+2)->name = _("Changing images");

  for ( ip = items; ip->name ; ip++ )
    {
      menu_item = gtk_radio_menu_item_new_with_label (group, ip->name);
      gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
			  (GtkSignalFunc) _gtk_immov_set_move_method, immov);
      gtk_object_set_data (GTK_OBJECT(menu_item), "user_data", omenu);
      gtk_object_set_data (GTK_OBJECT(menu_item), "menu_item", ip);
      group = gtk_radio_menu_item_group (GTK_RADIO_MENU_ITEM (menu_item));
      gtk_menu_append (GTK_MENU (menu), menu_item);
      if ( ip == items )
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (menu_item), TRUE);
      gtk_widget_show (menu_item);
    }

  gtk_option_menu_set_menu (GTK_OPTION_MENU (omenu), menu);
  gtk_option_menu_set_history (GTK_OPTION_MENU (omenu), 0);
  
  return omenu;
}

static GtkAdjustment*
_create_adjustment(GtkWidget *table, GtkImMov *immov, gchar *name, gint pos, 
                           gint *value, gint max)
/* create spinners for coordinate input
 * --- Parameters: ---
 * GtkImMov *immov  : image move structure
 * gchar    *name   : name of parameter
 * gint      pos    : position
 * gint      value  : initial value of parameter
 * gint      max    : maximum value of parameter
 */
{
  GtkAdjustment *adj;
  GtkWidget     *label, *spinner;
   
  label   = gtk_label_new(name);
  gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, pos, pos+1);
  gtk_widget_show(label);

  adj     = GTK_ADJUSTMENT(gtk_adjustment_new((gfloat)(*value), -max, max, 1, 0.0, 0.0));
  spinner = gtk_spin_button_new(GTK_ADJUSTMENT(adj), 1, 0);
  gtk_widget_set_usize(spinner, 80, 0);
  gtk_spin_button_set_update_policy(GTK_SPIN_BUTTON(spinner), GTK_UPDATE_ALWAYS);
  gtk_object_set_user_data(GTK_OBJECT(adj), value);
  gtk_object_set_user_data(GTK_OBJECT(spinner), immov);
  gtk_signal_connect(GTK_OBJECT(adj), "value_changed", 
                     GTK_SIGNAL_FUNC(gtk_immov_value_changed), spinner);
  gtk_table_attach_defaults(GTK_TABLE(table), spinner, 1, 2, pos, pos+1);
  gtk_widget_show(spinner);
  
  return adj;
}

static void
gtk_immov_value_changed(GtkAdjustment *adj, GtkWidget *spinner)
/* called while editing box parameters
 * --- Parameter: ---
 * GtkAdjustment *adj     : target for new value
 * GtkWidget     *spinner :
 */
{
  gint     *chg, new;
  GtkImMov *immov;
  
  g_return_if_fail ( adj );
  g_return_if_fail ( spinner );
  g_return_if_fail ( GTK_IS_OBJECT (adj) );
  g_return_if_fail ( GTK_IS_SPIN_BUTTON (spinner) );
  immov = gtk_object_get_user_data(GTK_OBJECT(spinner));
  g_return_if_fail ( immov );
  g_return_if_fail ( GTK_IS_IMMOV (immov) );

  chg = gtk_object_get_user_data(GTK_OBJECT(adj));
  new = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spinner));

  if      ( chg == &(immov->im->xo) ) {
    if ( !_set_parameter_if_valid(&(immov->im->xo), new, 
                                  new <= immov->maxvert && new >= -(immov->maxvert), adj) )
      return;
    immov->cur->xo = immov->im->xo;  /* This would be more efficient when done not at        *
				      * each update but only when switching pages or leaving *
                                      * the widget, but who cares?                           */
  } else if ( chg == &(immov->im->yo) ) {
    if ( !_set_parameter_if_valid(&(immov->im->yo), new, 
                                  new <= immov->maxhori && new >= -(immov->maxhori), adj) )
      return;
    immov->cur->yo = immov->im->yo;  /* This would be more efficient when done not at        *
                                      * each update but only when switching pages or leaving *
                                      * the widget, but who cares?                           */
  } else return;
  immov->show(GTK_WIDGET(immov));
}

static gint 
_set_parameter_if_valid(gint *val2set, gint val, gint cond, GtkAdjustment *adj)
/* Set value in address if condition is true
 */
{
  g_return_val_if_fail ( adj , FALSE );
  g_return_val_if_fail ( GTK_IS_ADJUSTMENT (adj), FALSE );

  if ( cond ) {
    *val2set = val;
    return TRUE;
  }
  if ( *val2set == val ) return FALSE;  /* it's not necessary to set the value */
  gdk_beep();
  gtk_adjustment_set_value(adj, *val2set);
  return FALSE;
}

