#include "Python.h"
#include <musicbrainz/mb_c.h>
#include "pymusicbrainz.h"
#include <string.h>
#include <stdio.h>

#define PY_MB(x) (((py_mb *)x)->mb)

typedef struct {
  PyObject_HEAD
  musicbrainz_t *mb;
} py_mb;

/* functions used as python interfaces */ 

static PyObject * py_mb_GetVersion (PyObject *self, PyObject *args);
static PyObject * py_mb_SetServer (PyObject *self, PyObject *args);
static PyObject * py_mb_SetDebug (PyObject *self, PyObject *args);
static PyObject * py_mb_SetProxy (PyObject *self, PyObject *args);
static PyObject * py_mb_Authenticate (PyObject *self, PyObject *args);
static PyObject * py_mb_SetDevice (PyObject *self, PyObject *args);
static PyObject * py_mb_SetDepth (PyObject *self, PyObject *args);
static PyObject * py_mb_SetMaxItems (PyObject *self, PyObject *args);
static PyObject * py_mb_Query (PyObject *self, PyObject *args);
static PyObject * py_mb_QueryWithArgs (PyObject *self, PyObject *args);
static PyObject * py_mb_GetWebSubmitURL (PyObject *self, PyObject *args);
static PyObject * py_mb_GetQueryError (PyObject *self, PyObject *args);
static PyObject * py_mb_Select (PyObject *self, PyObject *args);
static PyObject * py_mb_Select1 (PyObject *self, PyObject *args);
static PyObject * py_mb_SelectWithArgs (PyObject *self, PyObject *args);
static PyObject * py_mb_GetResultData (PyObject *self, PyObject *args);
static PyObject * py_mb_GetResultData1 (PyObject *self, PyObject *args);
static PyObject * py_mb_DoesResultExist (PyObject *self, PyObject *args);
static PyObject * py_mb_DoesResultExist1 (PyObject *self, PyObject *args);
static PyObject * py_mb_GetResultInt (PyObject *self, PyObject *args);
static PyObject * py_mb_GetResultInt1 (PyObject *self, PyObject *args);
static PyObject * py_mb_GetResultRDF (PyObject *self, PyObject *args);
static PyObject * py_mb_SetResultRDF (PyObject *self, PyObject *args);
static PyObject * py_mb_GetIDFromURL (PyObject *self, PyObject *args);
static PyObject * py_mb_GetFragmentFromURL (PyObject *self, PyObject *args);
static PyObject * py_mb_GetOrdinalFromList (PyObject *self, PyObject *args);
static PyObject * py_mb_CalculateSha1 (PyObject *self, PyObject *args);
static PyObject * py_mb_CalculateBitprint (PyObject *self, PyObject *args);
static PyObject * py_mb_GetMP3Info (PyObject *self, PyObject *args);
static PyObject * py_mb_getattr(PyObject *self, char *name);
static void py_mb_destructor (PyObject *self);

/* python methods declared by this module */
static PyMethodDef MB_Methods[] = {
  {"GetVersion",            py_mb_GetVersion, METH_VARARGS},
  {"SetServer",             py_mb_SetServer, METH_VARARGS},
  {"SetDebug",              py_mb_SetDebug, METH_VARARGS},
  {"SetProxy",              py_mb_SetProxy, METH_VARARGS},
  {"Authenticate",          py_mb_Authenticate, METH_VARARGS},
  {"SetDevice",             py_mb_SetDevice, METH_VARARGS},
  {"SetDepth",              py_mb_SetDepth, METH_VARARGS},
  {"SetMaxItems",           py_mb_SetMaxItems, METH_VARARGS},
  {"Query",                 py_mb_Query, METH_VARARGS},
  {"QueryWithArgs",         py_mb_QueryWithArgs, METH_VARARGS},
  {"GetWebSubmitURL",       py_mb_GetWebSubmitURL, METH_VARARGS},
  {"GetQueryError",         py_mb_GetQueryError, METH_VARARGS},
  {"Select",                py_mb_Select, METH_VARARGS},
  {"Select1",               py_mb_Select1, METH_VARARGS},
  {"SelectWithArgs",        py_mb_SelectWithArgs, METH_VARARGS},
  {"GetResultData",         py_mb_GetResultData, METH_VARARGS},
  {"GetResultData1",        py_mb_GetResultData1, METH_VARARGS},
  {"DoesResultExist",       py_mb_DoesResultExist, METH_VARARGS},
  {"DoesResultExist1",      py_mb_DoesResultExist1, METH_VARARGS},
  {"GetResultInt",          py_mb_GetResultInt, METH_VARARGS},
  {"GetResultInt1",         py_mb_GetResultInt1, METH_VARARGS},
  {"GetResultRDF",          py_mb_GetResultRDF, METH_VARARGS},
  {"SetResultRDF",          py_mb_SetResultRDF, METH_VARARGS},
  {"GetIDFromURL",          py_mb_GetIDFromURL, METH_VARARGS},
  {"GetFragmentFromURL",    py_mb_GetFragmentFromURL, METH_VARARGS},
  {"GetOrdinalFromList",    py_mb_GetOrdinalFromList, METH_VARARGS},
  {"CalculateSha1",         py_mb_CalculateSha1, METH_VARARGS},
  {"CalculateBitprint",     py_mb_CalculateBitprint, METH_VARARGS},
  {"GetMP3Info",            py_mb_GetMP3Info, METH_VARARGS},
  {NULL,      NULL}         /* Sentinel */
};

static char MB_Doc[] = "\
To be written.\n\
";

PyTypeObject py_mb_type = {
  PyObject_HEAD_INIT(&PyType_Type)
  0,
  "mb",
  sizeof(py_mb),
  0,

  /* Standard Methods */
  /* destructor */ py_mb_destructor,
  /* printfunc */ 0,
  /* getattrfunc */ py_mb_getattr,
  /* setattrfunc */ 0,
  /* cmpfunc */ 0,
  /* reprfunc */ 0,
  
  /* Type Categories */
  0, /* as number */
  0, /* as sequence */
  0, /* as mapping */
  0, /* hash */
  0, /* binary */
  0, /* repr */
  0, /* getattro */
  0, /* setattro */
  0, /* as buffer */
  0, /* tp_flags */
  MB_Doc,
};


PyObject * 
py_mb_new (PyObject *notused, PyObject *args) 
{
  py_mb *self;
  
  self = PyObject_NEW(py_mb, &py_mb_type);
  if (self == NULL)
    return NULL;
  self->mb = mb_New();
  /* turn on unicode */
  mb_UseUTF8(self->mb, 1);
  
  
  return (PyObject *)self;  
}

static PyObject*
py_mb_getattr(PyObject *self, char *name)
{
  return Py_FindMethod(MB_Methods, self, name);
}
  
static void
py_mb_destructor (PyObject *self)
{
  if (PY_MB(self))
    mb_Delete(PY_MB(self));
      
  PyMem_DEL(self);
}

static PyObject * 
py_mb_GetVersion (PyObject *self, PyObject *args)
{
  int major, minor, rev;
  PyObject * tuple;
  
  if (!PyArg_ParseTuple(args, ""))
    return NULL;
      
  mb_GetVersion(PY_MB(self), &major, &minor, &rev);

  tuple = PyTuple_New(3);
  if(PyTuple_SetItem(tuple, 0, PyInt_FromLong((long)major)))
    return NULL;
  if(PyTuple_SetItem(tuple, 1, PyInt_FromLong((long)minor)))
    return NULL;
  if(PyTuple_SetItem(tuple, 2, PyInt_FromLong((long)rev)))
    return NULL;
  
  return tuple;
}

static PyObject * 
py_mb_SetServer (PyObject *self, PyObject *args)
{
  char * serverAddr;
  short serverPort;
  int retval;

  if (!PyArg_ParseTuple(args, "sh", &serverAddr, &serverPort))
    return NULL;
        
  retval = mb_SetServer (PY_MB(self), serverAddr, serverPort);

  if(!retval)
    return py_mb_raise_exception("Could not set server to \"%s\", port %i", serverAddr, serverPort);

  Py_INCREF(Py_None); 
  return Py_None;
}

static PyObject * 
py_mb_SetDebug (PyObject *self, PyObject *args)
{
  int debug;
   
  if (!PyArg_ParseTuple(args, "i", &debug))
    return NULL;

  mb_SetDebug(PY_MB(self), debug);

  Py_INCREF(Py_None); 
  return Py_None;
}

static PyObject * 
py_mb_SetProxy (PyObject *self, PyObject *args)
{
  char *proxyAddr;
  short proxyPort;
  int retval;

  if (!PyArg_ParseTuple(args, "sh", &proxyAddr, &proxyPort))
    return NULL;
        
  retval = mb_SetProxy (PY_MB(self), proxyAddr, proxyPort);

  if(!retval)
    return py_mb_raise_exception("Could not set the proxy to \"%s\", port %i", proxyAddr, proxyPort);
    
  Py_INCREF(Py_None); 
  return Py_None;
}

static PyObject * 
py_mb_Authenticate (PyObject *self, PyObject *args) {
  char *username;
  char *passwd;
  int  retval;
  char error[256];
  
  if (!PyArg_ParseTuple(args, "ss", &username, &passwd))
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  retval = mb_Authenticate (PY_MB(self), username, passwd);
  Py_END_ALLOW_THREADS

  if(!retval)
  {
    mb_GetQueryError(PY_MB(self), error, 256);
    return py_mb_raise_exception("Authentication failed: %s", error);
  }
  Py_INCREF(Py_None); 
  return Py_None;
}

static PyObject * 
py_mb_SetDevice (PyObject *self, PyObject *args)
{
  char *device;
  int retval;
  
  if (!PyArg_ParseTuple(args, "s", &device))
    return NULL;

  retval = mb_SetDevice (PY_MB(self), device);

  if(!retval)
    return py_mb_raise_exception("Could not set the device to \"%s\"", device);

  Py_INCREF(Py_None); 
  return Py_None;
}

static PyObject * 
py_mb_SetDepth (PyObject *self, PyObject *args) {
  int depth;

  if (!PyArg_ParseTuple(args, "i", &depth))
    return NULL;

  mb_SetDepth(PY_MB(self), depth);

  Py_INCREF(Py_None); 
  return Py_None;
}

static PyObject * 
py_mb_SetMaxItems (PyObject *self, PyObject *args) {
  int maxItems;

  if (!PyArg_ParseTuple(args, "i", &maxItems))
    return NULL;

  mb_SetMaxItems(PY_MB(self), maxItems);

  Py_INCREF(Py_None); 
  return Py_None;
}

static PyObject * 
py_mb_Query (PyObject *self, PyObject *args)
{
  char * query;
  int retval;
  char error[256];
  
  if (!PyArg_ParseTuple(args, "s", &query))
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  retval = mb_Query(PY_MB(self), query);
  Py_END_ALLOW_THREADS
  
  if(!retval)
  {
      mb_GetQueryError(PY_MB(self), error, 256);
      return py_mb_raise_exception("Query failed: %s", error);
  }

  Py_INCREF(Py_None); 
  return Py_None;
}

static PyObject * 
py_mb_QueryWithArgs (PyObject *self, PyObject *args) 
{
  char * query;
  char * argptr;
  PyObject * arglist;
  PyObject * argstr;
  PyObject * myargstr;
  int retval;
  int ii;
  int deref;
  char * qargs[20];
  char error[256];
  
  if (!PyArg_ParseTuple(args, "sO!", &query, &PyList_Type, &arglist))
    return NULL;

  
  for(ii = 0; ii < PyList_Size(arglist); ii++) 
  {
    deref = 0;
    argstr = PyList_GetItem(arglist, ii);
    if (PyString_Check(argstr))
    {
        myargstr = argstr;
    }
    else if (PyUnicode_Check(argstr)) 
    {
        myargstr = PyUnicode_AsUTF8String(argstr);
        deref = 1;
    }
    else if (PyInt_Check(argstr) || PyLong_Check(argstr)) 
    {
        PyObject *tuple = PyTuple_New(1);
        PyTuple_SetItem(tuple, 0, argstr);
        Py_INCREF(argstr);
        myargstr = PyString_Format(PyString_FromString("%d"), tuple);
        Py_DECREF(tuple);
        deref = 1;
    }
    else
    {
        PyErr_SetString(PyExc_ValueError, "Args must all be strings, unicode strings, ints or longs.");
        return NULL;
    }

    argptr = malloc(PyString_Size(myargstr)+1);
    if (argptr == NULL)
    {
        PyErr_NoMemory();
        return NULL;
    }

    qargs[ii] = strncpy(argptr, PyString_AsString(myargstr), PyString_Size(myargstr)+1);

    if (deref)
    {
        Py_DECREF(myargstr);
    }
  }
  qargs[ii] = NULL;

  Py_BEGIN_ALLOW_THREADS
  retval = mb_QueryWithArgs(PY_MB(self), query, qargs);
  Py_END_ALLOW_THREADS
  
  if(!retval)
  {
    mb_GetQueryError(PY_MB(self), error, 256);
    return py_mb_raise_exception("Query failed: %s", error);
  }

  for(ii = 0; ii < PyList_Size(arglist); ii++)
  {
    free(qargs[ii]);
  }

  Py_INCREF(Py_None); 
  return Py_None;
}


static PyObject * 
py_mb_GetWebSubmitURL (PyObject *self, PyObject *args)
{
  int retval;
  int URL_SIZE = 1024;
  char url[URL_SIZE];
  
  if (!PyArg_ParseTuple(args, ""))
    return NULL;

  retval = mb_GetWebSubmitURL (PY_MB(self), url, URL_SIZE);

  if(retval) 
    return PyString_FromString(url);
  else 
    return py_mb_raise_exception("GetWebSubmitURL failed");
}

static PyObject * 
py_mb_GetQueryError (PyObject *self, PyObject *args)
{
  int ERR_SIZE = 1024;
  char error[ERR_SIZE];
  
  if (!PyArg_ParseTuple(args, ""))
    return NULL;

  mb_GetQueryError (PY_MB(self), error, ERR_SIZE);
  
  return PyString_FromString(error);
}


static PyObject * 
py_mb_Select (PyObject *self, PyObject *args)
{
  char * query;
  int retval;
  
  if (!PyArg_ParseTuple(args, "s", &query))
    return NULL;

  retval = mb_Select (PY_MB(self), query);

  return PyInt_FromLong((long)retval);
}

static PyObject * 
py_mb_Select1 (PyObject *self, PyObject *args)
{
  char * query;
  int ord;
  int retval;
  
  if (!PyArg_ParseTuple(args, "si", &query, &ord))
    return NULL;

  retval = mb_Select1 (PY_MB(self), query, ord);

  return PyInt_FromLong((long)retval);
}

static PyObject * 
py_mb_SelectWithArgs (PyObject *self, PyObject *args)
{
  char * query;
  PyObject * arglist;
  PyObject * argstr;
  int retval;
  int ii;
  int qargs[20];
  
  if (!PyArg_ParseTuple(args, "sO!", &query, &PyList_Type, &arglist))
    return NULL;
  
  for(ii = 0; ii < PyList_Size(arglist); ii++) 
  {
    argstr = PyList_GetItem(arglist, ii);
    if (!PyInt_Check(argstr)) 
    {  
        PyErr_SetString(PyExc_ValueError, "Args must all be ints");
        return NULL;
    }
    qargs[ii] = (int)PyInt_AsLong(argstr);
  }
  qargs[ii] = 0;
  retval = mb_SelectWithArgs (PY_MB(self), query, qargs);
  
  return PyInt_FromLong((long)retval);
}

static PyObject * 
py_mb_GetResultData (PyObject *self, PyObject *args)
{
  int retval;
  char * query;  
  int dataLen = 1024;
  char data[dataLen];
  
  if (!PyArg_ParseTuple(args, "s", &query))
    return NULL;

  retval = mb_GetResultData (PY_MB(self), query, data, dataLen);

  if(!retval)
    return py_mb_raise_exception("Error in GetResultData");

  return PyUnicode_DecodeUTF8(data, strlen(data), "strict");
}

static PyObject * 
py_mb_GetResultData1 (PyObject *self, PyObject *args)
{
  int retval;
  char * query;  
  int dataLen = 1024;
  char data[dataLen];
  int ordinal;
  
  if (!PyArg_ParseTuple(args, "si", &query, &ordinal))
    return NULL;

  retval = mb_GetResultData1 (PY_MB(self), query, data, dataLen, ordinal);

  if(!retval)
    return py_mb_raise_exception("Error in GetResultData1");

  return PyUnicode_DecodeUTF8(data, strlen(data), "strict");    
}

static PyObject * 
py_mb_DoesResultExist (PyObject *self, PyObject *args)
{
  int retval;
  char * query;  
  
  if (!PyArg_ParseTuple(args, "s", &query))
    return NULL;

  retval = mb_DoesResultExist (PY_MB(self), query);

  return PyInt_FromLong((long)retval);
}

static PyObject * 
py_mb_DoesResultExist1 (PyObject *self, PyObject *args)
{
  int retval;
  char * query;  
  int ordinal;
  
  if (!PyArg_ParseTuple(args, "si", &query, &ordinal))
    return NULL;

  retval = mb_DoesResultExist1 (PY_MB(self), query, ordinal);

  return PyInt_FromLong((long)retval);
}

static PyObject * 
py_mb_GetResultInt (PyObject *self, PyObject *args)
{
  int retval;
  char * query;  
  
  if (!PyArg_ParseTuple(args, "s", &query))
    return NULL;

  retval = mb_GetResultInt (PY_MB(self), query);

  return PyInt_FromLong((long)retval);
}

static PyObject * 
py_mb_GetResultInt1 (PyObject *self, PyObject *args)
{
  int retval;
  char * query;  
  int ordinal;
  
  if (!PyArg_ParseTuple(args, "si", &query, &ordinal))
    return NULL;

  retval = mb_GetResultInt1 (PY_MB(self), query, ordinal);

  return PyInt_FromLong((long)retval);
}

static PyObject * 
py_mb_GetResultRDF (PyObject *self, PyObject *args)
{
  char * rdf;
  int retval;
  
  if (!PyArg_ParseTuple(args, ""))
    return NULL;

  rdf = malloc(mb_GetResultRDFLen(PY_MB(self)));
  retval = mb_GetResultRDF(PY_MB(self), rdf, mb_GetResultRDFLen(PY_MB(self)));

  if(!retval)
    return py_mb_raise_exception("Couldn't return RDF");

  return PyUnicode_DecodeUTF8(rdf, mb_GetResultRDFLen(PY_MB(self)), "strict");    
}

static PyObject * 
py_mb_SetResultRDF (PyObject *self, PyObject *args)
{
  char * rdf;
  int retval;
  
  if (!PyArg_ParseTuple(args, "s", &rdf))
    return NULL;

  retval = mb_SetResultRDF(PY_MB(self), rdf);

  if(!retval)
    return py_mb_raise_exception("Couldn't set RDF");

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject * 
py_mb_GetIDFromURL (PyObject *self, PyObject *args)
{
  char * url;
  int idLen = 64;
  char id[idLen];

  if (!PyArg_ParseTuple(args, "s", &url))
    return NULL;
  
  mb_GetIDFromURL (PY_MB(self), url, id, idLen);
  
  return PyString_FromString(id);
}

static PyObject * 
py_mb_GetFragmentFromURL (PyObject *self, PyObject *args)
{
  char * url;
  int fragmentLen = 64;
  char fragment[fragmentLen];

  if (!PyArg_ParseTuple(args, "s", &url))
    return NULL;
  
  mb_GetFragmentFromURL (PY_MB(self), url, fragment, fragmentLen);
  
  return PyString_FromString(fragment);
}

static PyObject * 
py_mb_GetOrdinalFromList (PyObject *self, PyObject *args)
{
  char * resultList;
  char * URI;
  int retval;

  if (!PyArg_ParseTuple(args, "ss", &resultList, &URI))
    return NULL;
  
  retval = mb_GetOrdinalFromList (PY_MB(self), resultList, URI);
  
  return PyInt_FromLong((long)retval);
}

static PyObject * 
py_mb_CalculateSha1 (PyObject *self, PyObject *args)
{
  char * fileName;
  char sha1[41];
  int retval;
  
  if (!PyArg_ParseTuple(args, "s", &fileName))
    return NULL;
      
  Py_BEGIN_ALLOW_THREADS
  retval = mb_CalculateSha1(PY_MB(self), fileName, sha1);
  Py_END_ALLOW_THREADS
  
  if(!retval)
    return py_mb_raise_exception("Couldn't calculate sha1");

  return PyString_FromString(sha1);
}

static PyObject * 
py_mb_CalculateBitprint (PyObject *self, PyObject *args)
{
  char * fileName;
  int retval;
  BitprintInfo * info;
  PyObject * dict;

  info = malloc(sizeof(BitprintInfo));

  if (!PyArg_ParseTuple(args, "s", &fileName))
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  retval = mb_CalculateBitprint (PY_MB(self), fileName, info);
  Py_END_ALLOW_THREADS
  if(retval)
  {
    dict = PyDict_New();
    if(PyDict_SetItem(dict, PyString_FromString("filename"), PyString_FromString(info->filename)))
      return NULL;
    if(PyDict_SetItem(dict, PyString_FromString("bitprint"), PyString_FromString(info->bitprint)))
      return NULL;
    if(PyDict_SetItem(dict, PyString_FromString("first20"), PyString_FromString(info->first20)))
      return NULL;
    if(PyDict_SetItem(dict, PyString_FromString("audioSha1"), PyString_FromString(info->audioSha1)))
      return NULL;
    if(PyDict_SetItem(dict, PyString_FromString("length"), PyInt_FromLong((long)info->length)))
      return NULL;
    if(PyDict_SetItem(dict, PyString_FromString("duration"), PyInt_FromLong((long)info->duration)))
      return NULL;
    if(PyDict_SetItem(dict, PyString_FromString("samplerate"), PyInt_FromLong((long)info->samplerate)))
      return NULL;
    if(PyDict_SetItem(dict, PyString_FromString("bitrate"), PyInt_FromLong((long)info->bitrate)))
      return NULL;
    if(PyDict_SetItem(dict, PyString_FromString("stereo"), PyInt_FromLong((long)info->stereo)))
      return NULL;
    if(PyDict_SetItem(dict, PyString_FromString("vbr"), PyInt_FromLong((long)info->vbr)))
      return NULL;
    return dict;
  }
  else
    return py_mb_raise_exception("Couldn't calculate bitprint");
}

static PyObject * 
py_mb_GetMP3Info (PyObject *self, PyObject *args)
{
  char * fileName;
  int retval;
  PyObject * dict;
  int duration;
  int bitrate;
  int stereo;
  int samplerate;
  
  if (!PyArg_ParseTuple(args, "s", &fileName))
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  retval = mb_GetMP3Info (PY_MB(self), fileName, &duration, &bitrate, &stereo, &samplerate);
  Py_END_ALLOW_THREADS
  
  if(retval)
  {
    dict = PyDict_New();
    if(PyDict_SetItem(dict, PyString_FromString("duration"), PyInt_FromLong((long)duration)))
      return NULL;
    if(PyDict_SetItem(dict, PyString_FromString("samplerate"), PyInt_FromLong((long)samplerate)))
      return NULL;
    if(PyDict_SetItem(dict, PyString_FromString("bitrate"), PyInt_FromLong((long)bitrate)))
      return NULL;
    if(PyDict_SetItem(dict, PyString_FromString("stereo"), PyInt_FromLong((long)stereo)))
      return NULL;
    return dict;
  }
  else
    return py_mb_raise_exception("Couldn't examine mp3 file");
}

