/**
 * Copyright (C) MX4J.
 * All rights reserved.
 *
 * This software is distributed under the terms of the MX4J License version 1.0.
 * See the terms of the MX4J License in the documentation provided with this software.
 */

package javax.management.openmbean;

import java.io.Serializable;
import java.util.Iterator;
import java.util.Set;
import java.util.Collections;
import java.util.HashSet;

import javax.management.MBeanParameterInfo;

/**
 * @author <a href="mailto:shadow12@users.sourceforge.net">Bronwen Cassidy</a>
 * @version $Revision: 1.6 $
 */

/**
 * Describes a parameter used in one or more operations or constructors of an open MBean
 */
public class OpenMBeanParameterInfoSupport extends MBeanParameterInfo implements OpenMBeanParameterInfo, Serializable
{
	private static final long serialVersionUID = -7235016873758443122L;

	private OpenType openType = null;
	private Object defaultValue = null;
	private Set legalValues = null;
	private Comparable minValue = null;
	private Comparable maxValue = null;

	private transient int m_hashcode = 0;

	/**
	 * No validation done for null values is done here.
	 * Constructs an OpenMBeanParameterInfoSupport instance, which describes the parameter used in one or more operations or constructors of a class of open MBeans,
	 * with the specified name, openType and description.
	 *
	 * @param name - cannot be a null or empty string
	 * @param description - cannot be a null or empty string.
	 * @param openType - cannot be null
	 *
	 * @throws IllegalArgumentException - if name or description are null or empty string, or openType is null.
	 */
	public OpenMBeanParameterInfoSupport(String name, String description, OpenType openType)
	{
		super(name, openType == null ? "" : openType.getClassName(), description);
        if(name == null || name.trim().equals("")) throw new IllegalArgumentException("name parameter cannot be null or an empty string.");
		if(description == null || description.trim().equals("")) throw new IllegalArgumentException("description parameter cannot be null or an empty string.");
		if(openType == null) throw new IllegalArgumentException("OpenType parameter cannot be null.");
		this.openType = openType;
	}

	/**
	 * Constructs an OpenMBeanParameterInfoSupport instance, which describes the parameter used in one or more operations or constructors of a class of open MBeans, with the specified name,
	 * openType, description and defaultValue.
	 *
	 * @param name - cannot be a null or empty string.
	 * @param description - cannot be a null or empty string.
	 * @param openType - cannot be null.
	 * @param defaultValue - must be a valid value for the openType specified for this parameter;
	 * 						default value not supported for ArrayType and TabularType;
	 * 						can be null, in which case it means that no default value is set.
	 *
	 * @throws IllegalArgumentException - if name or description are null or empty string, or openType is null
	 * @throws OpenDataException - if defaultValue is not a valid value for the specified openType, or defaultValue is non null and openType is an ArrayType or a TabularType.
	 */
	public OpenMBeanParameterInfoSupport(String name, String description, OpenType openType, Object defaultValue) throws OpenDataException
	{
		this(name, description, openType);
		if(defaultValue != null)
		{
			if(openType.isArray() ||  openType instanceof TabularType) throw new OpenDataException("openType should not be an ArrayType or a TabularType when a default value is required.");
			if(!(openType.isValue(defaultValue))) throw new OpenDataException("defaultValue class name " + defaultValue.getClass().getName() + " does not match the one defined in openType.");
			this.defaultValue = defaultValue;
		}
	}

	/**
	 * Constructs an OpenMBeanParameterInfoSupport instance, which describes the parameter used in one or more operations or constructors of a class of open MBeans, with the specified name,
	 * openType, description, defaultValue and legalValues. The contents of legalValues are internally dumped into an unmodifiable Set, so subsequent modifications of the array referenced
	 * by legalValues have no impact on this OpenMBeanParameterInfoSupport instance
	 *
	 * @param name - cannot be a null or empty string.
	 * @param description - cannot be a null or empty string.
	 * @param openType - cannot be null.
	 * @param defaultValue - must be a valid value for the openType specified for this parameter;
	 * 						default value not supported for ArrayType and TabularType;
	 * 						can be null, in which case it means that no default value is set.
	 * @param legalValues - each contained value must be valid for the openType specified for this parameter; legal values not supported for ArrayType and TabularType; can be null or empty
	 *
	 * @throws IllegalArgumentException  - if name or description are null or empty string, or openType is null.
	 * @throws OpenDataException - if defaultValue is not a valid value for the specified openType, or one value in legalValues is not valid for the specified openType,
	 * 							    or defaultValue is non null and openType is an ArrayType or a TabularType,
	 * 								or legalValues is non null and non empty and openType is an ArrayType or a TabularType,
	 * 								or legalValues is non null and non empty and defaultValue is not contained in legalValues.
	 */
	public OpenMBeanParameterInfoSupport(String name, String description, OpenType openType, Object defaultValue, Object[] legalValues) throws OpenDataException
	{
		this(name, description, openType, defaultValue);
		if(legalValues != null && legalValues.length > 0)
		{
	        if(openType.isArray() || openType instanceof TabularType) throw new OpenDataException("legalValues not supported if openType is an ArrayType or an instanceof TabularType");
            for(int i = 0; i < legalValues.length; i++)
			{
                if(!(openType.isValue(legalValues[i]))) throw new OpenDataException("The element at index " + i + " of type " + legalValues[i] + " is not an value specified in openType.");
			}
			// all checked assign Object[] legalValues to set legalValues
            assignLegalValues(legalValues);
			if(hasDefaultValue() && hasLegalValues() && !(this.legalValues.contains(defaultValue))) throw new OpenDataException("LegalValues must contain the defaultValue");
		}
	}

	/**
	 * Constructs an OpenMBeanParameterInfoSupport instance, which describes the parameter used in one or more operations or constructors of a class of open MBeans,
	 * with the specified name, openType, description, defaultValue, minValue and maxValue. It is possible to specify minimal and maximal values only for an open type whose values are Comparable
	 *
	 * @param name - cannot be a null or empty string.
	 * @param description - cannot be a null or empty string
	 * @param openType - cannot be null.
	 * @param defaultValue - must be a valid value for the openType specified for this parameter;
	 * 						default value not supported for ArrayType and TabularType;
	 * 						can be null, in which case it means that no default value is set.
	 * @param minValue - must be valid for the openType specified for this parameter; can be null, in which case it means that no minimal value is set.
	 * @param maxValue - must be valid for the openType specified for this parameter; can be null, in which case it means that no maximal value is set.
	 *
	 * @throws IllegalArgumentException - if name or description are null or empty string, or openType is null.
	 * @throws OpenDataException - if defaultValue, minValue or maxValue is not a valid value for the specified openType,
	 * 								or defaultValue is non null and openType is an ArrayType or a TabularType,
	 * 								or both minValue and maxValue are non-null and minValue.compareTo(maxValue) > 0 is true,
	 * 								or both defaultValue and minValue are non-null and minValue.compareTo(defaultValue) > 0 is true,
	 * 								or both defaultValue and maxValue are non-null and defaultValue.compareTo(maxValue) > 0 is true.
	 */
	public OpenMBeanParameterInfoSupport(String name, String description, OpenType openType, Object defaultValue, Comparable minValue, Comparable maxValue) throws OpenDataException
	{
		this(name, description, openType, defaultValue);
		if(minValue != null)
		{
			/** test is a valid value for the specified openType */
			if(!(openType.isValue(minValue))) throw new OpenDataException("Comparable value of " + minValue.getClass().getName() + " does not match the openType value of " + openType.getClassName());
			this.minValue = minValue;
		}
		if(maxValue != null)
		{
			if(!(openType.isValue(maxValue))) throw new OpenDataException("Comparable value of " + maxValue.getClass().getName() + " does not match the openType value of " + openType.getClassName());
			this.maxValue = maxValue;
		}
		if(hasMinValue() && hasMaxValue() && minValue.compareTo(maxValue) > 0) throw new OpenDataException("minValue cannot be greater than maxValue.");
		if(hasDefaultValue() && hasMinValue() && minValue.compareTo((Comparable)defaultValue) > 0) throw new OpenDataException("minValue cannot be greater than defaultValue.");
		if(hasDefaultValue() && hasMaxValue() && ((Comparable)defaultValue).compareTo(maxValue) > 0) throw new OpenDataException("defaultValue cannot be greater than maxValue.");
	}

	/**
	 * assigns the validated Object[] into the set legal values as the constructor states this is unmodifiable will create an unmodifiable set
	 */
	private void assignLegalValues(Object[] legalValues)
	{
		HashSet modifiableSet = new HashSet();
		for(int i = 0; i < legalValues.length; i++)
		{
	        modifiableSet.add(legalValues[i]);
		}
		this.legalValues = Collections.unmodifiableSet(modifiableSet);
	}

	/**
	 * @return the open type for the values of the parameter described by this OpenMBeanParameterInfoSupport instance.
	 */
	public OpenType getOpenType()
	{
		return openType;
	}

	/**
	 * @return the default value for the parameter described by this OpenMBeanParameterInfoSupport instance, if specified, or null otherwise.
	 */
	public Object getDefaultValue()
	{
		return defaultValue;
	}

	/**
	 * @return an unmodifiable Set of legal values for the parameter described by this OpenMBeanParameterInfoSupport instance, if specified, or null otherwise
	 */
	public Set getLegalValues()
	{
		return legalValues;
	}

	/**
	 * @return the minimal value for the parameter described by this OpenMBeanParameterInfoSupport instance, if specified, or null otherwise.
	 */
	public Comparable getMinValue()
	{
		return minValue;
	}

	/**
	 * @return the maximal value for the parameter described by this OpenMBeanParameterInfoSupport instance, if specified, or null otherwise.
	 */
	public Comparable getMaxValue()
	{
		return maxValue;
	}

	/**
	 * @return true if defaultValue is specified (i.e not null) false otherwise
	 */
	public boolean hasDefaultValue()
	{
		return defaultValue != null;
	}

	/**
	 * @return true if legalValues is specified, false if legalValues is null
	 */
	public boolean hasLegalValues()
	{
		return legalValues != null;
	}

	/**
	 * @return true if minValue is specified, false if minValue is null
	 */
	public boolean hasMinValue()
	{
		return minValue != null;
	}

	/**
	 * @return true if maxValue is specified, false if maxValue is null
	 */
	public boolean hasMaxValue()
	{
		return maxValue != null;
	}

	/**
	 * Tests wether obj is a valid value for the parameter described by this OpenMBeanParameterInfo instance
	 *
	 * @param obj - the Object to test if is a valid value
	 * @return true if obj is a valid value false otherwise.
	 * 			A valid value is determined by
	 * 			<ul>
	 * 				<li>if <tt>openType.isValue(obj)</tt> returns true</li>
	 * 				<li>if legalValues is present and <tt>legalValues.contains(obj)</tt> returns true</li>
	 * 				<li>if minValue and maxValue compare to obj with minValue being less than obj and maxValue being greater than obj</li>
	 * 			</ul>
	 */
	public boolean isValue(Object obj)
	{
		// if any object is null and they have a defaultValue then isValue is true for anything else null obj returns false (must be first test) as the rest will return false for the null object
		if(hasDefaultValue() && obj == null) return true;

		if(!(openType.isValue(obj))) return false;
        if(hasLegalValues() && (!(legalValues.contains(obj)))) return false;
		if(hasMinValue() && minValue.compareTo(obj) > 0) return false;
        if(hasMaxValue() && maxValue.compareTo(obj) < 0) return false;
		return true;
	}

	/**
	 * Compares the specified obj parameter with this OpenMBeanParameterInfoSupport instance for equality.
	 * @return true if and only if all of the following statements are true:
	 * 		<ul>
	 * 			<li>obj is non null</li>
	 * 			<li>obj also implements the OpenMBeanParameterInfo interface</li>
	 * 			<li>their names are equal</li>
	 * 			<li>their open types are equal</li>
	 * 			<li>their default, min, max and legal values are equal and if present in this instance msut be present in obj</li>
	 * 		</ul>
	 */
	public boolean equals(Object obj)
	{
		if(!(obj instanceof OpenMBeanParameterInfo))
			return false;

		OpenMBeanParameterInfo paramObj = (OpenMBeanParameterInfo)obj;

		if(!getName().equals(paramObj.getName()) || !getOpenType().equals(paramObj.getOpenType()))
			return false;

		if(hasDefaultValue() && (!getDefaultValue().equals(paramObj.getDefaultValue())))
			return false;
		if(!hasDefaultValue() && paramObj.hasDefaultValue())
			return false;

		if(hasMinValue() && !(getMinValue().equals(paramObj.getMinValue())))
			return false;
		if(!hasMinValue() && paramObj.hasMinValue())
			return false;

		if(hasMaxValue() && !(getMaxValue().equals(paramObj.getMaxValue())))
			return false;
		if(!hasMaxValue() && paramObj.hasMaxValue())
			return false;

		if(hasLegalValues() && !(getLegalValues().equals(paramObj.getLegalValues())))
			return false;
		if(!hasLegalValues() && paramObj.hasLegalValues())
			return false;

		return true;
	}

	public int hashCode()
	{
		if(m_hashcode == 0)
		{
			m_hashcode = hashCode(this);
		}
		return m_hashcode;
	}

	public String toString()
	{
		StringBuffer buf = new StringBuffer(getClass().getName());
		buf.append("\t(name = ");
		buf.append(getName());
		buf.append("\topenType = ");
		buf.append(openType.toString());
		buf.append("\tdefault value = ");
		buf.append(String.valueOf(defaultValue));
		buf.append("\tmin value = ");
		buf.append(String.valueOf(minValue));
		buf.append("\tmax value = ");
		buf.append(String.valueOf(maxValue));
		buf.append("\tlegal values = ");
		buf.append(String.valueOf(legalValues));
		buf.append(")");
		return buf.toString();
	}

	private int hashCode(OpenMBeanParameterInfo info) 
	{
		int result = info.getName().hashCode();
		result += info.getOpenType().hashCode();
		result += (info.hasDefaultValue() == false) ? 0 : info.getDefaultValue().hashCode();
		result += (info.hasLegalValues() == false) ? 0 : hashCode(info.getLegalValues());
		result += (info.hasMinValue() == false) ? 0 : info.getMinValue().hashCode();
		result += (info.hasMaxValue() == false) ? 0 : info.getMaxValue().hashCode();
		return result;
	}

	private int hashCode(Set legalvalues)
	{
		int result = 0;
		Iterator i = legalvalues.iterator();
		while (i.hasNext())
		{
			Object v = i.next();
			result += v.hashCode();
		}
		return result;
	}
}
