/***************************************************************************
                          hbcistring.h  -  description
                             -------------------
    begin                : Sun Jul 1 2001
    copyright            : (C) 2001 by fabian kaiser
    email                : fabian.kaiser@gmx.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/

/** @file hbcistring.h
 *
 * @short Helper class HBCI::String with only static methods. No C
 * wrappers.  */


#ifndef HBCISTRING_H
#define HBCISTRING_H

#include <stdio.h>
#include <string>
/*#include <fstream> */
#include <openhbci/dllimport.h>

using namespace std;

namespace HBCI {

/**
 * This class offers some (only static) methods to operate on strings.
 * @short Some static methods to manipulate strings
 *@author fabian kaiser
 */
class DLLIMPORT String : public string  {
 private:
    String();
    ~String();
 public:

    /**
     * return the next DE (DatenElement, see HBCI-Specification)
     * from a given startIndex on
     */
    DLLIMPORT static string nextDE(const string &data, int startIndex=0);

    /**
     * return the next DE (DatenElementGruppe, see HBCI-Specification)
     * from a given startIndex on
     */
    DLLIMPORT static string nextDEG(const string &data, int startIndex=0);

    /**
     * return the next SEG (Segment, see HBCI-Specification)
     * from a given startIndex on
     */
    DLLIMPORT static string nextSEG(const string &data, int startIndex=0);

    /**
     * converts a number to a string
     * fills new string with <0> from the left side if <fillWithZero>==true and <length> > 0
     */
    DLLIMPORT static string num2string(int number, bool fillWithZero = false, int length = 0);

    /**
     * Converts a double to a string.
     * @param number Double number to convert.
     * @param pre number of digits in front of comma
     * @param post number of digits behind comma
     * @author Martin Preuss <martin@libchipcard.de>
     */
    /*DLLIMPORT static string double2string(double number,
      int pre=8, int post=2);*/

    /**
     * Converts a string into a double, according to %HBCI specification.
     *
     * This method follows exactly the same behavior as the
     * string-valued constructor of HBCI::Value, so please see the
     * details there. 
     *
     * For example, the HBCI-format "4000," is converted to the double
     * value "4000.0".
     *
     * In case of format errors, a HBCI::Error is thrown.
     */
    DLLIMPORT static double string2double(const string &data);

    /**
     * int the hbci-protocol, some characters have special meanings (eg +, ?, : ,').<br>
     * thus you can use them by escaping each occurence with a '?'
     */
    DLLIMPORT static string escape(const string &unescaped);

    /** @see escape */
    DLLIMPORT static string unEscape(const string &escaped);

    /**
     * returns a string-represenation of the current date (YYYYMMDD)
     */
    DLLIMPORT static string date2string();

    /**
     * returns a string-represenation of the current time (HHMMSS)
     */
    DLLIMPORT static string time2string();

    /**
     * write data to an open fstream
     */
    /*DLLIMPORT static bool toStream(ostream *f, const string &data);*/

    /**
     * read data from an open fstream
     */
    /*DLLIMPORT static string fromStream(istream *f);*/

    /**
     * reads the content of a whole file
     */
    /*DLLIMPORT static string readWholeFile(const string &filename);*/

    /**
     * transform given text to the dtaus0-format
     */
    DLLIMPORT static string transformToDTAUS0(const string &msg);

    /**
     * wrap data with @<length-of-data>@<data>
     */
    DLLIMPORT static string transformToHBCIBinData(const string &data);

    /**
     * return the data stored in a hbci-string eg <@23@lkasd083lkn23.,  ihpoh>
     * @throws HBCIMessage*
     */
    DLLIMPORT static string extractBinData(const string &data);

    /**
     * Returns the TLV starting at startIndex.
     * A TLV is a special string, whose first two bytes have a
     * special meaning:
     * Byte  0=Tag (whatever value you like)
     * Byte  1=Length of the data part
     * Bytes 2-x=Value
     * @author Martin Preuss<openhbci@aquamaniac.de>
     */
    DLLIMPORT static string nextTLV(const string &data, int startIndex=0);

    /**
     * Returns the type of the tag (tag number).
     * @author Martin Preuss<openhbci@aquamaniac.de>
     */
    DLLIMPORT static unsigned int typeTLV(const string &tlv);

    /**
     * Returns the size of the data in the TLV.
     * @author Martin Preuss<openhbci@aquamaniac.de>
     */
    DLLIMPORT static int sizeTLV(const string &tlv);

    /**
     * Returns the data of a tag.
     * @author Martin Preuss<openhbci@aquamaniac.de>
     */
    DLLIMPORT static string dataTLV(const string &tlv);

    /**
     * Creates a new TLV.
     * @author Martin Preuss<openhbci@aquamaniac.de>
     */
    DLLIMPORT static string newTLV(int tag, const string &data);

    /**
     * Creates a human-readable version of the HBCI string, i.e.
     * strips out binary data and replaces it with simple placeholder.
     * Used for debugging purposes.
     */
    DLLIMPORT static string dumpToString(const string &s);

    DLLIMPORT static const char *hbciSegmentName(const char *segment);

    /**
     * Simply dumps a string both hex and ASCII dump.
     * Does not interprete the data in any way.
     * @author Martin Preuss<openhbci@aquamaniac.de>
     * @param s string to dump
     * @param f FILE handle to dump this to.
     */
    DLLIMPORT static void simpleDump(const string &s, FILE *f=stderr);

private:

    DLLIMPORT static int lengthOfBinaryData(const string &data, int startIndex);

};

} /* namespace HBCI */

/**
 * Duplicates a const char* string into newly allocated
 * memory. Returns a pointer to the newly allocated char* array which
 * is owned by the caller and has to be freed by the caller after
 * usage.  
 * @author Christian <Stimming@tuhh.de>
 */
extern char* hbci_strdup(const char *c);
/** 
 * Duplicates a string into newly allocated memory. Returns a pointer
 * to the newly allocated char* array which is owned by the caller and
 * has to be freed by the caller after usage.  
 * @author Christian <Stimming@tuhh.de>
 */
extern char* hbci_strdup(const string &s);


#endif

