/***************************************************************************
 $RCSfile: swiftparser.cpp,v $
                             -------------------
    cvs         : $Id: swiftparser.cpp,v 1.28 2004/01/13 21:09:17 aquamaniac Exp $
    begin       : Sun Aug 05 2001
    copyright   : (C) 2001 by Martin Preuss
    email       : openhbci@aquamaniac.de


 ***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/


/*
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef __declspec
# if BUILDING_DLL
#  define DLLIMPORT __declspec (dllexport)
# else /* Not BUILDING_DLL */
#  define DLLIMPORT __declspec (dllimport)
# endif /* Not BUILDING_DLL */
#else
# define DLLIMPORT
#endif

#include <list>
#include <string>
#include <cctype>
#include <stdio.h>
#include <iostream>
#include <assert.h>

#include "hbcistring.h"
#include "swiftparser.h"


namespace HBCI {

transactionReport::transactionReport(const string& curr){
    _currency=curr;
}


transactionReport::~transactionReport(){
}


const list<Transaction>& transactionReport::transactions() const {
    return _transactions;
}


void transactionReport::addTransaction(const Transaction& t){
    _transactions.push_back(t);
}

void transactionReport::dump() const 
{
    for (list<Transaction>::const_iterator it = _transactions.begin();
	 it != _transactions.end();
	 it++)
	{
	    const Transaction& t = *it;
	    std::cerr << "Transaction: " 
		 << "Date: " << t.date().toString()
		 << "| ValutaDate: " << t.valutaDate().toString()
		 << "| Our AccountId: " << t.ourAccountId()
		 << "| Other AccountId: " << t.otherAccountId()
		 << "| Value: " << t.value().toReadableString()
		 << endl;
	}
}
    


string SWIFTparser::_normalize(const string &s) {
    unsigned int i;
    string result;

    i=0;
    while(i<s.length()) {
        if (s.substr(i,2)=="@@") {
            result+=13;
            result+=10;
            i++; // skip both
        }
        else
            result+=s[i];
        i++;
    } // while
    return result;
}


string SWIFTparser::nextTAG(string str, unsigned int pos){
    unsigned int lpos, spos;


    spos=pos;
    while(pos<str.length()) {
        /* The end of a tag is indicated by:
         * - sequence <CR> <LF> ':'
         *   that would mean we found the beginning of the next tag
         * - sequence <CR> <LF> <CR>
         *   because a S.W.I.F.T. document is terminated by a <CR> <LF> <->
         *   sequence the given sequence means that the end of the document
         *   is reached which implies that the tag itself ends, too ;-)
         */
        if (str.at(pos)==13) {
            /* check for sequences discussed above */
            lpos=pos;
            if (++lpos<str.length())
                if (str.at(lpos)==10)
                    if (++lpos<str.length())
                        if ((str.at(lpos)=='-') || (str.at(lpos)==':'))
                            return str.substr(spos,lpos-spos);
        } // if
        pos++;
    } // while
    /* we did not find any of the sequences discussed above, so this could not
     * be a valid swift document.
     * well, that would normally be handled this way ;-)
     * but since some institutes do not comply to SWIFT we have to
     * lighten up a bit. So if we did not find any of the sequences
     * discussed above we simply return the rest of the string given.
     */
    return str.substr(spos);
}


string SWIFTparser::nextFIELD(string str, unsigned int pos){
    unsigned int startpos;
    unsigned int endpos;

    if (str.at(pos)!='?')
        return "";

    startpos=pos;

    // skip possible CRLF after "?"
    pos++;
    while(pos<str.length()) {
	if ((unsigned char)(str.at(pos))>31)
	    break;
	pos++;
    }
    if (pos>=str.length())
        return "";

    // now find end
    endpos=pos;
    while(endpos<str.length()) {
        //if ((str.at(endpos)==13) || (str.at(endpos)=='?'))
        if (str.at(endpos)=='?')
            return str.substr(startpos,endpos-startpos);
        endpos++;
    } // while
    /* we found neither a '?' nor a <CR> so this can not
     * be a valid swift document. */
    return str.substr(startpos);
}


string SWIFTparser::tagId(string tag){
    unsigned int pos;
    
    /* tag must start with a colon, return if it doesnt */
    if (tag.at(0)!=':')
        return "";

    pos=1; // skip first colon
    /* now search for the second colon */
    while(pos<tag.length()) {
        if (tag.at(pos)==':')
            /* create new string containing only the id */
            return tag.substr(1,pos-1);
        pos++;
    }
    /* didnt find second colon because of a bad tag */
    return "";
}


string SWIFTparser::tagContent(string tag){
    unsigned int pos;
    
    /* tag must start with a colon, return if it doesnt */
    if (tag.at(0)!=':')
        return "";

    pos=1; // skip first colon
    /* now search for the second colon */
    while (pos<tag.length()) {
        if (tag.at(pos)==':') {
            // we have it ;-) content begins directly after this
            pos++;
            if ((pos+2)<tag.length())
                return tag.substr(pos,tag.length()-pos-2);
            else
                return "";
        }
        pos++;
    }
    return "";
}


int SWIFTparser::fieldId(string field, unsigned int &pos){
    string tmp;

    /* tag must start with a question mark, return if it doesnt */
    if (field.at(pos)!='?')
        return -1;
    pos++;

    /* ------------------------------------ get first character after "?" */
    /* skip possible CRLF */
    while(pos<field.length()) {
	if ((unsigned char)(field.at(pos))>31)
	    break;
	pos++;
    }
    if (pos>=field.length())
	return -1;
    tmp=field.at(pos);
    pos++;

    /* ----------------------------------- get second character after "?" */
    /* skip possible CRLF */
    while(pos<field.length()) {
	if ((unsigned char)(field.at(pos))>31)
	    break;
	pos++;
    }
    if (pos>=field.length())
	return -1;
    /* get next char */
    tmp+=field.at(pos);
    pos++;
    /* convert them into an integer */
    return atoi(tmp.c_str());
}


int SWIFTparser::fieldId(string tag){
    unsigned int pos;

    pos=0;
    return fieldId(tag,pos);
}


string SWIFTparser::fieldContent(string field){
    unsigned int lpos;
    string result;

    lpos=0;
    // seek position of data begin
    if (-1==fieldId(field,lpos))
        return "";
    field=field.substr(lpos);
    lpos=0;

    // search for a <CR> <LF> sequence, because we want to exclude them
    while(lpos<field.length()) {
        if (field.at(lpos)=='?')
            break;
        // skip CR/LF in fields
	if ((unsigned char)(field.at(lpos))>31)
	    result+=field.at(lpos);
	lpos++;
    }
    return result;
}


bool SWIFTparser::_mt940_61(string tc, Transaction &tda, string curr){
    string tmp;
    unsigned int pos;
    unsigned int lpos;
    double d;
    Date td;
    bool debit;
    
    pos=0;

    // get valuta date
    if (pos+6>=tc.length()) {
        // bad string
#if DEBUGMODE>0
        fprintf(stderr,"Err1\n");
#endif
        return false;
    }
    tmp=tc.substr(pos,6);
    tda.setValutaDate(Date(tmp,2));
    if (!tda.valutaDate().isValid()){
#if DEBUGMODE>0
        fprintf(stderr,"Err2\n");
#endif
        return false;
    }
    pos+=6;

    // check for transaction date
    if (pos>=tc.length())
        // bad string
        return false;
    if (tc.at(pos)<='9' && tc.at(pos)>='0') {
        // get transaction date
        if (pos+4>=tc.length())
            // bad string
            return false;
        tmp=tc.substr(pos,4);
        td=Date(tmp,0); // year not given !
        tda.setDate(Date(td.day(),
			 td.month(),
			 tda.valutaDate().year())); // take year from valuta date
	if (!tda.date().isValid()) {
#if DEBUGMODE>0
            fprintf(stderr,"Err3\n");
#endif
            return false;
        }
        pos+=4;
    }

    // get credit/debit mark
    if (pos+2>=tc.length()) {
        // bad string
#if DEBUGMODE>0
        fprintf(stderr,"Err4\n");
#endif
        return false;
    }
    if (tc.at(pos)=='R' || tc.at(pos+1)=='R') {
      tmp=tc.substr(pos,2);
    }
    else
      tmp=tc.substr(pos,1);
    debit=((tmp=="D") || (tmp=="RD") || (tmp=="DR"));
    pos+=tmp.length();

    // check if there is  anything but a number
    if (pos>=tc.length()) {
        // bad string
#if DEBUGMODE>0
        fprintf(stderr,"Err5\n");
#endif
        return false;
    }
    if (tc.at(pos)<'0' || tc.at(pos)>'9')
        // skip whatever that character means ;-)
        pos++;

    // read value
    // find 'N'
    lpos=pos;
    while(pos<tc.length()) {
        if (tc.at(pos)=='N')
            break;
        pos++;
    }
    if (pos>=tc.length()) {
        // did not find the f*cking "N"
#if DEBUGMODE>0
        fprintf(stderr,"Err6\n");
#endif
        return false;
    }
    tmp=tc.substr(lpos,pos-lpos);
    d=String::string2double(tmp);
    if (debit)
        d=-d;
    tda.setValue(Value(d,curr));
    pos++;

    // read the transaction key (3 chars)
    if (pos+2>=tc.length()) {
        // bad string
#if DEBUGMODE>0
        fprintf(stderr,"Err7\n");
#endif        
	return false;
    }
    tda.setTransactionKey(tc.substr(pos,3));
    pos+=3;

    // find "//" (if there is a bank reference) or <CR> (if there is more information)
    lpos=pos;
    while(pos<tc.length()) {
	if (tc.at(pos)=='/') {
	  // does another "/" follow ?
	  if (pos+1<tc.length())
	    if (tc.at(pos+1)=='/')
	      // there is a "/", so this is the end of the customer reference
	      break;
	}
	if (tc.at(pos) == 13)
	  break;
	pos++;
    } // while
    if (pos>lpos)
      tda.setCustomerReference(tc.substr(lpos,pos-lpos));

    if (pos>=tc.length())
        // nothing follows, so we have finished
        return true;

    if (pos+1>=tc.length())
        // nothing more follows after that but it should
        return false;

    if ((tc.at(pos)=='/') &&
	(tc.at(pos+1) == '/')) { 
	// '//' means bank reference existing; OTOH one '/' belongs to
	// the additional data.
	pos+=2; // skip both '/' 
        if (pos >= tc.length()) {
#if DEBUGMODE>0
            fprintf(stderr,"Err8\n");
#endif
            return false; // nothing follows but should
        }

	lpos = pos;
	while (pos<tc.length()){
	    if (tc.at(pos)== 13) // look for <CR>
		break;
	    pos++;
	}
	if (pos>lpos)
	    tda.setBankReference(tc.substr(lpos,pos-lpos));
    }
    pos++;
    if (pos>=tc.length())
        // nothing follows, so we have finished
        return true;

    // otherwise there is some additional data to parse (it SUCKS)
    // need <LF> <"/"> sequence
    if (pos+2>=tc.length()){
        // nothing follows, but should
#if DEBUGMODE>0
        fprintf(stderr,"Err9\n");
#endif
        return false;
    }

    // <LF> should follow
    if (tc.at(pos)!=10) {
        // false if it doesnt
#if DEBUGMODE>0
        fprintf(stderr,"Err10\n");
#endif
        return false;
    }
    pos++;
    if (tc.at(pos++)!='/') {
#if DEBUGMODE>0
        fprintf(stderr,"Err11\n");
#endif
        return false;
    }

    // now start reading additional data
    // ok, its enough for today. just skip additional data
    // pos points to the beginning of addittional data (behind the "/")
    // i.e. what follows would be: original currency amount, charges.

    return true;
}


bool SWIFTparser::_mt940_86(string tc, Transaction &tda){
    unsigned int pos;
    string field;
    int fid;   // field id
    string fd; // field content
    string tmp;

    // Check if the :86: tag is SWIFT-structured.
    if (tc.length()<3 || !isdigit(tc.at(0)) || 
		    !isdigit(tc.at(1)) || !isdigit(tc.at(2)) || 
		    (tc.length()>=4 && (tc.at(3)!='?'))) {
        // No, it is not. Return tag as description.
        tda.addDescription(tc);
        return true;
    }

    pos=0;
    
    // read transaction code
    tmp=tc.substr(0,3);
    tda.setTransactionCode(atoi(tmp.c_str()));
    pos=3;

    // check if there are fields
    if (pos>=tc.length())
        // no fields, finished
        return true;

    // skip blanks, get first real character
    while (pos<tc.length()) {
        if (!isspace(tc[pos]))
            break;
        pos++;
    }
    if (pos>=tc.length()) {
        fprintf(stderr, "empty :86: tag\n");
        return true;
    }
    if (tc[pos]!='?') {
        // simply add the whole rest as a description
        tda.addDescription(tc.substr(pos));
    }
    else {
        // otherwise parse all fields
        do {
            field=SWIFTparser::nextFIELD(tc,pos);
            pos+=field.length();
            if (!field.empty()) {
                // handle the field:
                fid=fieldId(field);
                fd=fieldContent(field);
                switch(fid) {
                case 0:   // transaction text
                    tda.setTransactionText(fd);
                    break;

                case 10:  // primanota
                    tda.setPrimanota(fd);
                    break;

                case 20:  // Y
                case 21:
                case 22:
                case 23:
                case 24:
                case 25:
                case 26:
                case 27:
                case 28:
                case 29:
                case 60:
                case 61:
                case 62:
                case 63:  // description
                    tda.addDescription(fd);
                    break;

                case 30:  // partners institute code
                    tda.setOtherBankCode(fd);
                    break;

                case 31:  // partners account id
                    tda.setOtherAccountId(fd);
                    break;

                case 32:  // Y
                case 33:  // partners name
                    tda.addOtherName(fd);
                    break;

                default:  // simply ignore all other fields ;-)
                    break;
                } // switch
            } // if field not empty
        } while((!(field.empty())) && pos<tc.length());
    }
    return true;
}


bool SWIFTparser::_mt940_25(string tc, string &instCode, string &id){
    unsigned int pos;
    bool found_inst;

    pos=0;
    found_inst=false;
    // first read the institute code (BLZ)
    while(pos<tc.length()) {
        if (tc.at(pos)=='/'){
            // store our institute code
            instCode=tc.substr(0,pos);
            found_inst=true;
            break;
        } // if
        pos++;
    } // while

    if(!found_inst)
        // no institute code, just bank number
        pos=0;
    else {
        // skip over institute code, continue with
        // account number
        pos++; // skip "/"
        if (pos>=tc.length())
            // no id
            return false;
    }
    id=tc.substr(pos);
    return true;
}


bool SWIFTparser::_mt940_60_62(string tc,
			       transactionReport &tr,
			       string tagId){
    string tmp;
    unsigned int pos;
    double d;
    bool debit;
    string curr;
    Balance bal;

    pos=0;

    // get credit/debit mark
    if (pos>=tc.length())
        // bad string
        return false;
    debit=tc.at(pos)=='D'?true:false;
    bal.setDebit(debit);
    pos++;

    // read date
    bal.setDate(Date(tc.substr(pos, 6), 2));
    pos+=6;

    // read currency (3 digit)
    if (pos+3>=tc.length())
        // bad string
        return false;
    curr=tc.substr(pos,3);
    pos+=3;
    tr.setCurrency(curr);

    // read value
    if (pos>=tc.length())
        // bad string
        return false;

    tmp=tc.substr(pos,tc.length()-pos);
    d=String::string2double(tmp);
    bal.setValue(Value(d, curr));

    tmp=tagId.substr(0,2);
    if (tmp=="60")
	// opening balance
	tr.setBalanceStart(bal);
    else
	tr.setBalanceEnd(bal);
    return true;
}


bool SWIFTparser::readMT940(string mt940record,
                            transactionReport &tr,
                            unsigned int &pos) {
    string ourInstCode;
    string ourId;
    string tag;
    string id, id2;
    string content;
    Transaction tda;
    bool endofcycle;
    bool hasTData;

#if DEBUGMODE>3
    fprintf(stderr,"SWIFT: Started handling mt940 (pos=%d of %d).\n",
            pos,mt940record.length());
#endif

    endofcycle=false;
    hasTData=false;

    /* a SWIFT document begins with a <CR> <LF> sequence,
     * and ends with a <CR><LF><-> sequence.
     * well, some institutes do NOT comply to this standard, so I had to
     * lighten up.
     */

    // ok, lets parse ;-)
    mt940record=_normalize(mt940record);
    while (pos<mt940record.length()) {
#if DEBUGMODE>3
        fprintf(stderr,"SWIFT: Started loop (pos=%d of %d).\n",
                pos,mt940record.length());
#endif
        // first skip CRLF and "-" at beginning
        while (pos<mt940record.length()) {
            if ((mt940record.at(pos)!=13) &&
                (mt940record.at(pos)!=10) &&
                (mt940record.at(pos)!='-'))
                break;
            pos++;
        } // while
        if (pos>=mt940record.length()) {
            // we are through
            break;
        }

        // copy the current TAG into a tmp var
        tag=nextTAG(mt940record,pos);
        if (tag.empty()) {
#if DEBUGMODE>1
            fprintf(stderr,"SWIFT: TAG is empty when it shouln't. (pos=%d)\n",
                    pos);
#endif
            return false;
        }
        // advance TAG-pointer to the next TAG
        pos+=tag.length();
        // get TAGs id and content
        id=tagId(tag);
        id2=id.substr(0,2);
        content=tagContent(tag);

        // now parse the tag
        if (id=="25") {
            if (!_mt940_25(content,ourInstCode,ourId)) {
#if DEBUGMODE>1
                fprintf(stderr,"SWIFT: Error in mt940_25\n");
#endif
                return false;
            }
        }
        else if (id=="61") {
            // update and store the current Transaction
            if (hasTData) {
#if DEBUGMODE>3
                fprintf(stderr,"SWIFT: Adding transaction\n");
#endif
                tr.addTransaction(tda);
            }
            // allow further 86 tags
            endofcycle=false;
            tda=Transaction();
            tda.setOurBankCode(ourInstCode);
            tda.setOurAccountId(ourId);
            if (!_mt940_61(content,tda, tr.currency())){
#if DEBUGMODE>1
                fprintf(stderr,"SWIFT: Error in mt940_61\n");
#endif
                return false;
            }
            hasTData=true;
        }
        else if (id=="86") {
            if (!endofcycle) {
                if (!_mt940_86(content,tda)){
#if DEBUGMODE>1
                    fprintf(stderr,"SWIFT: Error in mt940_86\n");
#endif
                    return false;
                }
            } // if not endofcycle
        } // :86:
        else if (id2=="60" || id2=="62") {
            // if is it the closing saldo ?
            if (id2=="62")
                // yes,  so ignore further ":86:" tags
                endofcycle=true;
            // read the closing saldo
            if (!_mt940_60_62(content,tr,id)){
#if DEBUGMODE>1
                fprintf(stderr,"SWIFT: Error in mt940_62\n");
#endif
                return false;
            }
        }
    } // while

    // update and store last Transaction if any
    if (hasTData) {
#if DEBUGMODE>3
        fprintf(stderr,"SWIFT: Adding transaction (2)\n");
#endif
        tr.addTransaction(tda);
    }

    // thats it
    return true;
}

} // namespace HBCI



extern int HBCI_SWIFTparser_readMT940(const char *mt940record,
				      HBCI_transactionReport *tr,
				      unsigned int *pos)
{
    assert(mt940record);
    assert(tr);
    assert(pos);
    return HBCI::SWIFTparser::readMT940(string(mt940record), 
					*tr,
					*pos);
}



HBCI_transactionReport *HBCI_transactionReport_new()
{
    return new HBCI::transactionReport();
}
void HBCI_transactionReport_delete(HBCI_transactionReport *h)
{
    assert(h);
    delete(h);
}

const list_HBCI_Transaction *
HBCI_transactionReport_transactions(const HBCI_transactionReport *h)
{
    assert(h);
    return &(h->transactions());
}
const HBCI_Balance *
HBCI_transactionReport_balanceStart(const HBCI_transactionReport *h)
{
    assert(h);
    return &(h->balanceStart());
}
const HBCI_Balance *
HBCI_transactionReport_balanceEnd(const HBCI_transactionReport *h)
{
    assert(h);
    return &(h->balanceEnd());
}
const char* 
HBCI_transactionReport_currency(const HBCI_transactionReport *h)
{
    assert(h);
    return h->currency().c_str();
}

