/***************************************************************************
 $RCSfile: outbox.cpp,v $
                             -------------------
    cvs         : $Id: outbox.cpp,v 1.25 2003/05/19 12:13:45 cstim Exp $
    begin       : Sat Jun 08 2002
    copyright   : (C) 2001 by Martin Preuss
    email       : openhbci@aquamaniac.de

 ***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "outbox.h"
#include <algorithm>



namespace HBCI {

customerQueue::customerQueue(Pointer<Customer> c)
:_cust(c)
{
}


customerQueue::~customerQueue(){
}


void customerQueue::addJob(Pointer<OutboxJob> job){
    _jobs.push_back(job);
}


const list<Pointer<OutboxJob> > &customerQueue::jobs() const {
    return _jobs;
}


// Function object for the predicate of matching a job status
class MatchStatus {
    OutboxJob_Status r;
  public:
    MatchStatus(OutboxJob_Status match) 
	: r(match) { };
    bool operator()(const Pointer<OutboxJob>& p) { 
	return r == HBCI_JOB_STATUS_NONE ||
	    r == p.ref().status();
    }
};
    
// Function object for the predicate of matching a job result
class MatchResult {
    OutboxJob_Result r;
  public:
    MatchResult(OutboxJob_Result match) 
	: r(match) { };
    bool operator()(const Pointer<OutboxJob>& p) { 
	return r == HBCI_JOB_RESULT_NONE ||
	    r == p.ref().result(); 
    }
};

void customerQueue::removeByState(OutboxJob_Status s){
    // This method calls MatchStatus::operator() for each list element
    // and removes that element if true is returned.
    _jobs.remove_if( MatchStatus(s) );
}

void customerQueue::removeByResult(OutboxJob_Result s){
    // This method calls MatchResult::operator() for each list element
    // and removes that element if true is returned.
    _jobs.remove_if( MatchResult(s) );
}


bool customerQueue::removeJob(Pointer<OutboxJob> job){
    list<Pointer<OutboxJob> >::iterator it1 = 
	find( _jobs.begin(), _jobs.end(), job);
    
    if ( it1 != _jobs.end() ) {
	_jobs.erase(it1);
	return true;
    }

    return false;
}


bool customerQueue::empty() const {
    return _jobs.empty();
}


int customerQueue::sizeByStatus(OutboxJob_Status s) const {
    // This algorithm calls MatchStatus::operator() for each list
    // element and counts the occurrences of 'true'
    return count_if( _jobs.begin(), _jobs.end(), MatchStatus(s) );
}

int customerQueue::sizeByResult(OutboxJob_Result s) const {
    // This algorithm calls MatchResult::operator() for each list
    // element and counts the occurrences of 'true'
    return count_if( _jobs.begin(), _jobs.end(), MatchResult(s) );
}


bool customerQueue::allDialogJobs() const {
    list<Pointer<OutboxJob> >::const_iterator it;

    for (it=_jobs.begin(); it!=_jobs.end(); it++) 
        if (!(*it).ref().isDialogJob())
	    return false;

    return true;
}






bankQueue::bankQueue(Pointer<Bank> b)
:_bank(b)
{
}


bankQueue::~bankQueue(){
}


bool bankQueue::empty() const {
    return _custQueues.empty();
}


int bankQueue::sizeByStatus(OutboxJob_Status s) const {
    int i;
    list<Pointer<customerQueue> >::const_iterator it;

    i=0;

    for (it=_custQueues.begin(); it!=_custQueues.end(); it++)
        i+=(*it).ref().sizeByStatus(s);

    return i;
}


int bankQueue::sizeByResult(OutboxJob_Result s) const {
    int i;
    list<Pointer<customerQueue> >::const_iterator it;

    i=0;

    for (it=_custQueues.begin(); it!=_custQueues.end(); it++)
        i+=(*it).ref().sizeByResult(s);
    return i;
}

bool bankQueue::allDialogJobs() const {
    list<Pointer<customerQueue> >::const_iterator it;

    for (it=_custQueues.begin(); it!=_custQueues.end(); it++)
        if (!(*it).ref().allDialogJobs())
	    return false;
    return true;
}

const list<Pointer<customerQueue> > &bankQueue::customerQueues() const {
    return _custQueues;
}


void bankQueue::addJob(Pointer<Customer> c,
                       Pointer<OutboxJob> job){
    list<Pointer<customerQueue> >::const_iterator it;

    // find customerQueue for the customer this job belongs to
    for (it=_custQueues.begin(); it!=_custQueues.end(); it++){
        if ((*it).ref().customer()==c) {
            // customerQueue found, add job
            (*it).ref().addJob(job);
            return;
        }
    } // for
    // otherwise create new customerQueue and add job there
    Pointer<customerQueue> q=new customerQueue(c);
    q.ref().addJob(job);
    _custQueues.push_back(q);
}


void bankQueue::removeByState(OutboxJob_Status s){
    list<Pointer<customerQueue> >::iterator it1;
    bool goon;

    // remove all jobs with the given state
    for (it1=_custQueues.begin(); it1!=_custQueues.end(); it1++)
        (*it1).ref().removeByState(s);

    // cleanup, empty customQueues are removed
    goon=true;
    while(goon) {
        for (it1=_custQueues.begin(); it1!=_custQueues.end(); it1++)
            if ((*it1).ref().empty()) {
                _custQueues.erase(it1);
                break;
            }
        goon=false;
    }
}


void bankQueue::removeByResult(OutboxJob_Result s){
    list<Pointer<customerQueue> >::iterator it1;
    bool goon;

    // remove all jobs with the given state
    for (it1=_custQueues.begin(); it1!=_custQueues.end(); it1++)
        (*it1).ref().removeByResult(s);

    // cleanup, empty customQueues are removed
    goon=true;
    while(goon) {
        for (it1=_custQueues.begin(); it1!=_custQueues.end(); it1++)
            if ((*it1).ref().empty()) {
                _custQueues.erase(it1);
                break;
            }
        goon=false;
    }
}


bool bankQueue::removeJob(Pointer<OutboxJob> job){
    list<Pointer<customerQueue> >::iterator it1;
    bool removed;
    bool goon;

    removed=false;
    for (it1=_custQueues.begin(); it1!=_custQueues.end(); it1++) {
        if ((*it1).ref().removeJob(job)) {
            removed=true;
            break;
        }
    } // for

    if (!removed)
        return false;

    // cleanup, empty customQueues are removed
    goon=true;
    while(goon) {
        for (it1=_custQueues.begin(); it1!=_custQueues.end(); it1++)
            if ((*it1).ref().empty()) {
                _custQueues.erase(it1);
                break;
            }
        goon=false;
    }
    return removed;
}


int bankQueue::customerCount() const {
    int i;
    int j;
    list<Pointer<customerQueue> >::const_iterator it;

    i=0;
    j=0;
    for (it=_custQueues.begin(); it!=_custQueues.end(); it++) {
        i=(*it).ref().sizeByStatus(HBCI_JOB_STATUS_TODO);
        if (i)
            j++;
    }

    return j;
}






int Outbox::_nextId=0;


Outbox::Outbox(const API *a)
:_hbci(a)
{
}


Outbox::~Outbox(){
}


void Outbox::addJob(Pointer<OutboxJob> job){
    list<Pointer<bankQueue> >::const_iterator it;
    Pointer<Bank> bank;
    Pointer<Customer> c;

    // gather some information about the job
    bank=job.ref().customer().ref().user().ref().bank();
    c=job.ref().customer();

    // set ID, since the job is no going to be enqueued
    job.ref().setId(nextId());
    // find bankQueue for the bank this job belongs to
    for (it=_banks.begin(); it!=_banks.end(); it++){
        if ((*it).ref().bank()==bank) {
            // bankQueue found, add job there
            (*it).ref().addJob(c,job);
            return;
        }
    } // for
    // otherwise create new bankQueue, add job there
    Pointer<bankQueue> q=new bankQueue(bank);
    q.ref().addJob(c, job);
    _banks.push_back(q);
}


void Outbox::removeByState(OutboxJob_Status s){
    list<Pointer<bankQueue> >::iterator it1;
    bool goon;

    // remove all jobs with the given state
    for (it1=_banks.begin(); it1!=_banks.end(); it1++)
        (*it1).ref().removeByState(s);

    // cleanup, empty bankQueues are removed
    goon=true;
    while(goon) {
        for (it1=_banks.begin(); it1!=_banks.end(); it1++)
            if ((*it1).ref().empty()) {
                _banks.erase(it1);
                break;
            }
        goon=false;
    }
}


void Outbox::removeByResult(OutboxJob_Result s){
    list<Pointer<bankQueue> >::iterator it1;
    bool goon;

    // remove all jobs with the given state
    for (it1=_banks.begin(); it1!=_banks.end(); it1++)
        (*it1).ref().removeByResult(s);

    // cleanup, empty bankQueues are removed
    goon=true;
    while(goon) {
        for (it1=_banks.begin(); it1!=_banks.end(); it1++)
            if ((*it1).ref().empty()) {
                _banks.erase(it1);
                break;
            }
        goon=false;
    }
}


void Outbox::removeJob(Pointer<OutboxJob> job){
    list<Pointer<bankQueue> >::iterator it1;
    bool removed;
    bool goon;

    removed=false;
    for (it1=_banks.begin(); it1!=_banks.end(); it1++) {
        if ((*it1).ref().removeJob(job)) {
            removed=true;
            break;
        }
    }

    if (!removed)
        return;

    // cleanup, empty bankQueues are removed
    goon=true;
    while(goon) {
        for (it1=_banks.begin(); it1!=_banks.end(); it1++)
            if ((*it1).ref().empty()) {
                _banks.erase(it1);
                break;
            }
        goon=false;
    }
}


int Outbox::sizeByStatus(OutboxJob_Status s) const {
    int i;
    list<Pointer<bankQueue> >::const_iterator it;

    i=0;
    // remove all jobs with the given state
    for (it=_banks.begin(); it!=_banks.end(); it++)
        i+=(*it).ref().sizeByStatus(s);

    return i;
}


int Outbox::sizeByResult(OutboxJob_Result s) const {
    int i;
    list<Pointer<bankQueue> >::const_iterator it;

    i=0;
    // coutnremove all jobs with the given state
    for (it=_banks.begin(); it!=_banks.end(); it++)
        i+=(*it).ref().sizeByResult(s);

    return i;
}

bool Outbox::allDialogJobs() const {
    list<Pointer<bankQueue> >::const_iterator it;

    for (it=_banks.begin(); it!=_banks.end(); it++)
        if (!(*it).ref().allDialogJobs())
	    return false;
    return true;
}


int Outbox::bankCount() const {
    int i;
    int j;
    list<Pointer<bankQueue> >::const_iterator it;

    i=0;
    j=0;
    for (it=_banks.begin(); it!=_banks.end(); it++) {
        i=(*it).ref().sizeByStatus(HBCI_JOB_STATUS_TODO);
        if (i)
            j++;
    }

    return j;
}


int Outbox::customerCount() const {
    int i;
    list<Pointer<bankQueue> >::const_iterator it;

    i=0;
    for (it=_banks.begin(); it!=_banks.end(); it++)
        i+=(*it).ref().customerCount();

    return i;
}



void Outbox::clear(){
    _banks.clear();
}


const list<Pointer<bankQueue> > &Outbox::bankQueues() const {
    return _banks;
}


list<Pointer<OutboxJob> > Outbox::jobs() const {
    list<Pointer<OutboxJob> > result;
    list<Pointer<OutboxJob> > jl2;
    list<Pointer<bankQueue> >::const_iterator it1;
    list<Pointer<customerQueue> >::const_iterator it2;
    list<Pointer<OutboxJob> >::const_iterator it3;

    for (it1=_banks.begin(); it1!=_banks.end(); it1++) {
        for (it2=(*it1).ref().customerQueues().begin();
             it2!=(*it1).ref().customerQueues().end();
             it2++) {
            for (it3=(*it2).ref().jobs().begin();
                 it3!=(*it2).ref().jobs().end();
                 it3++)
                result.push_back(*it3);
        } // for customer
    } // for bank

    return result;
}


int Outbox::nextId() {
    return _nextId++;
}


void Outbox::setNextId(int i) {
    if (i>_nextId)
        _nextId=i;
}

} // namespace HBCI
