/*----------------------------------------------------------------------------

   libtunepimp -- The MusicBrainz tagging library.
                  Let a thousand taggers bloom!

   Copyright (C) Robert Kaye 2003

   This file is part of libtunepimp.

   libtunepimp is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   libtunepimp is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with libtunepimp; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

   $Id: c_wrapper.cpp,v 1.40 2004/03/05 20:27:59 robert Exp $

----------------------------------------------------------------------------*/
#include "tunepimp.h"
#include "mutex.h"
#include "tp_c.h"

#define DB printf("%s:%d\n", __FILE__, __LINE__);

#define TP_OBJ_CHECK(o) TunePimp *obj = (TunePimp *)o; \
                        if (obj == NULL) return 0;
#define TP_OBJ_CHECKV(o) TunePimp *obj = (TunePimp *)o; \
                         if (obj == NULL) return;
#define TP_OBJ_CHECKN(o) TunePimp *obj = (TunePimp *)o; \
                         if (obj == NULL) return NULL;
#define TR_OBJ_CHECK(o) Track *obj = (Track *)t; \
                        if (obj == NULL) return 0;
#define TR_OBJ_CHECKV(o) Track *obj = (Track *)t; \
                         if (obj == NULL) return;

#include <deque>

class Callback : public TPCallback
{
    public:

        Callback(void)
        {
            notifyCallback = NULL;
            statusCallback = NULL;
            notifyData = statusData = NULL;
        };
        virtual ~Callback(void) {};

        void notify(TunePimp *pimp, TPCallbackEnum type, int fileId)
        {
            notifyMutex.acquire();
            if (notifyCallback)
                (*notifyCallback)((tunepimp_t)pimp, notifyData, type, fileId);
            else
            {
                pair<TPCallbackEnum, int> p, top;
                bool                      add = true;

                if (notifyQueue.size() > 0)
                {
                    top = notifyQueue.front();
                    if (top.first == type && top.second == fileId)
                        add = false; 
                }

                if (add)
                {
                    p.first = type;
                    p.second = fileId;
                    notifyQueue.push_back(p);
                }
            }
            notifyMutex.release();
        };
        void status(TunePimp *pimp, const string &status)
        {
            statusMutex.acquire();
            if (statusCallback)
                (*statusCallback)((tunepimp_t)pimp, statusData, status.c_str());
            else
                statusQueue.push_back(status);
            statusMutex.release();
        };
        bool getNotification(TPCallbackEnum &type, int &fileId)
        {
            bool ret = false;

            notifyMutex.acquire();
            if (notifyQueue.size() > 0)
            {
                pair<TPCallbackEnum, int> p;

                p = notifyQueue.front();
                notifyQueue.pop_front();

                type = p.first;
                fileId = p.second;

                ret = true;
            }
            notifyMutex.release();

            return ret;
        }
        bool getStatus(string &status)
        {
            bool ret = false;

            statusMutex.acquire(); 
            if (statusQueue.size() > 0)
            {
                status = statusQueue.front();
                statusQueue.pop_front();
                ret = true;
            }
            statusMutex.release();
            return ret;
        }

        tp_notify_callback  notifyCallback;
        tp_status_callback  statusCallback;
        void               *notifyData, *statusData;

    private:

        TunePimp                          *tunePimp;
        deque<pair<TPCallbackEnum, int> >  notifyQueue;
        deque<string>                      statusQueue;
        Mutex                              statusMutex, notifyMutex;
};

extern "C"
{

tunepimp_t tp_New(const char *appName, const char *appVersion)
{
    TunePimp *pimp = new TunePimp(appName, appVersion);
    Callback *cb = new Callback();

    pimp->setCallback(cb);

    return (tunepimp_t)pimp;
}

tunepimp_t tp_NewWithArgs(const char *appName, const char *appVersion, int startThreads)
{
    TunePimp *pimp = new TunePimp(appName, appVersion, startThreads);
    Callback *cb = new Callback();

    pimp->setCallback(cb);

    return (tunepimp_t)pimp;
}

void tp_Delete(tunepimp_t o)
{
    Callback *old;

    TP_OBJ_CHECKV(o);

    old = (Callback *)obj->getCallback();
    delete (TunePimp *)obj;
    delete old;
}

void tp_SetNotifyCallback(tunepimp_t o, tp_notify_callback callback, void *data)
{
    Callback *cb;

    TP_OBJ_CHECKV(o);

    cb = (Callback *)obj->getCallback();
    cb->notifyCallback = callback;
    cb->notifyData = data;
}

tp_notify_callback tp_GetNotifyCallback(tunepimp_t o)
{
    Callback *cb;

    TP_OBJ_CHECKN(o);

    cb = (Callback *)obj->getCallback();
    return cb->notifyCallback;
}

void tp_SetStatusCallback(tunepimp_t o, tp_status_callback callback, void *data)
{
    Callback *cb;

    TP_OBJ_CHECKV(o);

    cb = (Callback *)obj->getCallback();
    cb->statusCallback = callback;
    cb->statusData = data;
}

tp_status_callback tp_GetStatusCallback(tunepimp_t o)
{
    Callback *cb;

    TP_OBJ_CHECKN(o);

    cb = (Callback *)obj->getCallback();
    return cb->statusCallback;
}

int tp_GetNotification(tunepimp_t o, TPCallbackEnum *type, int *fileId)
{
    Callback *cb;

    TP_OBJ_CHECK(o);

    cb = (Callback *)obj->getCallback();
    return (int)cb->getNotification(*type, *fileId);
}

int tp_GetStatus(tunepimp_t o, char *status, int statusLen)
{
    Callback *cb;
    string    s;

    TP_OBJ_CHECK(o);

    cb = (Callback *)obj->getCallback();
    if (cb->getStatus(s))
    {
        strncpy(status, s.c_str(), statusLen - 1);
        status[statusLen - 1] = 0;
        return 1;
    }
    return 0;
}

void tp_GetVersion(tunepimp_t o, int *major, int *minor, int *rev)
{
    *major = *minor = *rev = 0;

    TP_OBJ_CHECKV(o);

    obj->getVersion(*major, *minor, *rev);
}

void tp_SetServer(tunepimp_t o, const char *serverAddr, short serverPort)
{
    TP_OBJ_CHECKV(o);

    obj->setServer(string(serverAddr), serverPort);
}

void tp_GetServer(tunepimp_t o,
                    char *serverAddr, int maxLen,
                    short *serverPort)
{
    string tmpServerAddr;

    TP_OBJ_CHECKV(o);

    obj->getServer(tmpServerAddr, *serverPort);
    strncpy(serverAddr, tmpServerAddr.c_str(), maxLen - 1);
    serverAddr[maxLen - 1] = 0;
}

void tp_SetDebug(tunepimp_t o, int debug)
{
    TP_OBJ_CHECKV(o);

    obj->setDebug((bool)debug);
}

int tp_GetDebug(tunepimp_t o)
{
    TP_OBJ_CHECK(o);

    return obj->getDebug();
}

void tp_SetProxy(tunepimp_t o, const char *proxyAddr, short proxyPort)
{
    TP_OBJ_CHECKV(o);

    string addr = "";
    if (proxyAddr)
        addr = proxyAddr;
    obj->setProxy(addr, proxyPort);
}

void tp_GetProxy(tunepimp_t o,
                 char *proxyAddr, int maxLen,
                 short *proxyPort)
{
    string tmpProxyAddr;

    TP_OBJ_CHECKV(o);

    obj->getProxy(tmpProxyAddr, *proxyPort);
    strncpy(proxyAddr, tmpProxyAddr.c_str(), maxLen - 1);
    proxyAddr[maxLen - 1] = 0;
}

void tp_SetUseUTF8(tunepimp_t o, int UTF8)
{
    TP_OBJ_CHECKV(o);
    obj->setUseUTF8((bool)UTF8);
}

int tp_GetUseUTF8(tunepimp_t o)
{
    TP_OBJ_CHECK(o);
    return obj->getUseUTF8();
}

void tp_SetUserInfo(tunepimp_t o, const char *userName, const char *password)
{
    TP_OBJ_CHECKV(o);
    obj->setUserInfo(userName, password);
}

void tp_GetUserInfo(tunepimp_t o,
                    char *userName, int maxUserNameLen,
                    char *password, int maxPasswordLen)
{
    string tmpUserName;
    string tmpPassword;

    TP_OBJ_CHECKV(o);

    obj->getUserInfo(tmpUserName, tmpPassword);
    strncpy(userName, tmpUserName.c_str(), maxUserNameLen - 1);
    userName[maxUserNameLen - 1] = 0;
    strncpy(password, tmpPassword.c_str(), maxPasswordLen - 1);
    password[maxPasswordLen - 1] = 0;
}

int tp_GetNumSupportedExtensions(tunepimp_t o)
{
    vector<string> extList;

    TP_OBJ_CHECK(o);

    obj->getSupportedExtensions(extList);

    return extList.size();
}

void tp_GetSupportedExtensions(tunepimp_t o, char extensions[][TP_EXTENSION_LEN])
{
    vector<string>            extList;
    vector<string>::iterator  i;
    int                       count;

    TP_OBJ_CHECKV(o);

    obj->getSupportedExtensions(extList);
    for(i = extList.begin(), count = 0; i != extList.end(); i++, count++)
       strcpy(extensions[count], (*i).c_str());
}

void tp_SetAnalyzerPriority(tunepimp_t o, TPThreadPriorityEnum priority)
{
    TP_OBJ_CHECKV(o);
    obj->setAnalyzerPriority(priority);
}

TPThreadPriorityEnum tp_GetAnalyzerPriority(tunepimp_t o)
{
    TunePimp *obj = (TunePimp *)o;

    if (o == NULL)
        return eNormal;

    return obj->getAnalyzerPriority();
}

void tp_SetAutoFileLookup(tunepimp_t o, int enable)
{
    TP_OBJ_CHECKV(o);
    obj->setAutoFileLookup((bool)enable);
}

int tp_GetAutoFileLookup(tunepimp_t o)
{
    TP_OBJ_CHECK(o);
    return obj->getAutoFileLookup();
}

void tp_GetError(tunepimp_t o, char *error, int maxLen)
{
    string err;

    TP_OBJ_CHECKV(o);

    obj->getError(err);
    strncpy(error, err.c_str(), maxLen - 1);
    error[maxLen - 1] = 0;
}

int tp_AddFile(tunepimp_t o, const char *fileName)
{
    TP_OBJ_CHECK(o);
    return (int)obj->addFile(fileName);
}

int tp_AddDir(tunepimp_t o, const char *dirPath)
{
    TP_OBJ_CHECK(o);
    return (int)obj->addDir(dirPath);
}

void tp_Remove(tunepimp_t o, int fileId)
{
    TP_OBJ_CHECKV(o);
    obj->remove(fileId);
}

void tp_Wake(tunepimp_t o, track_t track)
{
    TP_OBJ_CHECKV(o);
    obj->wake((Track *)track);
}

int tp_GetNumFiles(tunepimp_t o)
{
    TP_OBJ_CHECK(o);
    return obj->getNumFiles();
}

int tp_GetNumUnsubmitted(tunepimp_t o)
{
    TP_OBJ_CHECK(o);
    return obj->getNumUnsubmitted();
}

int tp_GetTrackCounts(tunepimp_t o, int *counts, int maxCounts)
{
    map<TPFileStatus, int> countMap;
    int                    i;

    TP_OBJ_CHECK(o);

    obj->getTrackCounts(countMap);

    for(i = 0; i < maxCounts && i < (int)eLastStatus; i++)
        counts[i] = countMap[(TPFileStatus)i];

    return i - 1;
}

int tp_GetNumUnsavedItems(tunepimp_t o)
{
    TP_OBJ_CHECK(o);
    return obj->getNumUnsavedItems();
}

int tp_GetNumFileIds(tunepimp_t o)
{
    vector<int> ids;

    TP_OBJ_CHECK(o);

    obj->getFileIds(ids);
    return ids.size();
}

void tp_GetFileIds(tunepimp_t o, int *ids, int numIds)
{
    vector<int>           vec;
    vector<int>::iterator i;

    TP_OBJ_CHECKV(o);

    obj->getFileIds(vec);
    for(i = vec.begin(); i != vec.end() && numIds > 0; i++, ids++, numIds--)
        *ids = *i;
}

track_t tp_GetTrack(tunepimp_t o, int fileId)
{
    TP_OBJ_CHECKN(o);
    return (track_t)obj->getTrack(fileId);
}

void tp_ReleaseTrack(tunepimp_t o, track_t track)
{
    TP_OBJ_CHECKV(o);

    if (track == NULL)
        return;

    obj->releaseTrack((Track *)track);
}

TPError tp_SelectResult(tunepimp_t o, track_t track, int resultIndex)
{
    TunePimp *obj = (TunePimp *)o;
    if (o == NULL)
        return tpInvalidObject;

    if (track == NULL)
        return tpInvalidObject;

    obj->selectResult((Track *)track, resultIndex);

    return tpOk;
}

void tp_Misidentified(tunepimp_t o, int fileId)
{
    TP_OBJ_CHECKV(o);
    obj->misidentified(fileId);
}

void tp_IdentifyAgain(tunepimp_t o, int fileId)
{
    TP_OBJ_CHECKV(o);
    obj->identifyAgain(fileId);
}

int tp_GetRecognizedFileList(tunepimp_t o, int threshold, int **fileIds, int *numIds)
{
    vector<int>           ids;
    vector<int>::iterator i;
    int                   ret, *ptr;

    TP_OBJ_CHECK(o);
    ret = obj->getRecognizedFileList(threshold, ids);

    if (ids.size() > 0)
    {
        *fileIds = (int *)malloc(sizeof(int) * ids.size());
        
        for(i = ids.begin(), ptr = *fileIds; i != ids.end(); i++, ptr++)
            *ptr = *i;
        *numIds = ids.size();
    }
    else
    {
        *numIds = 0;
        *fileIds = NULL;
    }

    return ret;
}

void tp_DeleteRecognizedFileList(tunepimp_t o, int *fileIds)
{
    if (fileIds)
        free(fileIds);
}

int tp_WriteTags(tunepimp_t o, int *fileIds, int numFileIds)
{
    vector<int> ids;

    TP_OBJ_CHECK(o);

    if (!fileIds)
        return obj->writeTags(NULL);

    for(; numFileIds > 0; numFileIds--, fileIds++)
        ids.push_back(*fileIds);

    return obj->writeTags(&ids);
}

void tp_AddTRMSubmission(tunepimp_t o, const char *trackId, const char *trmId)
{
    TP_OBJ_CHECKV(o);
    obj->addTRMSubmission(string(trackId), string(trmId));
}

TPError tp_SubmitTRMs(tunepimp_t o)
{
    TunePimp *obj = (TunePimp *)o;
    if (o == NULL)
        return tpInvalidObject;

    return obj->submitTRMs();
}

void tp_SetRenameFiles(tunepimp_t o, int rename)
{
    TP_OBJ_CHECKV(o);
    obj->setRenameFiles((bool)rename);
}

int tp_GetRenameFiles(tunepimp_t o)
{
    TP_OBJ_CHECK(o);
    return obj->getRenameFiles();
}

void tp_SetMoveFiles(tunepimp_t o, int move)
{
    TP_OBJ_CHECKV(o);
    obj->setMoveFiles((bool)move);
}

int tp_GetMoveFiles(tunepimp_t o)
{
    TP_OBJ_CHECK(o);
    return obj->getMoveFiles();
}

void tp_SetWriteID3v1(tunepimp_t o, int writeID3v1)
{
    TP_OBJ_CHECKV(o);
    obj->setWriteID3v1((bool)writeID3v1);
}

int tp_GetWriteID3v1(tunepimp_t o)
{
    TP_OBJ_CHECK(o);
    return obj->getWriteID3v1();
}

void tp_SetClearTags(tunepimp_t o, int clearTags)
{
    TP_OBJ_CHECKV(o);
    obj->setClearTags((bool)clearTags);
}

int tp_GetClearTags(tunepimp_t o)
{
    TP_OBJ_CHECK(o);
    return obj->getClearTags();
}

void tp_SetFileMask(tunepimp_t o, const char *fileMask)
{
    TP_OBJ_CHECKV(o);
    obj->setFileMask(fileMask);
}

void tp_GetFileMask(tunepimp_t o, char *fileMask, int maxLen)
{
    string tmpFileMask;

    TP_OBJ_CHECKV(o);

    tmpFileMask = obj->getFileMask();
    strncpy(fileMask, tmpFileMask.c_str(), maxLen - 1);
    fileMask[maxLen - 1] = 0;
}

void tp_SetVariousFileMask(tunepimp_t o, const char *variousFileMask)
{
    TP_OBJ_CHECKV(o);
    obj->setVariousFileMask(variousFileMask);
}

void tp_GetVariousFileMask(tunepimp_t o, char *variousFileMask, int maxLen)
{
    string tmpVariousFileMask;

    TP_OBJ_CHECKV(o);

    tmpVariousFileMask = obj->getVariousFileMask();
    strncpy(variousFileMask, tmpVariousFileMask.c_str(), maxLen - 1);
    variousFileMask[maxLen - 1] = 0;
}

void tp_SetAllowedFileCharacters(tunepimp_t o, const char *allowedCharacters)
{
    TP_OBJ_CHECKV(o);
    obj->setAllowedFileCharacters(allowedCharacters);
}

void tp_GetAllowedFileCharacters(tunepimp_t o, char *allowedCharacters, int maxLen)
{
    string tmpAllowedCharacters;

    TP_OBJ_CHECKV(o);

    tmpAllowedCharacters = obj->getAllowedFileCharacters();
    strncpy(allowedCharacters, tmpAllowedCharacters.c_str(), maxLen - 1);
    allowedCharacters[maxLen - 1] = 0;
}

void tp_SetDestDir(tunepimp_t o, const char *destDir)
{
    TP_OBJ_CHECKV(o);
    obj->setDestDir(destDir);
}

void tp_GetDestDir(tunepimp_t o, char *destDir, int maxLen)
{
    string tmpDestDir;

    TP_OBJ_CHECKV(o);

    tmpDestDir = obj->getDestDir();
    strncpy(destDir, tmpDestDir.c_str(), maxLen - 1);
    destDir[maxLen - 1] = 0;
}

void tp_SetTopSrcDir(tunepimp_t o, const char *topSrcDir)
{
    TP_OBJ_CHECKV(o);
    obj->setTopSrcDir(topSrcDir);
}

void tp_GetTopSrcDir(tunepimp_t o, char *topSrcDir, int maxLen)
{
    string tmpTopSrcDir;

    TP_OBJ_CHECKV(o);

    tmpTopSrcDir = obj->getTopSrcDir();
    strncpy(topSrcDir, tmpTopSrcDir.c_str(), maxLen - 1);
    topSrcDir[maxLen - 1] = 0;
}

void tp_SetTRMCollisionThreshold(tunepimp_t o, int trmThreshold)
{
    TP_OBJ_CHECKV(o);
    obj->setTRMCollisionThreshold(trmThreshold);
}

int tp_GetTRMCollisionThreshold(tunepimp_t o)
{
    TP_OBJ_CHECK(o);
    return obj->getTRMCollisionThreshold();
}

void tp_SetAutoSaveThreshold(tunepimp_t o, int autoSaveThreshold)
{
    TP_OBJ_CHECKV(o);
    obj->setAutoSaveThreshold(autoSaveThreshold);
}

int tp_GetAutoSaveThreshold(tunepimp_t o)
{
    TP_OBJ_CHECK(o);
    return obj->getAutoSaveThreshold();
}

void tp_SetMinTRMThreshold(tunepimp_t o, int minThreshold)
{
    TP_OBJ_CHECKV(o);
    obj->setMinTRMThreshold(minThreshold);
}

int tp_GetMinTRMThreshold(tunepimp_t o)
{
    TP_OBJ_CHECK(o);
    return obj->getMinTRMThreshold();
}

void tp_SetMaxFileNameLen(tunepimp_t o, int maxFileNameLen)
{
    TP_OBJ_CHECKV(o);
    obj->setMaxFileNameLen(maxFileNameLen);
}

int tp_GetMaxFileNameLen(tunepimp_t o)
{
    TP_OBJ_CHECK(o);
    return obj->getMaxFileNameLen();
}

int tp_GetAutoRemovedSavedFiles(tunepimp_t o)
{
    TP_OBJ_CHECK(o);
    return obj->getAutoRemoveSavedFiles();
}

void tp_SetAutoRemovedSavedFiles(tunepimp_t o, int autoRemoveSavedFiles)
{
    TP_OBJ_CHECKV(o);
    obj->setAutoRemoveSavedFiles(autoRemoveSavedFiles);
}

#ifdef WIN32
void tp_WSAInit(tunepimp_t o)
{
    TP_OBJ_CHECKV(o);
    obj->WSAInit();
}

void tp_WSAStop(tunepimp_t o)
{
    TP_OBJ_CHECKV(o);
    obj->WSAStop();
}
#endif

/* --------------------------------------------------------------------------
 * Track Interface
 * --------------------------------------------------------------------------*/

TPFileStatus tr_GetStatus(track_t t)
{
    Track *obj = (Track *)t;
    if (t == NULL)
        return eError;

    return obj->getStatus();
}

void tr_GetFileName(track_t t, char *fileName, int maxLen)
{
    string file;

    TR_OBJ_CHECKV(t);

    obj->getFileName(file);
    strncpy(fileName, file.c_str(), maxLen - 1);
    fileName[maxLen - 1] = 0;
}

void tr_GetTRM(track_t t, char *trmArg, int maxLen)
{
    string trm;

    TR_OBJ_CHECKV(t);

    obj->getTRM(trm);
    strncpy(trmArg, trm.c_str(), maxLen - 1);
    trmArg[maxLen - 1] = 0;
}

void tr_GetLocalMetadata(track_t t, metadata_t *mdata)
{
    Metadata data;

    TR_OBJ_CHECKV(t);

    obj->getLocalMetadata(data);
    md_Clear(mdata);

    mdata->artist = strdup(data.artist.c_str());
    mdata->sortName = strdup(data.sortName.c_str());
    mdata->album = strdup(data.album.c_str());
    mdata->track = strdup(data.track.c_str());
    mdata->trackNum = data.trackNum;
    mdata->variousArtist = (bool)data.variousArtist;
    mdata->artistId = strdup(data.artistId.c_str());
    mdata->albumId = strdup(data.albumId.c_str());
    mdata->trackId = strdup(data.trackId.c_str());
    mdata->fileTrm = strdup(data.fileTrm.c_str());
    mdata->albumArtistId = strdup(data.albumArtistId.c_str());
    mdata->duration = data.duration;
    mdata->albumType = data.albumType;
    mdata->albumStatus = data.albumStatus;
    mdata->fileFormat = strdup(data.fileFormat.c_str());
    mdata->numTRMIds = data.numTRMIds;
    mdata->releaseYear = data.releaseYear;
    mdata->releaseMonth = data.releaseMonth;
    mdata->releaseDay = data.releaseDay;
    strcpy(mdata->releaseCountry, data.releaseCountry.c_str());
}

void tr_SetLocalMetadata(track_t t, const metadata_t *mdata)
{
    Metadata data;

    TR_OBJ_CHECKV(t);

    data.artist = mdata->artist ? mdata->artist : ""; 
    data.sortName = mdata->sortName ? mdata->sortName : "";
    data.album = mdata->album ? mdata->album : "";
    data.track = mdata->track ? mdata->track : "";
    data.trackNum = mdata->trackNum;
    data.variousArtist = (bool)mdata->variousArtist;
    data.artistId = mdata->artistId ? mdata->artistId : "";
    data.albumId = mdata->albumId ? mdata->albumId : "";
    data.trackId = mdata->trackId ? mdata->trackId : "";
    data.fileTrm = mdata->fileTrm ? mdata->fileTrm : "";
    data.albumArtistId = mdata->albumArtistId ? mdata->albumArtistId : "";
    data.duration = mdata->duration;
    data.albumType = mdata->albumType;
    data.albumStatus = mdata->albumStatus;
    data.numTRMIds = mdata->numTRMIds;
    data.releaseYear = mdata->releaseYear;
    data.releaseMonth = mdata->releaseMonth;
    data.releaseDay = mdata->releaseDay;
    data.releaseCountry = mdata->releaseCountry ? mdata->releaseCountry : "";

    obj->setLocalMetadata(data);
}

void tr_GetServerMetadata(track_t t, metadata_t *mdata)
{
    Metadata data;

    TR_OBJ_CHECKV(t);

    obj->getServerMetadata(data);
    md_Clear(mdata);

    mdata->artist = strdup(data.artist.c_str());
    mdata->sortName = strdup(data.sortName.c_str());
    mdata->album = strdup(data.album.c_str());
    mdata->track = strdup(data.track.c_str());
    mdata->trackNum = data.trackNum;
    mdata->variousArtist = (bool)data.variousArtist;
    mdata->artistId = strdup(data.artistId.c_str());
    mdata->albumId = strdup(data.albumId.c_str());
    mdata->trackId = strdup(data.trackId.c_str());
    mdata->fileTrm = strdup(data.fileTrm.c_str());
    mdata->albumArtistId = strdup(data.albumArtistId.c_str());
    mdata->duration = data.duration;
    mdata->albumType = data.albumType;
    mdata->albumStatus = data.albumStatus;
    mdata->numTRMIds = data.numTRMIds;
    mdata->releaseYear = data.releaseYear;
    mdata->releaseMonth = data.releaseMonth;
    mdata->releaseDay = data.releaseDay;
    strcpy(mdata->releaseCountry, data.releaseCountry.c_str());
}

void tr_SetServerMetadata(track_t t, const metadata_t *mdata)
{
    Metadata data;

    TR_OBJ_CHECKV(t);

    data.artist = mdata->artist ? mdata->artist : ""; 
    data.sortName = mdata->sortName ? mdata->sortName : "";
    data.album = mdata->album ? mdata->album : "";
    data.track = mdata->track ? mdata->track : "";
    data.trackNum = mdata->trackNum;
    data.variousArtist = (bool)mdata->variousArtist;
    data.artistId = mdata->artistId ? mdata->artistId : "";
    data.albumId = mdata->albumId ? mdata->albumId : "";
    data.trackId = mdata->trackId ? mdata->trackId : "";
    data.fileTrm = mdata->fileTrm ? mdata->fileTrm : "";
    data.albumArtistId = mdata->albumArtistId ? mdata->albumArtistId : "";
    data.duration = mdata->duration;
    data.albumType = mdata->albumType;
    data.albumStatus = mdata->albumStatus;
    data.numTRMIds = mdata->numTRMIds;
    data.releaseYear = mdata->releaseYear;
    data.releaseMonth = mdata->releaseMonth;
    data.releaseDay = mdata->releaseDay;
    data.releaseCountry = mdata->releaseCountry ? mdata->releaseCountry : "";

    obj->setServerMetadata(data);
}

void tr_GetError(track_t t, char *error, int maxLen)
{
    string err;

    TR_OBJ_CHECKV(t);

    obj->getError(err);
    strncpy(error, err.c_str(), maxLen - 1);
    error[maxLen - 1] = 0;
}

int tr_GetSimilarity(track_t t)
{
    TR_OBJ_CHECK(t);
    return obj->getSimilarity();
}

int tr_HasChanged(track_t t)
{
    TR_OBJ_CHECK(t);
    return (int)obj->hasChanged();
}

void tr_SetStatus(track_t t, const TPFileStatus status)
{
    TR_OBJ_CHECKV(t);
    obj->setStatus(status);
}

int tr_GetNumResults(track_t t)
{
    vector<TPResult *> res;
    TPResultType       type;

    TR_OBJ_CHECK(t);

    obj->getResults(type, res);
    return res.size();
}

static artistresult_t *convertArtistResult(TPArtistResult *res)
{
    artistresult_t *ar;

    ar = (artistresult_t *)calloc(sizeof(artistresult_t), 1);
    ar->relevance = res->getRelevance();
    ar->name = strdup(res->getName().c_str());
    ar->sortName = strdup(res->getSortName().c_str());
    ar->id = strdup(res->getId().c_str());

    //printf("Artist res: %d %s %s %s\n", ar->relevance, ar->name, ar->sortName, ar->id);

    return ar;
}

static void deleteArtistResult(artistresult_t *res)
{
    if (!res)
        return;

    if (res->name)
        free(res->name);

    if (res->sortName)
        free(res->sortName);

    if (res->id)
        free(res->id);

    free(res);
}

static albumresult_t *convertAlbumResult(TPAlbumResult *res)
{
    albumresult_t *al;
    TPArtistResult artist;

    al = (albumresult_t *)calloc(sizeof(albumresult_t), 1);
    al->relevance = res->getRelevance();
    al->name = strdup(res->getName().c_str());
    al->id = strdup(res->getId().c_str());
    al->numTracks = res->getNumTracks();
    al->numCDIndexIds = res->getNumCDIndexIds();
    al->isVA = (int)res->getVariousArtists();
    al->status = res->getStatus();
    al->type = res->getType();

    artist = res->getArtist();
    al->artist = convertArtistResult(&artist);

    return al;
}

static void deleteAlbumResult(albumresult_t *res)
{
    if (!res)
        return;

    if (res->name)
        free(res->name);

    if (res->id)
        free(res->id);

    free(res);
}

static albumtrackresult_t *convertAlbumTrackResult(TPAlbumTrackResult *res)
{
    albumtrackresult_t *tr;
    TPArtistResult     artist;
    TPAlbumResult      album;

    tr = (albumtrackresult_t *)calloc(sizeof(albumtrackresult_t), 1);
    tr->relevance = res->getRelevance();
    tr->name = strdup(res->getName().c_str());
    tr->id = strdup(res->getId().c_str());
    tr->numTRMIds = res->getNumTRMIds();
    tr->trackNum = res->getTrackNum();
    tr->duration = res->getDuration();

    artist = res->getArtist();
    tr->artist = convertArtistResult(&artist);

    album = res->getAlbum();
    tr->album = convertAlbumResult(&album);

    return tr;
}

static void deleteAlbumTrackResult(albumtrackresult_t *res)
{
    if (!res)
        return;

    if (res->name)
        free(res->name);

    if (res->id)
        free(res->id);

    free(res);
}

void tr_GetResults(track_t t, TPResultType *type, result_t *results, int *numResults)
{
    vector<TPResult *> res;
    vector<TPResult *>::iterator i;
    int                          count;

    TR_OBJ_CHECKV(t);

    obj->getResults(*type, res);
    for(i = res.begin(), count = 0; i != res.end() && *numResults >= 0;
        i++, (*numResults)--, results++, count++)
    {
        switch(*type)
        {
            case eArtistList:
                *results = convertArtistResult((TPArtistResult *)(*i));
                break;
            case eAlbumList:
                *results = convertAlbumResult((TPAlbumResult *)(*i));
                break;
            case eTrackList:
                *results = convertAlbumTrackResult((TPAlbumTrackResult *)(*i));
                break;
            default:
                *results = NULL;
                break;
        }
    }
    *numResults = count;
}

void tr_Lock(track_t t)
{
    TR_OBJ_CHECKV(t);
    obj->lock();
}

void tr_Unlock(track_t t)
{
    TR_OBJ_CHECKV(t);
    obj->unlock();
}

/* --------------------------------------------------------------------------
 * Result Interface
 * --------------------------------------------------------------------------*/

void rs_Delete(TPResultType type, result_t *result, int numResults)
{
    for(; numResults > 0; numResults--, result++)
        switch(type)
        {
            case eArtistList:
                deleteArtistResult((artistresult_t *)*result);
                break;
            case eAlbumList:
                deleteAlbumResult((albumresult_t *)*result);
                break;
            case eTrackList:
                deleteAlbumTrackResult((albumtrackresult_t *)*result);
                break;
            default:
                break;
        }
}

/* --------------------------------------------------------------------------
 * Metadata Interface
 * --------------------------------------------------------------------------*/

metadata_t *md_New(void)
{
    return (metadata_t *)calloc(sizeof(metadata_t), 1);
}

void md_Delete(metadata_t *data)
{
    if (!data)
        return;

    md_Clear(data);

    free(data);
}

void md_Clear(metadata_t *data)
{
    if (!data)
        return;

    if (data->artist)
        free(data->artist);
    if (data->sortName)
        free(data->sortName);
    if (data->album)
        free(data->album);
    if (data->track)
        free(data->track);
    if (data->artistId)
        free(data->artistId);
    if (data->albumId)
        free(data->albumId);
    if (data->trackId)
        free(data->trackId);
    if (data->fileTrm)
        free(data->fileTrm);
    if (data->albumArtistId)
        free(data->albumArtistId);
    if (data->fileFormat)
    free(data->fileFormat);

    memset(data, 0, sizeof(metadata_t));
}

TPAlbumStatus md_ConvertToAlbumStatus(const char *albumStatus)
{
    return convertToAlbumStatus(albumStatus);
}

TPAlbumType md_ConvertToAlbumType(const char *albumType)
{
    return convertToAlbumType(albumType);
}

void md_ConvertFromAlbumStatus(TPAlbumStatus status, char *albumStatus, int maxLen)
{
    string statusStr;

    convertFromAlbumStatus(status, statusStr);
    strncpy(albumStatus, statusStr.c_str(), maxLen - 1);
    albumStatus[maxLen - 1] = 0;
}

void md_ConvertFromAlbumType(TPAlbumType type, char *albumType, int maxLen)
{
    string typeStr;

    convertFromAlbumType(type, typeStr);
    strncpy(albumType, typeStr.c_str(), maxLen - 1);
    albumType[maxLen - 1] = 0;
}

}
