/* Copyright (C) 1995 Bjoern Beutel. */

/* Description. =============================================================*/

/* This module manages the emission of instructions and keeps track of the 
 * stack index. 
 * It supports constant folding. 
 * It also holds buffers for the compiled code. */

/* Constants. ===============================================================*/

/* Possible values of file_type. */
enum {ALLO_RULE_FILE, MORPHO_RULE_FILE, SYNTAX_RULE_FILE};

/* Types. ===================================================================*/

typedef struct /* Structure that contains compiled Malaga rule code. */
{ 
  int_t file_type; /* ALLO_RULE_FILE, MORPHO_RULE_FILE or SYNTAX_RULE_FILE. */
  int_t rule_count; /* The number of rules currently defined. */
  int_t stack_index; /* The current stack index. */
  int_t instr_count; /* The number of instraction already emitted. */

  /* The following values must be copied to the rule file: */
  int_t initial_rule_set; /* Index into RULE_SET_POOL. */
  int_t initial_fs; /* Index into VALUE_POOL. */
  int_t robust_rule; /* Index into RULE_POOL. */
  int_t pruning_rule; /* Index into RULE_POOL. */
  int_t allo_rule; /* Index into RULE_POOL. */
  int_t input_filter; /* Index into RULE_POOL. */
  int_t output_filter; /* Index into RULE_POOL. */
  pool_t rule_pool; /* The pool of all rules. */
  pool_t rule_set_pool; /* The pool of rule sets. */
  pool_t instr_pool; /* The pool of all instructions. */
  pool_t value_pool; /* The pool of all constant Malaga values. */
  pool_t src_line_pool; /* The pool of all associations
			 * between source lines and rule code. */
  pool_t var_pool; /* The pool of all variables */
  pool_t var_scope_pool; /* The pool of all variable scopes. */
  pool_t string_pool; /* The pool of all strings:
                       * variable and rule names, patterns. */
  pool_t constant_pool; /* The pool of all named constants. */
} code_t;

/* Variables. ===============================================================*/

extern code_t code;

/* Functions. ===============================================================*/

extern void init_rule_code( int_t file_type );
/* Initialise this module. 
 * CODE will contain compilation data for a file of FILE_TYPE.
 * FILE_TYPE may be ALLO_RULE_FILE, MORPHO_RULE_FILE, or SYNTAX_RULE_FILE. */

extern void terminate_rule_code( void );
/* Terminate this module. */

extern void write_code( string_t file_name );
/* Write CODE to FILE_NAME. */

/* Functions for constant folding. ==========================================*/

extern void buffer_instr( int_t opcode, int_t info );
/* Buffer the instructions BUILD_LIST, BUILD_RECORD, PUSH_SYMBOL,
 * and PUSH_CONST for constant folding. */

extern void buffer_push_number_instr( double number );
/* Buffer the instruction PUSH_CONST with NUMBER converted to a value. */

extern void buffer_push_string_instr( string_t string, string_t string_end );
/* Buffer the instruction PUSH_CONST with STRING converted to a value.
 * STRING_END points to the string end if STRING_END != NULL. */

extern void flush_buffer( void );
/* Emit the instructions that are still in the buffer. */

extern value_t get_buffer_top_value( void );
/* Test if the buffer contains a value and return the top value. */

extern value_t pop_buffer_top_value( void );
/* Pop the top value in the buffer and return it. */

extern instr_t *emit_instr( int_t opcode, int_t info );
/* Emit an instruction to the instruction pool (flushes buffer before)
 * and return the address of the instruction in the pool. */

/* End of file. =============================================================*/
