(*
 *                     The OCaml-gtk interface
 *
 * Copyright (c) 1997-99   Sven Luther
 *
 * This file is distributed under the conditions described in
 * the file LICENSE.  
 *)

(* $Id: gtkDrawing.ml,v 1.7 2000/01/24 18:32:36 sven Exp $ *)
open Gdk
open GtkObj
open GtkEasy
open GtkEasy.Layout
open Gdk.Event
open Gdk.Event.Extract

let default_log_mesg s = prerr_endline s
let log_mesg_opt = ref None
let log_mesg s = match !log_mesg_opt with
  | None -> ()
  | Some l -> l s
let log_mesg_set l = log_mesg_opt := Some l
let log_mesg_remove () = log_mesg_opt := None

(* Drawing area definition ... *)

(* Backing pixmap *)

exception No_pixmap

let get_pixmap d =
  try Gtk.Unsafe.object_get_user_data d#get_gtkobject
  with Failure _ -> raise No_pixmap

let free_pixmap d = Gtk.Unsafe.object_free_user_data d#get_gtkobject

let set_pixmap d width height depth =
  let w = Gtk.Unsafe.window_of d#get_gtkobject
  in let p = Gdk.pixmap_new w width height depth
  in let () = Gtk.Unsafe.object_set_user_data d#get_gtkobject p
  in let () =
    try ignore (Gtk.Unsafe.object_get_user_data d#get_gtkobject)
    with Failure _ -> log_mesg "set_pixmap : Error, wrote nothing ..."
  in p

(* The configure event takes place at initialization and every time
   the size of the widget changes *)
let drawingarea_configure c drawingarea gtkobject x y width height = 
  let p = set_pixmap drawingarea width height (-1)
  (* The black Graphic Context to fill the empty pixmap *)
  and black_gc = 
    let st = Gtk.Unsafe.style_of drawingarea#get_gtkobject
    in st.Gtk.black_gc
  (* We draw a black rectangle all over the backing pixmap *)
  in let d = Gdk.drawable_from_pixmap p 
  in let () = Gdk.draw_rectangle d black_gc true 0 0 width height
  in let () = c drawingarea x y width height
  in let () = log_mesg "configure"
  in true 
let empty_c drawingarea x y width height = ()

(* The expose event takes place after each configure event,
   and each time the widget becomes visible again (after having been
   hidden *)
let drawingarea_expose e drawingarea gtkobject x y width height =
  try
    let p = get_pixmap drawingarea
    (* The foreground Graphic Context to draw the backing pixmap *)
    and fg_gc =
      let st = Gtk.Unsafe.style_of drawingarea#get_gtkobject
      in st.Gtk.fg_gc.(Gtk.Unsafe.state_of drawingarea#get_gtkobject)
    (* We draw the backing pixmap on the window *)
    in let d = Gdk.drawable_from_window
      (Gtk.Unsafe.window_of drawingarea#get_gtkobject)
    in let () = Gdk.draw_pixmap d fg_gc p x y x y width height
    in let () = e drawingarea x y width height
    in let () = log_mesg "expose"
    in false
  with No_pixmap -> log_mesg "No pixmap present at expose event"; false
let empty_e drawingarea x y width height = ()

(* Button pressed event *)
let drawingarea_button_pressed b drawingarea gtkobject x y button state =
  let () = b drawingarea x y button state
  in true 
let empty_b drawingarea x y button state = ()

(* Motion notify event *)
let drawingarea_motion_notify m drawingarea gtkobject x y is_hint state =
  let () = m drawingarea x y is_hint state
  in true 
let empty_m drawingarea x y is_hint state = ()

(* Key pressed event *)
let drawingarea_key_pressed k drawingarea gtkobject keyval state =
  let () = k drawingarea keyval state
  in true
let empty_k drawingarea keyval state = ()

(*** Initializations ***)
exception Graphic_failure of string
let make_drawing_area c e b m k x y = 
  let drawingarea = drawing_area_new ()
  in let _ = drawingarea#set_events
    [
      EXPOSURE_MASK;
		LEAVE_NOTIFY_MASK;
      BUTTON_PRESS_MASK;
		POINTER_MOTION_MASK;
		POINTER_MOTION_HINT_MASK;
    ]
  in let _ = drawingarea#connect_configure
    (drawingarea_configure c drawingarea)
  in let _ = drawingarea#connect_expose
    (drawingarea_expose e drawingarea)
  in let _ = drawingarea#connect_button_press
    (drawingarea_button_pressed b drawingarea)
  in let _ = drawingarea#connect_motion_notify
    (drawingarea_motion_notify m drawingarea)
  in let _ = drawingarea#connect_key_press
    (drawingarea_key_pressed k drawingarea)
  in let _ = drawingarea#size x y
  in drawingarea
let make_drawing_area_empty x y = make_drawing_area
  empty_c empty_e empty_b empty_m empty_k

let rect_of_drawingarea drawingarea =
  let a = Gtk.Unsafe.allocation_of drawingarea#get_gtkobject
  in { x=0; y=0; width=a.Gtk.width; height=a.Gtk.height }
let clear_graph drawingarea =
  try
    let p = get_pixmap drawingarea
    in let black_gc =
      (Gtk.Unsafe.style_of drawingarea#get_gtkobject).Gtk.black_gc
    in let rect = rect_of_drawingarea drawingarea
    in let d = Gdk.drawable_from_pixmap p
    in let () = Gdk.draw_rectangle d black_gc true rect.x rect.y
      rect.width rect.height
    in Gtk.Unsafe.widget_draw drawingarea#get_gtkobject rect
  with No_pixmap -> log_mesg "No pixmap present in GtkDrawing.clear_graph"

let size_x drawingarea = 
  let rect = rect_of_drawingarea drawingarea
  in rect.width
let size_y drawingarea = 
  let rect = rect_of_drawingarea drawingarea
  in rect.height

let trans drawingarea (ox,oy) =
  let rect = rect_of_drawingarea drawingarea
  in (rect.width/2+ox, rect.height/2+oy)
let trans' drawingarea (ox,oy) =
  let rect = rect_of_drawingarea drawingarea
  in (ox-rect.width/2, oy-rect.height/2)

(*** Colors ***)
type color = Gtk.Unsafe.gtkobject -> Gdk.gc

let gc_of_rgb widget col =
  let win = Gtk.Unsafe.window_of widget
  in let gc = Gdk.gc_new win
  and cmap = Gdk.window_get_colormap win
  in let new_col = Gdk.colormap_alloc_color cmap col false true 
  in let () = Gdk.gc_set_foreground gc new_col
  in gc

let rgb r g b drawingarea =
  gc_of_rgb drawingarea
  {
    pixel=r*65536+g*256+b;
    red=r*(65535/255);
    green=g*(65535/255);
    blue=b*(65535/255);
  }

let black = rgb 0 0 0
let white = rgb 255 255 255
let red = rgb 255 0 0
let green = rgb 0 255 0
let blue = rgb 0 0 255
let yellow = rgb 255 255 0
let cyan = rgb 0 255 255
let magenta = rgb 255 0 255

let foreground = white
let background = black

(*** Drawing and Filling ***)
let plot drawingarea gc (x,y) =
  try
    (* The drawable where to draw ... *)
    let p = get_pixmap drawingarea
    (* The rectangle to draw, and later to refresh *)
    and rect = {x=x-1; y=y-1; width=3; height=3}
    (* We draw the point in the backing pixmap *)
    in let d = Gdk.drawable_from_pixmap p
    in let _ = Gdk.draw_point d (gc drawingarea#get_gtkobject) x y
    (* And ask the widget to refresh the drawn zone. *)
    in Gtk.Unsafe.widget_draw drawingarea#get_gtkobject rect
  with No_pixmap -> log_mesg "No pixmap present in plot"
let col_to_256 col = 
  {
    pixel=col.Gdk.pixel;
    red=col.Gdk.red/256;
    green=col.Gdk.green/256;
    blue=col.Gdk.blue/256;
  }
let point_color drawingarea (x,y) =
  let win = Gtk.Unsafe.window_of drawingarea#get_gtkobject
  in let cmap = window_get_colormap win
  and img = Gdk.image_get win 0 0 (x+1) (y+1)
  in let pixel = Gdk.image_get_pixel img x y
  in col_to_256 (Gdk.colormap_get_color cmap pixel)
let line drawingarea gc (x,y) (x',y') =
  try
    (* The drawable where to draw ... *)
    let p = get_pixmap drawingarea
    (* The rectangle to draw, and later to refresh *)
    and rect = {
      x = (min x x') - 1;
      y = (min y y') - 1;
      width = (abs (x'-x)) + 2;
      height = (abs (y'-y)) + 2;
    }
    (* We draw the point in the backing pixmap *)
    in let d = Gdk.drawable_from_pixmap p
    in let _ = Gdk.draw_line d (gc drawingarea#get_gtkobject) x y x' y'
    (* And ask the widget to refresh the drawn zone. *)
    in Gtk.Unsafe.widget_draw drawingarea#get_gtkobject rect
  with No_pixmap -> log_mesg "No pixmap present in line"
let draw_rect_full fill drawingarea gc x y width height =
  try
    (* The drawable where to draw ... *)
    let p = get_pixmap drawingarea
    (* The rectangle to draw, and later to refresh *)
    and rect = {
      x = x-width/2-1;
      y = y-height/2-1;
      width = width+2;
      height = height+2;
    }
    (* We draw the point in the backing pixmap *)
    in let d = Gdk.drawable_from_pixmap p
    in let _ = Gdk.draw_rectangle d (gc drawingarea#get_gtkobject)
      fill (x-width/2) (y-height/2) width height
    (* And ask the widget to refresh the drawn zone. *)
    in Gtk.Unsafe.widget_draw drawingarea#get_gtkobject rect
  with No_pixmap -> log_mesg "No pixmap present in draw_rect"
let draw_rect drawingarea = draw_rect_full false drawingarea
let fill_rect drawingarea = draw_rect_full true drawingarea 
let draw_arc_full fill drawingarea gc (x,y) rx ry a1 a2 =
  try
    (* The drawable where to draw ... *)
    let p = get_pixmap drawingarea
    (* The rectangle to draw, and later to refresh *)
    and rect = {
      x = x-rx-1;
      y = y-ry-1;
      width = 2*rx+2;
      height = 2*ry+2;
    }
    (* We draw the point in the backing pixmap *)
    in let d = Gdk.drawable_from_pixmap p
    in let _ = Gdk.draw_arc d (gc drawingarea#get_gtkobject)
      fill (x-rx) (y-ry) (2*rx) (2*ry) (a1*64) (a2*64)
    (* And ask the widget to refresh the drawn zone. *)
    in Gtk.Unsafe.widget_draw drawingarea#get_gtkobject rect
  with No_pixmap -> log_mesg "No pixmap present in draw_arc"
let draw_arc drawingarea = draw_arc_full false drawingarea 
let fill_arc drawingarea = draw_arc_full true drawingarea
let draw_ellipse d gc p rx ry = draw_arc_full false d gc p rx ry 0 360
let fill_ellipse d gc p rx ry = draw_arc_full true d gc p rx ry 0 360
let draw_circle d gc p r = draw_arc_full false d gc p r r 0 360
let fill_circle d gc p r = draw_arc_full true d gc p r r 0 360
let draw_poly_full fill drawingarea gc point_array = 
  try
    (* The drawable where to draw ... *)
    let p = get_pixmap drawingarea
    (* The rectangle to draw, and later to refresh *)
    in let rec get_boundaries pa n =
      if n = 0 then 0, 0, 0, 0 else
        let xmin, xmax, ymin, ymax = get_boundaries pa (n-1)
        and x, y = Array.get pa (n-1)
	in min x xmin, max x xmax, min y ymin, max y ymax
    in let xmin, xmax, ymin, ymax =
      get_boundaries point_array (Array.length point_array) 
    in let rect = {
      x = xmin-1;
      y = ymin-1;
      width = xmax-xmin+2;
      height = ymax-ymin+2;
    }
    (* We draw the point in the backing pixmap *)
    in let d = Gdk.drawable_from_pixmap p
    in let _ = Gdk.draw_polygon d (gc drawingarea#get_gtkobject)
      fill point_array
    (* And ask the widget to refresh the drawn zone. *)
    in Gtk.Unsafe.widget_draw drawingarea#get_gtkobject rect
  with No_pixmap -> log_mesg "No pixmap present in draw_poly"
let draw_poly drawingarea = draw_poly_full false drawingarea 
let fill_poly drawingarea = draw_poly_full true drawingarea

(*** Text drawing ***)
let default_font = Gdk.font_load "-*-*-*-*-*--*-*-*-*-*-*-iso8859-1"
let text_size font s = string_width font s, string_height font s
let draw_string drawingarea font gc (x,y) s =
  try
    (* The drawable where to draw ... *)
    let p = get_pixmap drawingarea
	 and ext = Gdk.string_extents font s
    (* The rectangle to draw, and later to refresh *)
    in let rect = {
      x = x-ext.lbearing-1;
      y = y-ext.ascent-1;
      width = x+ext.rbearing+1;
      height = y+ext.descent+1;
    }
    (* We draw the point in the backing pixmap *)
    in let d = Gdk.drawable_from_pixmap p
    in let () = Gdk.draw_string d font (gc drawingarea#get_gtkobject)
      x y s
    (* And ask the widget to refresh the drawn zone. *)
    in Gtk.Unsafe.widget_draw drawingarea#get_gtkobject rect
  with No_pixmap -> log_mesg "No pixmap present in draw_string"
let draw_char drawingarea font gc (x,y) c =
  let s = String.make 1 c
  in draw_string drawingarea font gc (x,y) s
let set_font gc font =
  let () = gc_set_font gc font
  in gc

(*** Images ***)
type image = Gdk.image
(* TODO ... *)
let transp = black
(* TODO ... FALSE ... *)
let make_image a = ()
(* TODO ... *)
let dump_image img = [| [||] |]
let draw_image drawingarea img x y = ()
(*
  try
    let p = get_pixmap drawingarea
    (* The foreground Graphic Context to draw the backing pixmap *)
    and fg_gc =
      let st = Gtk.Unsafe.style_of drawingarea#get_gtkobject
      in st.Gtk.fg_gc.(Gtk.Unsafe.state_of drawingarea#get_gtkobject)
      (* We draw the backing pixmap on the window *)
    in let d = Gdk.drawable_from_window
      (Gtk.Unsafe.window_of drawingarea#get_gtkobject)
    in let () = Gdk.draw_pixmap d fg_gc p x y x y width height
    in false
  with No_pixmap -> log_mesg "No pixmap present in draw_image"; false
*)

(* TODO ... FALSE ... *)
let get_image x y w h = ()
(* TODO ... FALSE ... *)
let create_image w h = ()
(* TODO ... *)
let blit_image i x y = ()

(*** Mouse and keyboard events *)
type status =
{
  mouse_x : int;
  mouse_y : int;
  button : bool;
  button_number : int;
  keypressed : bool;
  key : char
}
type event =
  | Button_down
  | Button_up
  | Key_pressed
  | Mouse_motion
  | Poll
(* TODO ... *)
let wait_next_event l =
{
  mouse_x=0;
  mouse_y=0;
  button=false;
  button_number=0;
  keypressed=false;
  key=' ';
}

(*** Mouse and keyboard polling ***)
let mouse_pos () = 
  let e = wait_next_event [Poll]
  in (e.mouse_x, e.mouse_y)
let button_down () = 
  let e = wait_next_event [Poll]
  in e.button
let button_number () =
  let e = wait_next_event [Poll]
  in e.button_number
let read_key () =
  let e = wait_next_event [Key_pressed]
  in e.key
let key_pressed () =
  let e = wait_next_event [Poll]
  in e.keypressed

(*** Sound ***)
let sound freq dur = ()

module Compatibility = struct
  (*** Initializations ***)
  exception Graphic_failure of string
  exception Graphics_unitialized
  let drawingarea = ref None
  (* TODO ... *)
  let open_graph s = drawingarea :=
    Some (make_drawing_area empty_c empty_e empty_b empty_m empty_k 300 200)
  (* TODO ... *)
  let close_graph () = ()
  let clear_graph () = match !drawingarea with
    | None -> raise Graphics_unitialized
    | Some d -> clear_graph d
  let size_x () = match !drawingarea with
    | None -> raise Graphics_unitialized
    | Some d -> size_x d
  let size_y () = match !drawingarea with
    | None -> raise Graphics_unitialized
    | Some d -> size_y d

  (*** Colors ***)
  type color = Gtk.Unsafe.gtkobject -> Gdk.gc

  let black = black
  let white = white
  let red = red
  let green = green
  let blue = blue
  let yellow = yellow
  let cyan = cyan
  let magenta = magenta

  let foreground = foreground
  let background = background

  let rgb r g b = match !drawingarea with
    | None -> raise Graphics_unitialized
    | Some d -> rgb r g b

  let cur_color = ref foreground
  let set_color col = cur_color := col

  (*** Point and line drawing ***)
  let cur_p = ref (0,0)
  let plot x y = match !drawingarea with
    | None -> raise Graphics_unitialized
    | Some d -> cur_p := (x,y); plot d !cur_color (x,y)
  let point_color x y = white
  let moveto x y = cur_p := (x,y)
  let current_point () = !cur_p
  let lineto x y = match !drawingarea with
    | None -> raise Graphics_unitialized
    | Some d -> line d !cur_color !cur_p (x,y); cur_p := (x,y)
  let draw_arc x y rx ry a1 a2 = match !drawingarea with
    | None -> raise Graphics_unitialized
    | Some d -> draw_arc d !cur_color (x,y) rx ry a1 a2
  let draw_ellipse x y rx ry = match !drawingarea with
    | None -> raise Graphics_unitialized
    | Some d -> draw_ellipse d !cur_color (x,y) rx ry
  let draw_circle x y r = match !drawingarea with
    | None -> raise Graphics_unitialized
    | Some d -> draw_circle d !cur_color (x,y) r
  let set_line_width w = 
    cur_color := function drawingarea ->
      let gc = !cur_color drawingarea
      in let () = Gdk.gc_set_line_attributes gc w
        Gdk.Style.LINE_SOLID Gdk.Style.CAP_NOT_LAST Gdk.Style.JOIN_MITER
      in gc

  (*** Text drawing ***)
  let cur_font = ref default_font
  let draw_char c = match !drawingarea with
    | None -> raise Graphics_unitialized
    | Some d -> draw_char d !cur_font !cur_color !cur_p c
  let draw_string s = match !drawingarea with
    | None -> raise Graphics_unitialized
    | Some d -> draw_string d !cur_font !cur_color !cur_p s
  let set_font s =
    let font = Gdk.font_load s
	 in let () = cur_font := font
	 in cur_color :=
      function d -> set_font (!cur_color d) font
  let set_text_size size =
    raise (Failure "Not even implemented in ocaml's Graphic module")
  let text_size s = text_size !cur_font s

  (*** Filling ***)
  let fill_rect x y w h =  match !drawingarea with
    | None -> raise Graphics_unitialized
    | Some d -> fill_rect d !cur_color x y w h
  let fill_poly poly =  match !drawingarea with
    | None -> raise Graphics_unitialized
    | Some d -> fill_poly d !cur_color poly
  let fill_arc x y rx ry a1 a2 =  match !drawingarea with
    | None -> raise Graphics_unitialized
    | Some d -> fill_arc d !cur_color (x,y) rx ry a1 a2
  let fill_ellipse x y rx ry =  match !drawingarea with
    | None -> raise Graphics_unitialized
    | Some d -> fill_ellipse d !cur_color (x,y) rx ry
  let fill_circle x y r =  match !drawingarea with
    | None -> raise Graphics_unitialized
    | Some d -> fill_circle d !cur_color (x,y) r

  (*** Images ***)
  (*
  type image = Gdk.image
  *)
  type image = Not_done
  (* TODO ... *)
  let transp = black
  (* TODO ... *)
  let make_image a = Not_done
  (* TODO ... *)
  let dump_image i = [| [||] |]
  (* TODO ... *)
  let draw_image i x y = ()
  (* TODO ... *)
  let get_image x y w h = Not_done
  (* TODO ... *)
  let create_image w h = Not_done
  (* TODO ... *)
  let blit_image i x y = ()

  (*** Mouse and keyboard events *)
  type status =
  {
    mouse_x : int;
    mouse_y : int;
    button : bool;
    button_number : int;
    keypressed : bool;
    key : char
  }
  type event =
    | Button_down
    | Button_up
    | Key_pressed
    | Mouse_motion
    | Poll
  let wait_next_event = wait_next_event

  (*** Mouse and keyboard polling ***)
  let mouse_pos = mouse_pos
  let button_down = button_down
  let button_number = button_number
  let read_key = read_key
  let key_pressed = key_pressed

  (*** Sound ***)
  let sound = sound
end
