/***************************************************************************
 Mutella - A commandline/HTTP client for the Gnutella filesharing network.

 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2
 of the License, or (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 property.cpp  -  Property classes for holding preferences in.
                             -------------------
    begin                : Sat Jul 7 2001
    copyright            : (C) 2001 by 
    email                : maksik@gmx.co.uk
 ***************************************************************************/
 
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "mutella.h"
#include "property.h"
#include "inifile.h"
#include "conversions.h"
#include "common.h"

bool MProperty::SetFromStr(char * buf)
{
	if (PT_STRING!=GetPropType())
		MakeLower(buf);
	buf = StripWhite(buf);

	if (IsSet())
	{
		return GetSet()->SetFromStr(buf);
	}
	
	int n;
	double d;
	IP ip;
	IPNET ipn;
	switch (GetPropType())
	{
		case PT_BOOL:
			if ( 0==strcmp(buf,"true") || 0==strcmp(buf,"1") )
			{
				SetBoolValue(true);
				return true;
			}
			else if ( 0==strcmp(buf,"false") || 0==strcmp(buf,"0") )
			{
				SetBoolValue(false);
				return true;
			}
			break;
		case PT_INT:
			if (asc2num(buf, &n))
			{
				SetIntValue(n);
				return true;
			}
			break;
		case PT_DWORD:
			if (asc2num(buf, &n))
			{
				SetDwordValue(n);
				return true;
			}
			break;
		case PT_DOUBLE:
			if (asc2num(buf, &d))
			{
				SetDoubleValue(d);
				return true;
			}
			break;
		case PT_STRING:
			// TODO: parse ""
			// workaround to set an empty string
			if (0==strcmp("\"\"",buf) || 0==strcmp("\'\'",buf))
				SetStringValue("");
			else
				SetStringValue(buf);
			return true;
		case PT_IP:
			if (Str2Ip(buf, ip))
			{
				SetIpValue(ip);
				return true;
			}
			break;
		case PT_IPNET:
			if (Str2IpNet(buf, ipn))
			{
				SetIpnetValue(ipn);
				return true;
			}
			break;
	}
	return false;
}

CString MProperty::Format()
{
	if (IsSet())
	{
		return GetSet()->Format();
	}
	const char* t;
	IP ip;
	IPNET ipn;
	CString sTmp;
	switch (GetPropType()){
		case PT_BOOL:   return GetBoolValue() ? "true" : "false";
		case PT_INT:    return sTmp.format("%d", GetIntValue());
		case PT_DWORD:  return sTmp.format("%u", GetDwordValue());
		case PT_DOUBLE: return sTmp.format("%.4g", GetDoubleValue());
		case PT_STRING: return GetStringValue();
		case PT_IP:
			ip = GetIpValue();
			return sTmp.format("%d.%d.%d.%d", ip.S_ip.a, ip.S_ip.b, ip.S_ip.c, ip.S_ip.d);
		case PT_IPNET:
			ipn = GetIpnetValue();
			if (ipn.net_bits != 32)
				return sTmp.format("%d.%d.%d.%d/%d", ipn.S_ip.a, ipn.S_ip.b, ipn.S_ip.c, ipn.S_ip.d, ipn.net_bits);
			return sTmp.format("%d.%d.%d.%d", ipn.S_ip.a, ipn.S_ip.b, ipn.S_ip.c, ipn.S_ip.d); // will look just as IP
	}
	ASSERT(0);
	return "unsupported_type";
}

LPCSTR MProperty::FormatPropType()
{
	switch (GetPropType())
	{
		case PT_BOOL : return "bool";
		case PT_INT : return "int";
		case PT_DWORD : return "uint";
		case PT_DOUBLE : return "real";
		case PT_STRING : return "string";
		case PT_IP : return "ip";
		case PT_IPNET : return "net";
	}
	return "none";
}

bool MProperty::Read(MIniFile* pIniFile)
{
	// somewhat lasy way
	// TODO: test if it always works
	char tmp[1024]; // TODO: check if it sufficient
	if (pIniFile->ReadStr(name, tmp, "", true)) // do trim
	{
		if (strlen(tmp) || PT_STRING==GetPropType())
			return SetFromStr(tmp);
		if (PT_STRING==GetPropType())
			return true; // its OK when string is set to ""
	}
	return false;
}

bool MProperty::Write(MIniFile* pIniFile)
{
	/*switch (GetPropType()){
		case PT_BOOL:   return pIniFile->WriteBool(name, GetBoolValue());
		case PT_INT:    return pIniFile->WriteNum(name, (long)GetIntValue());
		case PT_DWORD:  return pIniFile->WriteNum(name, GetDwordValue());
		case PT_DOUBLE: return pIniFile->WriteNum(name, GetDoubleValue());
		case PT_STRING: return pIniFile->WriteStr(name, GetStringValue());
		case PT_IP:     return pIniFile->WriteStr(name, Ip2Str(GetIpValue()).c_str());
		case PT_IPNET:  return pIniFile->WriteStr(name, IpNet2Str(GetIpnetValue()).c_str());
	}
	return false;*/
	return pIniFile->WriteStr(name, Format().c_str());
}

////////////////////////////////////////////////////////////////////////////////////////
// partial specialisation of TPropertySet<CString>
template<>
bool TPropertySet<CString>::SetFromStr(const CString& str)
{
	if (m_pMutexSet)
		m_pMutexSet->lock();
	// search for '{' than locate next '}' and pass
	// what is in between to SetFromStr...
	int nPosStart = 0;
	int nPosOpen, nPosClose;
	bool bResult = false; // be optimistic, return true if any succedded
	m_set.clear();
	while ( (nPosOpen = str.find('{', nPosStart)) >= 0  &&
			(nPosClose = str.find('}',nPosOpen+1)) >= 0  )
	{
		m_set.insert(str.substr(nPosOpen+1, nPosClose-nPosOpen-1));
		bResult = true;
		nPosStart = nPosClose+1;
	}
	if (m_pMutexSet)
		m_pMutexSet->unlock();
	return bResult;
}

template<>
CString TPropertySet<CString>::Format()
{
	if (m_pMutexSet)
		m_pMutexSet->lock();
	CString sTmp;
	for (iterator it=m_set.begin(); it!=m_set.end(); ++it)
	{
		sTmp += "{" + *it + "},";
	}
	if (sTmp.size())
		sTmp.cut(sTmp.size()-1); // remove the trailing ','
	if (m_pMutexSet)
		m_pMutexSet->unlock();
	return sTmp;
}

template<>
void TPropertySet<CString>::CopyToStrSet(set<CString>& setStr)
{
	if (m_pMutexSet)
		m_pMutexSet->lock();
	
	setStr = m_set;
	
	if (m_pMutexSet)
		m_pMutexSet->unlock();
}

template<>
bool TPropertySet<CString>::InsertStr(const CString& str)
{
	Insert(str);
	return true;
}

template<>
bool TPropertySet<CString>::RemoveStr(const CString& str)
{
	return Remove(str);
}

template<>
bool TPropertySet<CString>::IsInStr(const CString& str)
{
	return IsIn(str);
}

// partial specification for IP and NET
template<>
template<>
bool TPropertySet<IPNET>::IsIn(const IP& ip)
{
	if (m_pMutexSet)
		m_pMutexSet->lock();
	IPNET ipn;
	ipn.S_addr = ip.S_addr;
	ipn.net_bits = 32;
	iterator it = m_set.find(ipn);
	bool bFound = it != m_set.end();
	if (!bFound)
	{
		// we still have means for it...
		it = m_set.upper_bound(ipn); //Finds the first element whose key greater than ...
		// the way the > < operators defined the net is always greater than the single IP
		if (it != m_set.end())
			bFound = IsIpInSubnet(ip, *it);
	}
	if (m_pMutexSet)
		m_pMutexSet->unlock();
	return bFound;
}

// for IPNETs we have to compact the list after insertion...
template<>
void TPropertySet<IPNET>::Insert(const IPNET& item)
{
	if (m_pMutexSet)
		m_pMutexSet->lock();
	iterator it = m_set.upper_bound(item); //Finds the first element whose key greater than ...
	if (it != m_set.end() && IsSubnetInSubnet(item, *it))
	{
		if (m_pMutexSet)
			m_pMutexSet->unlock();
		return;
	}
	iterator itIn = m_set.insert(it, item);
	if (itIn != m_set.end() && itIn != m_set.begin())
	{
		it = itIn;
		--it;
		while (it != m_set.end() && itIn != m_set.begin() && IsSubnetInSubnet(*it, item))
		{
			m_set.erase(it);
			it = itIn;
			--it;
		}
	}
	if (m_pMutexSet)
		m_pMutexSet->unlock();
}

template<>
void TPropertySet<IPNET>::Compact()
{
	if (m_set.size() <= 1)
		return;
	reverse_iterator itCurr = m_set.rbegin();
	reverse_iterator itPrev = itCurr;
	++itPrev;
	while (itPrev != m_set.rend() && itCurr != m_set.rend())
	{
		if (IsSubnetInSubnet(*itPrev, *itCurr))
		{
			m_set.erase(*itPrev);
			itPrev = itCurr;
		}
		else
			++itCurr;
		++itPrev;
	}
}

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
//
// MPropertyContainer implementation

MPropertyContainer::MPropertyContainer()
{
	m_pCurSec = NULL;
}

MPropertyContainer::~MPropertyContainer()
{
	sec_iterator its;
	for (its = m_secs.begin(); its != m_secs.end(); its++)
	{
		delete its->second;
	}
	prop_iterator it;
	for (it = m_props.begin(); it != m_props.end(); it++)
	{
		delete it->second;
	}
}

bool MPropertyContainer::DoMigration(LPCSTR szOldVersion, LPCSTR szCurrentVersion, MIniFile* pInifile)
{
	return false; // nothing was modified
}

bool MPropertyContainer::Read(LPCSTR szName, LPCSTR szCurrentVersion /*=NULL*/)
{
	CString path = ExpandPath(szName);
	MIniFile inifile;
	if (!inifile.LoadFile(path.c_str()))
		return false;
	bool bOK = true;
	// version control
	if (szCurrentVersion)
	{
		char tmp[1024];
		inifile.SetSection("Version");
		if (!inifile.ReadStr("CurrentVersion", tmp, "", true))
			bOK = false;
		if (strcmp(tmp, szCurrentVersion)<0)
			if (DoMigration(tmp, szCurrentVersion, &inifile))
			{
				inifile.FlushFile();
			}
	}
	//
	map<CString, MProperty*>map_copy = m_props;
	for (sec_iterator its = sec_begin();its!=sec_end();its++)
	{
		MPropertySection* pS = its->second;
		inifile.SetSection(pS->m_sName.c_str());
		for (iterator it = pS->begin();it!=pS->end();it++)
		{
			if (!it->second->IsPersistent())
				map_copy.erase(it->first);
			else if(it->second->Read(&inifile))
				map_copy.erase(it->first);
		}
	}
	inifile.ResetFile();
	for (iterator it = map_copy.begin();it!=map_copy.end();it++)
	{
		if (it->second->Read(&inifile))
			map_copy.erase(it->first);
	}
	inifile.CloseFile();
	return map_copy.empty();
}

bool MPropertyContainer::Write(LPCSTR szName, LPCSTR szCurrentVersion /*=NULL*/)
{
	CString path = ExpandPath(szName);
	MIniFile inifile(INI_FxALLOWxCREATE);
	if (!inifile.LoadFile(path.c_str()))
		return false;
	bool bOK = true;
	// version control
	if (szCurrentVersion)
	{
		inifile.SetSection("Version");
		if (!inifile.WriteStr("CurrentVersion", szCurrentVersion))
			bOK = false;
	}
	//
	map<CString, MProperty*>map_copy = m_props;
	for (sec_iterator its = sec_begin();its!=sec_end();its++)
	{
		MPropertySection* pS = its->second;
		for (iterator it = pS->begin();it!=pS->end();it++)
			map_copy.erase(it->first);
	}
	//
	for (iterator it = map_copy.begin();it!=map_copy.end();it++)
	{
		if (it->second->IsPersistent() && !it->second->Write(&inifile))
			bOK = false;
	}
	for (sec_iterator its = sec_begin();its!=sec_end();its++)
	{
		MPropertySection* pS = its->second;
		inifile.SetSection(pS->m_sName.c_str());
		for (iterator it = pS->begin();it!=pS->end();it++)
		{
			if (it->second->IsPersistent() && !it->second->Write(&inifile))
				bOK = false;
		}
	}
	inifile.FlushFile();
	inifile.CloseFile();
	return bOK;
}

int MPropertyContainer::Transfer(MPropertyContainer* pPC)
{
	int nCountAdded = 0;
	std::queue<CString> toErase;
	map<CString, MProperty*>map_copy = pPC->m_props;
	for (sec_iterator its = pPC->sec_begin();its!=pPC->sec_end();its++)
	{
		MPropertySection* pS = its->second;
		AddSection(pS->m_sName.c_str());
		map_copy.erase(pS->m_sName);
		for (iterator it = pS->begin();it!=pS->end();it++)
		{
			if (!FindProperty(it->first.c_str()))
			{
				m_props[it->first] = it->second;
				if (m_pCurSec)
					m_pCurSec->m_props[it->first] = it->second;
				toErase.push(it->first);
				nCountAdded++;
			}
		}
	}
	SetCurrentSection(NULL);
	for (iterator it = map_copy.begin();it!=map_copy.end();it++)
	{
		if (!FindProperty(it->first.c_str()))
		{
			m_props[it->first] = it->second;
			toErase.push(it->first);
			nCountAdded++;
		}
	}
	while(toErase.size())
	{
		pPC->m_props.erase(toErase.front());
		toErase.pop();
	}
	return nCountAdded;
}
	
MProperty* MPropertyContainer::AddNewProperty(LPCSTR szName, int nType, int nStrSize)
{
	if (FindProperty(szName))
		return NULL;
	MProperty* pP = NULL;
	switch (nType) {
		case PT_BOOL: pP = new MPropBool(szName);
			break;
		case PT_INT: pP = new MPropInt(szName);
			break;
		case PT_DOUBLE: pP = new MPropDbl(szName);
			break;
		case PT_STRING: pP = new MPropString(szName, "", nStrSize);
			break;
	}
	ASSERT(pP);
	if (pP)
	{
		m_props[szName] = pP;
		if (m_pCurSec)
			m_pCurSec->m_props[szName] = pP;
	}
	return pP;
}

MProperty* MPropertyContainer::AddSet(LPCSTR szName, MPropertySet* pSet)
{
	if (FindProperty(szName))
		return false;
	MProperty* pP = new TProperty<MPropertySet,PT_SET>(szName, pSet);
	ASSERT(pP);
	m_props[szName] = pP;
	if (m_pCurSec)
		m_pCurSec->m_props[szName] = pP;
	return pP;
}

MProperty* MPropertyContainer::AddSet(LPCSTR szName, MPropertySet* pSet, LPCSTR szSetInit)
{
	if (FindProperty(szName))
		return false;
	MProperty* pP = new TProperty<MPropertySet,PT_SET>(szName, pSet);
	ASSERT(pP);
	m_props[szName] = pP;
	if (m_pCurSec)
		m_pCurSec->m_props[szName] = pP;
	pSet->SetFromStr(szSetInit);
	return pP;
}

MProperty* MPropertyContainer::AddProperty(LPCSTR szName, bool * pb)
{
	if (FindProperty(szName))
		return NULL;
	MProperty* pP = new MPropBool(szName, pb);
	ASSERT(pP);
	m_props[szName] = pP;
	if (m_pCurSec)
		m_pCurSec->m_props[szName] = pP;
    return pP;
}

MProperty* MPropertyContainer::AddProperty(LPCSTR szName, bool * pb, bool bInit)
{
	if (FindProperty(szName))
		return NULL;
    MProperty* pP = new MPropBool(szName, pb, bInit);
    ASSERT(pP);
    m_props[szName] = pP;
    if (m_pCurSec)
		m_pCurSec->m_props[szName] = pP;
    return pP;
}

MProperty* MPropertyContainer::AddProperty(LPCSTR szName, int * pn)
{
	if (FindProperty(szName))
		return NULL;
    MProperty* pP = new MPropInt(szName, pn);
    ASSERT(pP);
    m_props[szName] = pP;
    if (m_pCurSec)
		m_pCurSec->m_props[szName] = pP;
    return pP;
}

MProperty* MPropertyContainer::AddProperty(LPCSTR szName, int * pn, int nInit)
{
	if (FindProperty(szName))
		return NULL;
    MProperty* pP = new MPropInt(szName, pn, nInit);
    ASSERT(pP);
    m_props[szName] = pP;
    if (m_pCurSec)
		m_pCurSec->m_props[szName] = pP;
    return pP;
}

MProperty* MPropertyContainer::AddProperty(LPCSTR szName, DWORD * pdw)
{
	if (FindProperty(szName))
		return NULL;
    MProperty* pP = new MPropDword(szName, pdw);
    ASSERT(pP);
    m_props[szName] = pP;
    if (m_pCurSec)
		m_pCurSec->m_props[szName] = pP;
    return pP;
}

MProperty* MPropertyContainer::AddProperty(LPCSTR szName, DWORD * pdw, DWORD dwInit)
{
	if (FindProperty(szName))
		return NULL;
    MProperty* pP = new MPropDword(szName, pdw, dwInit);
    ASSERT(pP);
    m_props[szName] = pP;
    if (m_pCurSec)
		m_pCurSec->m_props[szName] = pP;
    return pP;
}

MProperty* MPropertyContainer::AddProperty(LPCSTR szName, double * pd)
{
	if (FindProperty(szName))
		return NULL;
    MProperty* pP = new MPropDbl(szName, pd);
    ASSERT(pP);
    m_props[szName] = pP;
    if (m_pCurSec)
		m_pCurSec->m_props[szName] = pP;
    return pP;
}

MProperty* MPropertyContainer::AddProperty(LPCSTR szName, double * pd, double dInit)
{
	if (FindProperty(szName))
		return NULL;
    MProperty* pP = new MPropDbl(szName, pd, dInit);
    ASSERT(pP);
    m_props[szName] = pP;
    if (m_pCurSec)
		m_pCurSec->m_props[szName] = pP;
    return pP;
}

MProperty* MPropertyContainer::AddProperty(LPCSTR szName, IP * pip)
{
	if (FindProperty(szName))
		return NULL;
    MProperty* pP = new MPropIP(szName, pip);
    ASSERT(pP);
    m_props[szName] = pP;
    if (m_pCurSec)
		m_pCurSec->m_props[szName] = pP;
    return pP;
}

MProperty* MPropertyContainer::AddProperty(LPCSTR szName, IP * pip, const IP& ipInit)
{
	if (FindProperty(szName))
		return NULL;
    MProperty* pP = new MPropIP(szName, pip, ipInit);
    ASSERT(pP);
    m_props[szName] = pP;
    if (m_pCurSec)
		m_pCurSec->m_props[szName] = pP;
    return pP;
}

MProperty* MPropertyContainer::AddProperty(LPCSTR szName, IPNET * pnet)
{
	if (FindProperty(szName))
		return NULL;
    MProperty* pP = new MPropIPNET(szName, pnet);
    ASSERT(pP);
    m_props[szName] = pP;
    if (m_pCurSec)
		m_pCurSec->m_props[szName] = pP;
    return pP;
}

MProperty* MPropertyContainer::AddProperty(LPCSTR szName, IPNET * pnet, const IPNET& netInit)
{
	if (FindProperty(szName))
		return NULL;
    MProperty* pP = new MPropIPNET(szName, pnet, netInit);
    ASSERT(pP);
    m_props[szName] = pP;
    if (m_pCurSec)
		m_pCurSec->m_props[szName] = pP;
    return pP;
}

MProperty* MPropertyContainer::AddProperty(LPCSTR szName, LPSTR pstr, int nStrSize)
{
	if (FindProperty(szName))
		return NULL;
    MProperty* pP = new MPropString(szName, pstr, nStrSize);
    ASSERT(pP);
    m_props[szName] = pP;
    if (m_pCurSec)
		m_pCurSec->m_props[szName] = pP;
    return pP;
}

MProperty* MPropertyContainer::AddProperty(LPCSTR szName, LPSTR pstr, int nStrSize, LPCSTR szInit)
{
	if (FindProperty(szName))
		return NULL;
    MProperty* pP = new MPropString(szName, pstr, nStrSize, szInit);
    ASSERT(pP);
    m_props[szName] = pP;
    if (m_pCurSec)
		m_pCurSec->m_props[szName] = pP;
    return pP;
}

MProperty* MPropertyContainer::FindProperty(LPCSTR szName)
{
	prop_iterator it = m_props.find(szName);
	if (it != m_props.end())
		return it->second;
	return NULL;
}

bool MPropertyContainer::AddSection(LPCSTR szName)
{
	if (!szName || !strlen(szName))
		return false;
	//
	sec_iterator its = m_secs.find(szName);
	if (its==m_secs.end())
	{
		MPropertySection* pSec = new MPropertySection;
		ASSERT(pSec);
		pSec->m_sName = szName;
		m_secs[szName] = pSec;
		m_pCurSec = pSec;
	}
	else
		m_pCurSec = its->second;
	return true;
}

bool MPropertyContainer::SetCurrentSection(LPCSTR szName)
{
	if (!szName || !strlen(szName))
	{
		m_pCurSec = NULL;
		return true;
	}
	//
	sec_iterator its = m_secs.find(szName);
	if (its==m_secs.end())
		return false;
	m_pCurSec = its->second;
	return true;
}

MPropertySection* MPropertyContainer::FindSection(LPCSTR szName)
{
	sec_iterator it = m_secs.find(szName);
	if (it != m_secs.end())
		return it->second;
	return NULL;
}

