/***************************************************************************
 Mutella - A commandline/HTTP client for the Gnutella filesharing network.

 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2
 of the License, or (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 common.cpp  -  Functions for cross-platform compat and general utility.

 the original version of this file was taken from Gnucleus (http://gnucleus.sourceforge.net)

    begin                : Tue May 29 2001
    copyright            : (C) 2001 by
    email                : maksik@gmx.co.uk
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#if TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif

#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include "mutella.h"
#include "basicstruct.h"
#include "common.h"
#include "conversions.h"
#include "xsleep.h"

//////////////////////////
// mutex-protected time()
// not anymore
time_t xtime()
{
	//MLock lock(g_LibcMutex);
	//return time(NULL);
	time_t curr;
	time(&curr);
	return curr;
}

extern "C"
int safe_nanosleep(const struct timespec *rqtp, struct timespec *rmtp)
{
#ifdef HAVE_NANOSLEEP
	// Generally located in <time.h>.
	// Note that Solaris has a granularity of 10ms.
	return nanosleep(rqtp,rmtp);
#else
	struct timeval enterTime, nowTime;
	struct timezone aZone;
	struct timeval tv, origtv;
	int result;

	// Note that this is, unlike usleep(), MT-safe. 
	// - Select has random granularity.
	// - Select *will* wake early in heavy signal environments.
	// - Select *generally* uses per-thread signalling as impl.
	// However, for those with no choice, this will do.

	TIMESPEC_TO_TIMEVAL(&origtv, rqtp);
	TIMESPEC_TO_TIMEVAL(&tv, rqtp);

	gettimeofday( &enterTime, &aZone );
	result = select(0, (fd_set *)NULL, (fd_set *)NULL, (fd_set *)NULL, &tv);
	// Now we're going to try to make it look like a nanosleep.
	if (result==0) return 0;  // Time limit reached.
	else
	{
		if (errno==EINTR)
		{
			// Let's see if rmtp has changed.
			if ((tv.tv_sec == origtv.tv_sec)&&
				(tv.tv_usec == tv.tv_usec))
			{
				long nsecs;
				long orignsecs;
				// Solaris, MacOS X, non-linux behavior.
				// Use our timestamps to generate elapsed time.
				gettimeofday(&nowTime, &aZone);

				// Extract our tvs into long nsecs.
				nsecs = ( nowTime.tv_sec - enterTime.tv_sec ) * 1000000;
                     		nsecs += nowTime.tv_usec - enterTime.tv_usec;
				orignsecs = origtv.tv_sec * 1000000;
				orignsecs += origtv.tv_usec;

				// From that, calculate time remaining.
				tv.tv_sec =  nsecs / 1000000;
				orignsecs = orignsecs - (tv.tv_sec * 1000000);
				nsecs = nsecs - (tv.tv_sec * 1000000);
				tv.tv_usec = orignsecs - nsecs;
			}
			if (rmtp!=NULL) TIMEVAL_TO_TIMESPEC(&tv, rmtp);	
			// rmtp now contains the time elapsed.
			return -1;
		} 
		else 	// For any non-EINTR error value,
		{
			return -1;
		}
	}
#endif
}

/////////////////////////////////////////////////////////////////////////////////
// generate the unique numeric ID -- just count and hope that int never gets over
MMutex mutex_id;
DWORD g_currentID = 1;
//
DWORD GenID()
{
	ASSERT(g_currentID);
	MLock lock(mutex_id);
	return ++g_currentID;
}

bool operator<(const IP& ip1, const IP& ip2){
	return htonl(ip1.S_addr)<htonl(ip2.S_addr);
}

bool operator>(const IP& ip1, const IP& ip2){
	return htonl(ip1.S_addr)>htonl(ip2.S_addr);
}

////////////////////////////////////////
// defines if IP belongs to the sub-net
bool IsIpInSubnet(const IP& ip, const IPNET& subnet){
	u_long mask = htonl(GetSubnetMask(subnet));
	return (ip.S_addr&mask)==(subnet.S_addr&mask);
}

bool IsSubnetInSubnet(const IPNET& subnet1, const IPNET& subnet2){
	u_long mask = htonl(GetSubnetMask(min(subnet1.net_bits, subnet2.net_bits)));
	return (subnet1.S_addr&mask)==(subnet2.S_addr&mask);
}

bool operator==(const IPNET& ipn1, const IPNET& ipn2){
	if (ipn1.net_bits!=ipn2.net_bits)
		return false;
	u_long net_bits = min(ipn1.net_bits, ipn2.net_bits);
	u_long mask = htonl(GetSubnetMask(net_bits));
	return (ipn1.S_addr&mask)==(ipn2.S_addr&mask);
}

bool operator<(const IPNET& ipn1, const IPNET& ipn2){
	u_long net_bits = min(ipn1.net_bits, ipn2.net_bits);
	u_long mask = htonl(GetSubnetMask(net_bits));
	if (htonl(ipn1.S_addr&mask)<htonl(ipn2.S_addr&mask))
		return true;
	if (htonl(ipn1.S_addr&mask)>htonl(ipn2.S_addr&mask))
		return false;
	return ipn1.net_bits>ipn2.net_bits; // the net is smaller if the subnet number is greater
}

bool operator>(const IPNET& ipn1, const IPNET& ipn2){
	u_long net_bits = min(ipn1.net_bits, ipn2.net_bits);
	u_long mask = htonl(GetSubnetMask(net_bits));
	if (htonl(ipn1.S_addr&mask)>htonl(ipn2.S_addr&mask))
		return true;
	if (htonl(ipn1.S_addr&mask)<htonl(ipn2.S_addr&mask))
		return false;
	return ipn1.net_bits<ipn2.net_bits; // the net is smaller if the subnet number is greater
}

////////////////////////////////////////////
// GUID support functions
void GUID::Create(GUID* pGuid)
{
	for(int n=0; n<16; ++n)
		pGuid->a[n]= (u_char)(0xFF & rand());// hacky, isnt it?
}

const GUID GUID::G_NULL = {{0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0}};

///////////////////////////////////////
// Some string manipulations
void MakeLower(char * szStr)
{
	for (int n = strlen(szStr)-1; n>=0; --n)
	{
		if (szStr[n]>='A' && szStr[n]<='Z')
			szStr[n]+= 'a' - 'A';
	}
}

void MakeUpper(char * szStr)
{
	for (int n = strlen(szStr)-1; n>=0; --n)
	{
		if (szStr[n]>='a' && szStr[n]<='z')
			szStr[n]-= 'a' - 'A';
	}
}

int CompareNoCase(LPCSTR sz1, LPCSTR sz2, int nChars /*=INT_MAX*/)
{
	// sanity check
	if (!sz1)
		if (!sz2)
			return 0;
		else
			return -1;
	if (!sz2)
		return 1;
	//
	register char cDiff;
	for (int i=0; i<nChars; ++i, ++sz1, ++sz2)
	{
		if (!*sz1)
			if (!*sz2)
				return 0;
			else
				return -1;
		if (!*sz2)
			return 1;
		cDiff = ToLower(*sz1) - ToLower(*sz2);
		if (cDiff)
			return (cDiff > 0) ? 1 : -1;
	}
	return 0;
}

int CompareNoCase(const CString& s1, const CString& s2, int nChars /*=INT_MAX*/)
{
	return CompareNoCase(s1.c_str(), s2.c_str(), nChars);
}

char* SubstringNoCase(LPSTR szHaystack, LPCSTR szNeedle)
{
	char * szHaystackLow = strdup (szHaystack);
	char * szNeedleLow = strdup (szNeedle);
	char * szRetval;

	MakeLower(szHaystackLow);
	MakeLower(szNeedleLow);

	szRetval = strstr (szHaystack, szNeedle);

	free(szNeedleLow);
	free(szHaystackLow);

	if (szRetval == NULL)
		return (NULL);

	return szHaystack + (int) (szRetval - szHaystackLow);
}

inline bool IsWhite(char c)
{
	return isspace(c);
}

char* StripWhite( char* string )
{
	register char *s, *t;
	for (s = string; IsWhite(*s); s++)
		;
	if (*s == 0)
		return (s);
	
	t = s + strlen (s) - 1;
	while (t > s && IsWhite(*t))
		t--;
	*++t = '\0';
	
	return s;
}

CString StripWhite(const CString& str)
{
	int s,t;
	int l = str.length();
	for (s = 0; s<l && IsWhite(str[s]); s++)
		;
	if (s == l)
		return "";
	
	for (t = l-1; t>=s && IsWhite(str[t]); t--)
		;
	if (t<s)
		return "";
	return str.substr(s,t-s+1);
}

CString StripAnyOf(const CString& str, const char * szCharSet)
{
	int s,t;
	int l = str.length();
	for (s = 0; s<l && strchr(szCharSet, str[s]); s++)
		;
	if (s == l)
		return "";
	
	for (t = l-1; t>=s && strchr(szCharSet, str[t]); t--)
		;
	if (t<s)
		return "";
	return str.substr(s,t-s+1);
}


void ReplaceSubStr(CString& str, const CString& search, const CString& replace)
{
	int nPos = 0;
	do
	{
		nPos = str.find(search,nPos);
		if (nPos>=0)
		{
			// for the case when search.length()==replace.length() it could be done in more efficient way
			str = str.substr(0,nPos) + replace + str.substr(nPos+search.length());
			nPos += replace.length();
		}
	}
	while (nPos >= 0);
}

//////////////////////////////////////////
// extract number from the acsii string
//   handles various suffixes like K or Mb
bool asc2num( char * arg, double * pRet )
{
	char* szClean = StripWhite(arg);
	int neFactor = 0;
	int len = strlen(szClean);
	if ( 0 == len )
		return false;
	char pat[256];
	memset(pat,0,256);
	int j = 1;
	for (int i=1;i<255;i++)
		if (NULL==strchr("0123456789.bBkKmMgGeE-+",(char)i))
			if (strchr(szClean, (char)i))
				return false;
	// it still can be cheated with some --++10+2Ke15
	switch (szClean[len-1])
	{
		case 'b':
		case 'B':
			szClean[len-1]='\0';
			--len;
	}
	if (len<=0)
		return false;
	switch (szClean[len-1])
	{
		case 'g':
		case 'G':
			neFactor += 10;
		case 'm':
		case 'M':
			neFactor += 10;
		case 'k':
		case 'K':
			neFactor += 10;
			szClean[len-1]='\0';
			--len;
	}
	if (len<=0)
		return false;
	*pRet = atof(szClean)*(1<<neFactor);
	//*pRet = strtod(szClean, NULL)*(1<<neFactor);
	return true;
}

bool asc2num( char * arg, int * pRet )
{
	double d;
	if (asc2num(arg,&d))
	{
		if (d > 0)
			*pRet = int(0.5+d);
		else
			*pRet = int(-0.5+d);
		return true;
	}
	return false;
}

bool asc2num( const CString& sArg, double * pRet )
{
	char * szTmp = (char*) alloca(sArg.size()+1);
	ASSERT(szTmp);
	strcpy(szTmp, sArg.c_str());
	return asc2num(szTmp, pRet);
}

bool asc2num( const CString& sArg, int * pRet )
{
	char * szTmp = (char*) alloca(sArg.size()+1);
	ASSERT(szTmp);
	strcpy(szTmp, sArg.c_str());
	return asc2num(szTmp, pRet);
}

//////////////////////////////////
// few file manipulation functions
bool CreateDirectory(const CString& dir)
{
	return 0==mkdir(ExpandPath(dir).c_str(), S_IRWXU);
}

bool MoveFile(const CString& old_path, const CString& new_path)
{
	return 0==rename(ExpandPath(old_path).c_str(), ExpandPath(new_path).c_str());
}

bool DeleteFile(const CString& path)
{
	return 0==unlink(ExpandPath(path).c_str());
}

bool FileExists(const CString& path)
{
	struct stat st;
	return 0==stat(ExpandPath(path).c_str(), &st) && (S_ISREG(st.st_mode) || S_ISDIR(st.st_mode));
}

////////////////////////////////
// Expands ~ into value of $HOME
CString ExpandPath(CString path)
{
	MLock lock(g_LibcMutex);
	if (path.find("~/") != -1)
	{
		CString home = getenv("HOME");
		if (home.length()>1)
		{
			home += "/";
			ReplaceSubStr(path,"~/",home);
		}
	}
	ReplaceSubStr(path,"//","/");
	return path;
}

////////////////////////////////////////////////////////////
// Query matching -- mainly used by MGnuSearch and MGnuShare
bool QueryMatch(CString Result, const CString& Query)
{
	vector<char*> QWords;
	
	Result.make_lower();

    int nQLen = Query.length();
    char* szQuery = new char[nQLen+1];
    ASSERT(szQuery);
    strcpy(szQuery, Query.c_str());
    MakeLower(szQuery);
    // Break Query into words
    MakeWordList(szQuery, QWords);
    bool bMatch = false;
    if (QWords.size())
    	bMatch = MatchWordList(Result, QWords);
	delete [] szQuery;
	return bMatch;
}

void MakeWordList(LPSTR szQuery, vector<char*>& QWords)
{
	int nQLen = strlen(szQuery);
	// Break Query into words
	char* szWord;
	int i;
	char c;
	for (i = 0; i<nQLen; ++i)
	{
		c = szQuery[i];
		if ( isspace(c) || c=='*' || c=='+' )
			szQuery[i] = '\0';
	}
	for (i = 0; i<nQLen; ++i)
	{
		if ( szQuery[i] && (i==0 || !szQuery[i-1]) )
		{
			szWord = StripWhite(szQuery+i);
			if (strlen(szWord))
			{
				QWords.push_back(szWord);
			}
		}
	}
}

void MakeWordList(LPSTR szQuery, vector<char*>& QWordsInc, vector<char*>& QWordsExc)
{
	int nQLen = strlen(szQuery);
	// Break Query into words
	char* szWord;
	int i;
	char c;
	for (i = 0; i<nQLen; ++i)
	{
		c = szQuery[i];
		if ( isspace(c) || c=='*' || c=='+' )
			szQuery[i] = '\0';
	}
	for (i = 0; i<nQLen; ++i)
	{
		if ( szQuery[i] && (i==0 || !szQuery[i-1]) )
		{
			szWord = StripWhite(szQuery+i);
			if (strlen(szWord))
			{
				// if szWord starts with '-' -- this is an exclusive word
				if (szWord[0]=='-')
				{
					if (szWord[1] != '\0')
						QWordsExc.push_back(szWord+1);
				}
				else
					QWordsInc.push_back(szWord);
			}
		}
	}
}

void MakeWordList(LPSTR szQuery, set<CString>& QWords)
{
	int nQLen = strlen(szQuery);
	// Break Query into words
	char* szWord;
	int i;
	char c;
	for (i = 0; i<nQLen; ++i)
	{
		c = szQuery[i];
		if ( isspace(c) || c=='*' || c=='+' )
			szQuery[i] = '\0';
	}
	for (i = 0; i<nQLen; ++i)
	{
		if ( szQuery[i] && (i==0 || !szQuery[i-1]) )
		{
			szWord = StripWhite(szQuery+i);
			if (strlen(szWord))
			{
				QWords.insert(szWord);
			}
		}
	}
}

bool MatchWordList(const CString& ResultLower, const set<CString>& QWords, bool bMatchAll /*=true*/ )
{
	if (!QWords.size())
		return false;
	set<CString>::const_iterator itWord;
	// Search result for the those words
	for(itWord = QWords.begin(); itWord != QWords.end(); ++itWord)
	{
		if( ResultLower.find(*itWord) != -1 )
		{
			if (!bMatchAll) // means match-any
				return true;
		}
		else
		{
			if (bMatchAll)
				return false;
		}
	}
	return bMatchAll;
}

bool MatchWordList(const CString& ResultLower, const vector<char*>& QWords, bool bMatchAll /*=true*/)
{
	if (!QWords.size())
		return false;
	vector<char*>::const_iterator itWord;
	// Search result for the those words
	for(itWord = QWords.begin(); itWord != QWords.end(); itWord++)
	{
		if( ResultLower.find(*itWord) != -1 )
		{
			if (!bMatchAll) // means match-any
				return true;
		}
		else
		{
			if (bMatchAll)
				return false;
		}
	}
	return bMatchAll;
}

//////////////////////////////////////////////////////
// Used by MGnuDownload to generate automatic searches
CString MakeSearchOfFilename(const CString& Name)
{
	int nLen = Name.length();
	int bWhite = false;
	CString Search;
	char c;
	for (int i = 0; i<nLen; i++)
	{
		c = Name[i];
		if ( (c>='0' && c<='9') ||
		     (c>='A' && c<='Z') ||
		     (c>='a' && c<='z') )
		{
			if (bWhite)
			{
				Search+=' ';
				bWhite = false;
			}
			// make it lower
			if (c>='A' && c<='Z')
				c += 'a'-'A';
			Search+=c;
		}
		else
			bWhite = true;
	}
	// remove some redundant 'keywords'
	ReplaceSubStr(Search,"divx"," ");
	ReplaceSubStr(Search,"divx4"," ");
	ReplaceSubStr(Search,"mpeg4"," ");
	ReplaceSubStr(Search,"dvdrip"," ");
	ReplaceSubStr(Search," rip "," ");
	ReplaceSubStr(Search," en "," ");
	ReplaceSubStr(Search,"dvd"," ");
	ReplaceSubStr(Search,"vhs"," ");
	ReplaceSubStr(Search,"vhsrip"," ");
	ReplaceSubStr(Search,"dvdivx"," ");
	ReplaceSubStr(Search," cherami "," ");
	ReplaceSubStr(Search,"dominion"," ");
	ReplaceSubStr(Search,"smr"," ");
	ReplaceSubStr(Search,"ts pdivx"," ");//TS-PDivX
	ReplaceSubStr(Search," ts "," ");
	ReplaceSubStr(Search,"humr "," ");
	ReplaceSubStr(Search,"schizo"," ");
	ReplaceSubStr(Search," sharereactor"," ");
	ReplaceSubStr(Search," vite"," ");
	ReplaceSubStr(Search," qix"," ");
	ReplaceSubStr(Search," cd1"," 1");
	ReplaceSubStr(Search," cd2"," 2");
	ReplaceSubStr(Search," cd3"," 3");
	ReplaceSubStr(Search,"1of1"," ");
	ReplaceSubStr(Search," 1of2"," 1");
	ReplaceSubStr(Search," 2of2"," 2");
	ReplaceSubStr(Search," 1 of 2"," 1");
	ReplaceSubStr(Search," 2 of 2"," 2");
	ReplaceSubStr(Search," part1"," 1");
	ReplaceSubStr(Search," part2"," 2");
	ReplaceSubStr(Search," part3"," 3");
	ReplaceSubStr(Search," part 1"," 1");
	ReplaceSubStr(Search," part 2"," 2");
	ReplaceSubStr(Search," part 3"," 3");
	//
	ReplaceSubStr(Search,"   "," ");
	ReplaceSubStr(Search,"  "," ");
	// remove starting and trailing spaces
	return StripWhite(Search);
}


CString FormatSize(u_long n)
{
	char tmp[16]; // overkill
	if ( n < 1e3 )
		sprintf(tmp,"%d", n);
	else if ( n < 1u<<10 )
		sprintf(tmp,"%.2gK", n/1024.0);
	else if ( n < 1e6 )
		sprintf(tmp,"%.3gK", n/1024.0);
	else if ( n < 1u<<20 )
		sprintf(tmp,"%.2gM", n/1048576.0);
	else if ( n < 1e9 )
		sprintf(tmp,"%.3gM", n/1048576.0);
	else if ( n < 1u<<30 )
		sprintf(tmp,"%.2gG", n/1073741824.0);
	else
		sprintf(tmp,"%.3gG", n/1073741824.0);
	return tmp;
}

CString FormatSizeLL(long long n)
{
	char tmp[16]; // overkill
	if ( n < 1e3 )
		sprintf(tmp,"%d", n);
	else if ( n < 1ll<<10 )
		sprintf(tmp,"%.2gK", n/1024.0);
	else if ( n < 1e6 )
		sprintf(tmp,"%.3gK", n/1024.0);
	else if ( n < 1ll<<20 )
		sprintf(tmp,"%.2gM", n/1048576.0);
	else if ( n < 1e9 )
		sprintf(tmp,"%.3gM", n/1048576.0);
	else if ( n < 1ll<<30 )
		sprintf(tmp,"%.2gG", n/1073741824.0);
	else if ( n < 1e12 )
		sprintf(tmp,"%.3gG", n/1073741824.0);
	else if ( n < 1ll<<40 )
		sprintf(tmp,"%.2gT", n/1.0995116278e+12);
	else
		sprintf(tmp,"%.3gT", n/1.0995116278e+12);
	return tmp;
}

CString FormatKSize(u_long n)
{
	char tmp[16]; // overkill
	if ( n < 1000u )
		sprintf(tmp,"%.3gK", n*1.0);
	else if ( n < 1u<<10 )
		sprintf(tmp,"%.2gM", n/1024.0);
	else if ( n < 1000000u )
		sprintf(tmp,"%.3gM", n/1024.0);
	else if ( n < 1u<<20 )
		sprintf(tmp,"%.2gG", n/1048576.0);
	else if ( n < 1000000000u )
		sprintf(tmp,"%.3gG", n/1048576.0);
	else
		sprintf(tmp,"%.3gT", n/1073741824.0);
	return tmp;
}

CString FormatKSizeLL(long long n)
{
	char tmp[16]; // overkill
	if ( n < 1e3 )
		sprintf(tmp,"%.3gK", n*1.0);
	else if ( n < 1ll<<10 )
		sprintf(tmp,"%.2gM", n/1024.0);
	else if ( n < 1e6 )
		sprintf(tmp,"%.3gM", n/1024.0);
	else if ( n < 1ll<<20 )
		sprintf(tmp,"%.2gG", n/1048576.0);
	else if ( n < 1e9 )
		sprintf(tmp,"%.3gG", n/1048576.0);
	else if ( n < 1ll<<30 )
		sprintf(tmp,"%.2gT", n/1073741824.0);
	else
		sprintf(tmp,"%.3gT", n/1073741824.0);
	return tmp;
}

CString FormatMSize(u_long n)
{
	char tmp[16]; // overkill
	if ( n < 1000u )
		sprintf(tmp,"%.3gM", n*1.0);
	else if ( n < 1u<<10 )
		sprintf(tmp,"%.2gG", n/1024.0);
	else if ( n < 1000000u )
		sprintf(tmp,"%.3gG", n/1024.0);
	else if ( n < 1u<<10 )
		sprintf(tmp,"%.2gT", n/1048576.0);
	else
		sprintf(tmp,"%.3gT", n/1048576.0);
	return tmp;
}

CString FormatNumber(double d)
{
	char tmp[16]; // overkill
	if ( d < 1e3 )
		sprintf(tmp,"%.3g", d);
	else if ( d < 1e6 )
		sprintf(tmp,"%.3gK", d*1e-3);
	else if ( d < 1e9 )
		sprintf(tmp,"%.3gM", d*1e-6);
	else
		sprintf(tmp,"%.3gG", d*1e-9);
	return tmp;
}

CString FormatPercent(double part, double whole)
{
	double d = 100.0*part/whole;
	
	char tmp[16]; // overkill
	if (d>=1.0)
		sprintf(tmp,"%.3g%%",d);
	else if (d>=0.1)
		sprintf(tmp,"%.2g%%",d);
	else if (d>=0.01)
		sprintf(tmp,"%.1g%%",d);
	else
		sprintf(tmp,"0.00%%");
	return tmp;
}

CString FormatTime(time_t nSec)
{
	int nMin = nSec/60;
	int nHour = nMin/60;
	int nDay = nHour/24;
	CString sTmp;
	if (nDay) sTmp.format("%dd%dh", nDay, nHour%24);
	else if (nHour) sTmp.format("%dh%dm", nHour, nMin%60);
	else if (nMin) sTmp.format("%dm%ds", nMin, nSec%60);
	else sTmp.format("%ds", nSec%60);
	return sTmp;
}

CString FormatTimeFull(time_t nSec)
{
	int nMin = nSec/60;
	int nHour = nMin/60;
	int nDay = nHour/24;
	CString sTmp;
	if (nDay) sTmp.format("%dd%dh%dm%ds", nDay, nHour%24, nMin%60, nSec%60);
	else if (nHour) sTmp.format("%dh%dm%ds", nHour, nMin%60, nSec%60);
	else if (nMin) sTmp.format("%dm%ds", nMin, nSec%60);
	else sTmp.format("%ds", nSec%60);
	return sTmp;
}

CString FormatAbsTime(time_t nSec)
{
	MLock lock(g_LibcMutex);
	struct tm * pTime = localtime(&nSec);
	if (!pTime)
		return "--:--:--";
	time_t currTime;
	time(&currTime);
	time_t deltaSec = currTime - nSec; // time diff
	time_t deltaMin = deltaSec/60;
	time_t deltaHour = deltaMin/60;
	time_t deltaDay = deltaHour/24;
	time_t deltaWeek = deltaDay/7;
	static char szTmp[256]; // we are protcted by the mutex, it's OK to use static vars
	if (deltaWeek==0)
		if (deltaDay==0)
			strftime(szTmp, 255, "%H:%M:%S", pTime); // hh:mm:ss
		else
			strftime(szTmp, 255, "%a %H:%M", pTime); // abbr.day+hh:mm
	else
		strftime(szTmp, 255, "%a %b %d", pTime);     // abbr.day abbr.month day
	return szTmp;
}

CString FormatAbsTimeFull(time_t nSec)
{
	MLock lock(g_LibcMutex);
	struct tm * pTime = localtime(&nSec);
	if (!pTime)
		return "--:--:--";
	static char szTmp[256]; // we are protcted by the mutex, it's OK to use static vars
	strftime(szTmp, 255, "%d.%m.%y %H:%M:%S", pTime); // dd.mm.yy hh:mm:ss
	return szTmp;
}

time_t ParseAbsTimeFull(const CString& sTime)
{
	time_t dummy_time = xtime();
	MLock lock(g_LibcMutex);
	struct tm time;
	memcpy(&time, localtime(&dummy_time), sizeof(time));
	//static char szTmp[256]; // we are protcted by the mutex, it's OK to use static vars
	//strftime(szTmp, 255, "%d.%m.%y %H:%M:%S", &time); // dd.mm.yy hh:mm:ss
	//memset(&time, 0, sizeof(time));
	sscanf(sTime.c_str(),"%ld.%ld.%ld %ld:%ld:%ld", &time.tm_mday,&time.tm_mon,&time.tm_year,&time.tm_hour,&time.tm_min,&time.tm_sec);
	//sscanf(szTmp,"%ld.%ld.%ld %ld:%ld:%ld", &time.tm_mday,&time.tm_mon,&time.tm_year,&time.tm_hour,&time.tm_min,&time.tm_sec);
	if (time.tm_year < 70)
		time.tm_year += 100;
	time.tm_mon -= 1; // its strange, but localtime numbers months from 0
	//time.tm_isdst = 1;
	time_t tmp_time = mktime(&time);
	return tmp_time;
}

CString InsertElypsis(const CString& s, int nMaxLen)
{
	if (s.length()>nMaxLen && nMaxLen>4)
	{
		return s.substr(0,nMaxLen/2-2) + "..." + s.substr(s.length()-(nMaxLen - nMaxLen/2)+1);
	}
	return s;
}

int split_str(const CString& str, char cSeparator,  list<CString>& parts)
{
	int nStart=0;
	int nPos;
	while ((nPos=str.find(cSeparator, nStart))>=0){
		parts.push_back(str.substr(nStart, nPos-nStart));
		nStart = nPos + 1;
	}
	parts.push_back(str.substr(nStart));
	return parts.size();
}

int split_str(const CString& str, const char* lpszSeparator,  list<CString>& parts)
{
	int nSepLen = strlen(lpszSeparator);
	if (!nSepLen)
		return 0;
	int nStart=0;
	int nPos;
	while ((nPos=str.find(lpszSeparator, nStart))>=0){
		parts.push_back(str.substr(nStart, nPos-nStart));
		nStart = nPos + nSepLen;
	}
	parts.push_back(str.substr(nStart));
	return parts.size();
}

bool parse_params(const CString& sBuffer, const char* lpszPairsSep, const char* lpszEqualSep,  map<CString, CString>& values)
{
	list<CString> pairs;
	if (0==split_str(sBuffer, lpszPairsSep, pairs))
		return false;
	for(list<CString>::iterator it=pairs.begin(); it!=pairs.end(); ++it){
		list<CString> pair;
		list<CString>::iterator it1;
		switch (split_str(*it, lpszEqualSep,pair)){
			case 0: break;
			case 1: values[*pair.begin()] = "";
				break;
			case 2:
				it1 = pair.begin();
				++it1;
				values[*pair.begin()] = *it1;
				break;
			default:
				TRACE("parse_params: split returned something strange");
		}
	}
	return true;
}

CString find_header(CString Name, const CString& sHahdshake, const CString& sLowHahdshake, LPCSTR szDefault /*=NULL*/)
{
	Name.make_lower();
	int keyPos = sLowHahdshake.find(Name);
	if (keyPos >= 0)
	{
		keyPos += Name.length();
		while (sLowHahdshake[keyPos] == ' ')
			++keyPos;
		if (sLowHahdshake[keyPos] == ':')
		{
			++keyPos;
			return StripWhite( sHahdshake.substr(keyPos, sHahdshake.find("\r\n", keyPos) - keyPos) );
		}
	}
	return szDefault ? szDefault : "";
}

#define CONV_HEX_DIGIT(_d) \
	if (_d>='0'&&_d<='9')\
		_d -= '0';\
	else if (_d>='a'&&_d<='f')\
		_d -= 'a'-10;\
	else if (_d>='A'&&_d<='F')\
		_d -= 'A'-10;

CString restore_string(CString o, bool bFormMode /*=true*/)
{
	int nPos=0;
	if (bFormMode)
	{
		// change '+' to space
		while ((nPos=o.find('+',nPos))>=0)
			o[nPos++]=' ';
	}
	
	// recover %xx hex values
	nPos = 0;
	while ((nPos=o.find('%',nPos))>=0 && nPos < o.length()-2)
	{
		u_char c1 = o[nPos+1],
			   c2 = o[nPos+2];
		if ( ((c1>='0'&&c1<='9')||(c1>='a'&&c1<='f')||(c1>='A'&&c1<='F')) &&
			 ((c2>='0'&&c2<='9')||(c2>='a'&&c2<='f')||(c2>='A'&&c2<='F')) )
		{
			o = o.substr(0,nPos+1) + o.substr(nPos+3);
			CONV_HEX_DIGIT(c1)
			CONV_HEX_DIGIT(c2)
			o[nPos] = (c1<<4) | c2;
		}
		++nPos;
	}
	return o;
}

