/***************************************************************************
 Mutella - A commandline/HTTP client for the Gnutella filesharing network.

 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2
 of the License, or (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 gnuhash.cpp  -  Definition of a hashtable for node management

 the original version of this file was taken from Gnucleus (http://gnucleus.sourceforge.net)

    begin                : Tue May 29 2001
    copyright            : (C) 2001 by
    email                : maksik@gmx.co.uk
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "mutella.h"
#include "basicstruct.h"
#include "gnuhash.h"
#include "common.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction

MGnuHash::MGnuHash()
{
	m_nHashEntries = 0;

	for(int i = 0; i < HASH_SIZE; i++)
	{
		m_Table[i] = NULL;
	}
}

MGnuHash::~MGnuHash()
{
	for(int i = 0; i < HASH_SIZE; i++)
	{
		if(m_Table[i] != NULL)
			delete m_Table[i];
	}
}

void MGnuHash::Insert(const GUID* pGuid, DWORD dwOriginID)
{
	// lock access
	MLock lock(m_mutex);
	//
	DWORD key = CreateKey(pGuid);
	DWORD dwOldest = 0;
	time_t timeOldest = xtime();

	for(int nAttempts = 0; nAttempts < MAX_REHASH; nAttempts++)
	{
		// if it is NULL we are done
		if (m_Table[key] == NULL)
		{
			VERIFY(m_Table[key] = new key_Value(*pGuid, dwOriginID, xtime()));
			m_nHashEntries++;
			return;
		}
		// track the oldest
		if (m_Table[key]->time < timeOldest)
		{
			timeOldest = m_Table[key]->time;
			dwOldest = key;
		}
		// and create a next key
		key = (key + REHASH_VALUE) % HASH_SIZE;
	}
    // the table is full we have to replace the oldest key with this one
    delete  m_Table[dwOldest];
    VERIFY(m_Table[dwOldest] = new key_Value(*pGuid, dwOriginID, xtime()));
	//TRACE3("hash table replaced the key which was ",time(NULL)-timeOldest, " seconds older");
	// note that number of entries doesnt change
}


key_Value* MGnuHash::FindValue(const GUID *pGuid)
{
	// lock access
	MLock lock(m_mutex);
	//
	DWORD key = CreateKey(pGuid);
	// Search the current table
	for(int nAttempts = 0; nAttempts < MAX_REHASH; nAttempts++)
	{
		if(m_Table[key] != NULL && CompareGuid(&m_Table[key]->Guid, pGuid))
			return m_Table[key];
		key = (key + REHASH_VALUE) % HASH_SIZE;
	}
	// not found ...
	return NULL;
}


DWORD MGnuHash::CreateKey(const GUID *pGuid)
{
	BYTE* pGuidRaw = (BYTE*) pGuid;

	// XOR every 4 BYTEs together ...
	DWORD key =  (pGuidRaw[0] ^ pGuidRaw[4] ^ pGuidRaw[8] ^ pGuidRaw[12]) +
				((pGuidRaw[1] ^ pGuidRaw[5] ^ pGuidRaw[9] ^ pGuidRaw[13])  << 8) +
				((pGuidRaw[2] ^ pGuidRaw[6] ^ pGuidRaw[10] ^ pGuidRaw[14]) << 16) +
				((pGuidRaw[3] ^ pGuidRaw[7] ^ pGuidRaw[11] ^ pGuidRaw[15]) << 24);

	// And modulo it down to size
	key %= HASH_SIZE;
	return key;
}


bool  MGnuHash::CompareGuid(const GUID* pGuid1, const GUID* pGuid2)
{
	return 0==memcmp(pGuid1,pGuid2,16);
}


void  MGnuHash::ClearUpTable(int nSeconds)
{
	time_t timeLimit = xtime()-nSeconds;
	for(int i = 0; i < HASH_SIZE; i++)
		if(m_Table[i] != NULL && m_Table[i]->time < timeLimit)
		{
			delete m_Table[i];
			m_Table[i] = NULL;
			m_nHashEntries--;
		}
}

