(* $Id: ocamldap.ml4,v 1.13 2004/09/09 23:39:23 eric Exp $ *)
#include "config.h"

type conn
type msgid

(* kerberos support *)
#ifdef HAVE_LDAP_KERBEROS_BIND_S
type auth_method = [`SIMPLE | `KRBV41 | `KRBV42]
#else
type auth_method = [`SIMPLE ];;
#endif

type error_code = [
    `SUCCESS
  | `OPERATIONS_ERROR
  | `PROTOCOL_ERROR
  | `TIMELIMIT_EXCEEDED
  | `SIZELIMIT_EXCEEDED
  | `COMPARE_FALSE
  | `COMPARE_TRUE
  | `AUTH_METHOD_NOT_SUPPORTED
  | `STRONG_AUTH_NOT_SUPPORTED
  | `STRONG_AUTH_REQUIRED
  | `PARTIAL_RESULTS
  | `REFERRAL
  | `ADMINLIMIT_EXCEEDED
  | `UNAVAILABLE_CRITICAL_EXTENSION
  | `CONFIDENTIALITY_REQUIRED
  | `SASL_BIND_IN_PROGRESS
  | `NO_SUCH_ATTRIBUTE
  | `UNDEFINED_TYPE
  | `INAPPROPRIATE_MATCHING
  | `CONSTRAINT_VIOLATION
  | `TYPE_OR_VALUE_EXISTS
  | `INVALID_SYNTAX
  | `NO_SUCH_OBJECT
  | `ALIAS_PROBLEM
  | `INVALID_DN_SYNTAX
  | `IS_LEAF
  | `ALIAS_DEREF_PROBLEM
  | `INAPPROPRIATE_AUTH
  | `INVALID_CREDENTIALS
  | `INSUFFICIENT_ACCESS
  | `BUSY
  | `UNAVAILABLE
  | `UNWILLING_TO_PERFORM
  | `LOOP_DETECT
  | `NAMING_VIOLATION
  | `OBJECT_CLASS_VIOLATION
  | `NOT_ALLOWED_ON_NONLEAF
  | `NOT_ALLOWED_ON_RDN
  | `ALREADY_EXISTS
  | `NO_OBJECT_CLASS_MODS
  | `RESULTS_TOO_LARGE
  | `AFFECTS_MULTIPLE_DSAS
  | `OTHER
  | `SERVER_DOWN
  | `LOCAL_ERROR
  | `ENCODING_ERROR
  | `DECODING_ERROR
  | `TIMEOUT
  | `AUTH_UNKNOWN
  | `FILTER_ERROR
  | `USER_CANCELLED
  | `PARAM_ERROR
  | `NO_MEMORY
  | `CONNECT_ERROR
  | `NOT_SUPPORTED
  | `CONTROL_NOT_FOUND
  | `NO_RESULTS_RETURNED
  | `MORE_RESULTS_TO_RETURN
  | `CLIENT_LOOP
  | `REFERRAL_LIMIT_EXCEEDED
]

type mod_op = [`ADD | `DELETE | `REPLACE]
type search_scope = [`BASE | `ONELEVEL | `SUBTREE]
type attr = { attr_name: string; attr_values: string array }
type entry = { entry_dn: string; entry_attrs: attr list }
type modattr = mod_op * string * string list
type result = entry list

exception LDAP_Failure of error_code

(* ldap c-interface functions *)
external add_s_c: 
  conn -> string -> (mod_op * string * string array) array -> unit = "ocamldap_add_s"
external bind_s_c: conn -> string -> string -> auth_method -> unit = "ocamldap_bind_s"
external delete_s_c: conn -> string -> unit = "ocamldap_delete_s"
external err2string_c: error_code -> string = "err2string"
external init_c: string -> int -> int -> conn = "ocamldap_init"

(* kerberos support *)
#ifdef HAVE_LDAP_KERBEROS_BIND_S
external kerberos_bind_s_c: conn -> string -> unit = "ocamldap_kerberos_bind_s"
#endif

#ifdef HAVE_LDAP_KERBEROS_BIND1_S
external kerberos_bind1_s_c: conn -> string -> unit = "ocamldap_kerberos_bind1_s"
#endif

#ifdef HAVE_LDAP_KERBEROS_BIND2_S
external kerberos_bind2_s_c: conn -> string -> unit = "ocamldap_kerberos_bind2_s";;
#endif

external modify_s_c: 
  conn -> string -> (mod_op * string * string array) array -> unit = "ocamldap_modify_s"
external modrdn_s_c: conn -> string -> string -> unit = "ocamldap_modrdn_s"
external modrdn2_s_c: conn -> string -> string -> bool -> unit = "ocamldap_modrdn2_s"
external search_c: conn -> string -> search_scope -> string -> string array -> int -> msgid =
  "ocamldap_search_bytecode" "ocamldap_search_native"
external get_search_entry_c: conn -> msgid -> entry = "get_search_entry"
external simple_bind_s_c: conn -> string -> string -> unit = "ocamldap_simple_bind_s"
external unbind_c: conn -> unit = "ocamldap_unbind"

(* ldap ocaml-interface functions *)
let modattr_to_oparr (x,y,z) = (x, y, Array.of_list z)

let add_s c ~dn ~attr =
  add_s_c c dn (Array.of_list (List.map modattr_to_oparr attr))

let bind_s ?(who = "") ?(cred = "") ?(auth_method = `SIMPLE) c =
  bind_s_c c who cred auth_method

let delete_s c ~dn =
  delete_s_c c dn

let err2string err =
  err2string_c err

let unbind c =
  unbind_c c

let init ?(version = 3) ?(port = 389) host =
  if ((version < 2) || (version > 3)) then
    (* don't allow an invalid protocal version to make it past *)
    raise (LDAP_Failure(`LOCAL_ERROR))
  else
    init_c host port version

(* kerberos support *)
#ifdef HAVE_LDAP_KERBEROS_BIND_S
let kerberos_bind_s c ~who =
  kerberos_bind_s_c c who
#endif

#ifdef HAVE_LDAP_KERBEROS_BIND1_S
let kerberos_bind1_s c ~who =
  kerberos_bind1_s_c c who
#endif

#ifdef HAVE_LDAP_KERBEROS_BIND2_S
let kerberos_bind2_s c ~who =
  kerberos_bind2_s_c c who
#endif

let modify_s c ~dn ~mods =
  modify_s_c c dn (Array.of_list (List.map modattr_to_oparr mods))

let modrdn_s c ~dn ~newdn =
  modrdn_s_c c dn newdn

let modrdn2_s c ~dn ~newdn ~deleteoldrdn =
  modrdn2_s_c c dn newdn deleteoldrdn

let print_entry ?(channel = stdout) e =
  let p = Printf.fprintf channel in
  p "dn: %s\n" e.entry_dn;
  let print_attr a =
    let print_val a v = Printf.fprintf channel "%s: %s\n" a v in
    let pv = print_val a.attr_name in
      Array.iter pv a.attr_values
  in
    List.iter print_attr e.entry_attrs;
    output_char channel '\n'

let get_search_entry = get_search_entry_c

let search_s ?(base = "") ?(scope = `BASE) ?(attrs = []) ?(attrsonly = false) c filter =
  let msgid = search_c c base scope filter 
		(Array.of_list attrs) 
		(match attrsonly with false -> 0 | true -> 1)
  in
  let result = ref [] in   
    (try while true
     do
       result := (get_search_entry c msgid) :: !result
     done
     with 
	 LDAP_Failure `SUCCESS 
       | LDAP_Failure `LOCAL_ERROR -> ());
    !result

let search ?(base = "") ?(scope = `BASE) ?(attrs = []) ?(attrsonly = false) c filter =
  search_c c base scope filter 
    (Array.of_list attrs) 
    (match attrsonly with false -> 0 | true -> 1)

let simple_bind_s ?(who = "") ?(password = "") c =
  simple_bind_s_c c who password

(* library initialization *)
let _ =
  Callback.register_exception "LDAP_Failure" (LDAP_Failure(`SUCCESS))
