/*
 * archive-stream.cc --
 *
 *      Archive stream object
 *
 * Copyright (c) 1997-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * @(#) $Header: /usr/mash/src/repository/mash/mash-1/archive/archive-stream.cc,v 1.16 2002/02/03 03:09:26 lim Exp $
 */

#include "archive/archive-stream.h"
#include "archive/timeval.h"
#include "misc/mtrace.h"
#include "misc/nethost.h"

#ifndef WIN32
#include <unistd.h>
#include <pwd.h>
#include <sys/time.h>
#else
#include <time.h>
#endif

#include <sys/types.h>
#include <sys/types.h>
#include <timer.h>


#ifdef MTRACE
int ArchiveStream::count_ = 0;
#endif


/*
 * FIXME:
 *    logical time of 0.0 is assumed to represent non-std conditions
 *    In PlaybackStream::NextEvent(), it is used to indicate that there are
 *    no more events to schedule
 *    In PlaybackStream::Clip(), it is used to indicate a non-existent
 *    start or end of clip (i.e. if start=0.0 then start from beginning,
 *    if end=0.0 then end at end-of-file)
 */


/*
 * <otcl> Class ArchiveStream
 * FIXME base class for recording/playing media streams to/from a file
 * on a per-source basis.
 */
DEFINE_OTCL_CLASS(ArchiveStream, "ArchiveStream") {
	INSTPROC(datafile);
	INSTPROC(indexfile);
}


/*
 * <otcl> Class ArchiveStream/Play
 * FIXME base class for playing media streams from a file
 * on a per-source basis.
 */
DEFINE_ABSTRACT_OTCL_CLASS(PlaybackStream, "ArchiveStream/Play") {
	INSTPROC(lts);
	INSTPROC(clip);
}


/*
 * <otcl> Class ArchiveStream/Record
 * FIXME base class for recording media streams to a file
 * on a per-source basis.
 */
DEFINE_OTCL_CLASS(RecordStream, "ArchiveStream/Record") {
}


int
ArchiveStream::datafile(int argc, const char * const *argv)
{
	Tcl &tcl = Tcl::instance();
	if (argc==2) {
		DataFile *file = DataFile_();
		if (file==NULL || file->name()==NULL)
			tcl.resultf("");
		else
			tcl.resultf("%s", file->name());
		return TCL_OK;
	}
	else {
		TclObject *file;
		BEGIN_PARSE_ARGS(argc, argv);
		ARG(file);
		END_PARSE_ARGS;
		DataFile_((DataFile*)file);
		return TCL_OK;
	}
}


int
ArchiveStream::indexfile(int argc, const char * const *argv)
{
	Tcl &tcl = Tcl::instance();
	if (argc==2) {
		IndexFile *file = IndexFile_();
		if (file==NULL || file->name()==NULL)
			tcl.resultf("");
		else
			tcl.resultf("%s", file->name());
		return TCL_OK;
	}
	else {
		TclObject *file;
		BEGIN_PARSE_ARGS(argc, argv);
		ARG(file);
		END_PARSE_ARGS;

		IndexFile_((IndexFile*)file);
		return TCL_OK;
	}
}


int
PlaybackStream::lts(int argc, const char * const *argv)
{
	Tcl &tcl = Tcl::instance();
	if (argc==2) {
		LTS *lts = LTS_();
		if (lts==NULL || lts->name()==NULL)
			tcl.resultf("");
		else
			tcl.resultf("%s", lts->name());
		return TCL_OK;
	}
	else {
		TclObject *lts;
		BEGIN_PARSE_ARGS(argc, argv);
		ARG(lts);
		END_PARSE_ARGS;

		LTS_((LTS*)lts);
		return TCL_OK;
	}
}


int
PlaybackStream::clip(int argc, const char * const * argv)
{
	double start, end;
	BEGIN_PARSE_ARGS(argc, argv);
	ARG(start);
	ARG(end);
	END_PARSE_ARGS;

	Clip(ftotv(start), ftotv(end));
	return TCL_OK;
}



void
PlaybackStream::LTS_Changed()
{
	double speed;
	timeval next;


	// cancel the current timer
	cancel();

	// get stats for log - not currently used
	/*timeval tv;
	timeval temp;
	passwd *pw;
	FILE *fp;

	pw = getpwuid(getuid());
	temp = LTS_()->NowLogical();
	fp = fopen("/h/mash/archive/.log/log", "a");
	if (is_first_ == 0) {
		fprintf(fp, "------ \n");
		is_first_=1;
	}
	tv = LTS_()->NowSystem();
	*/

	speed = lts_->Speed();
	if (speed==0.0) {
		// we have paused the system
		MTrace(trcArchive, ("LTS speed is zero; system has paused "
				    "(%d)", MYNUM(this)));
		//fprintf(fp, "%s %s %u %u %u %u %u %u Pause \n", pw->pw_name, (DataFile_())->get_name(), tv.tv_sec, tv.tv_usec, temp.tv_sec, temp.tv_usec, 0, 0);
		//fclose(fp);
		return;
	}
	else {
		MTrace(trcArchive, ("LTS has changed state; trying to "
				    "re-determine next timeout (%d)",
				    MYNUM(this)));
		if (NextEvent(next)==TCL_ERROR) {
			// FIXME: bgerror
			return;
		}
		MTrace(trcArchive, ("Scheduling event for %s (%d)",
				    tvtoa(next), MYNUM(this)));
		if (next.tv_sec==0 && next.tv_usec==0) {
			// we're done! so just return
			//fprintf(fp, "%s %s %u %u %u %u %u %u Done \n", pw->pw_name, (DataFile_())->get_name(), tv.tv_sec, tv.tv_usec, temp.tv_sec, temp.tv_usec, next.tv_sec, next.tv_usec);
			//fclose(fp);
			MTrace(trcArchive, ("stream is done (%d)",
					    MYNUM(this)));
			return;
		}
		//fprintf(fp, "%s %s %u %u %u %u %u %u \n", pw->pw_name, (DataFile_())->get_name(), tv.tv_sec, tv.tv_usec, temp.tv_sec, temp.tv_usec, next.tv_sec, next.tv_usec);
		//fclose(fp);
		sched(next);
	}
}


void
PlaybackStream::timeout()
{
	timeval next;
	MTrace(trcArchive|trcVerbose, ("Stream timeout occurred (%d)",
				    MYNUM(this)));
	do {
		DoEvent();
		if (NextEvent(next)==TCL_ERROR) {
			// FIXME: bgerror
			MTrace(trcArchive, ("Error in NextEvent: %s (%d)",
					    Tcl::instance().result(),
					    MYNUM(this)));
			return;
		}
		MTrace(trcVerbose|trcArchive, ("Scheduling event for %s (%d)",
				    tvtoa(next), MYNUM(this)));
		if (next.tv_sec==0 && next.tv_usec==0) {
			// we're done!
			MTrace(trcArchive, ("stream is done (%d)",
					    MYNUM(this)));
			return;
		}
	} while (next <= LTS_()->NowLogical());
	sched(next);
}
