/*
 * audio-sun.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1991-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

static const char rcsid[] =
    "@(#) $Header: /usr/mash/src/repository/mash/mash-1/audio/audio-sun.cc,v 1.14 2002/02/03 03:10:46 lim Exp $";

#undef resource_h

#include <stdlib.h>
#ifdef __svr4__
#include <sys/audioio.h>
#include <sys/filio.h>
#else
#include <sun/audioio.h>
#endif
#include <unistd.h>
#include <stropts.h>
#include <errno.h>
#include <fcntl.h>
#include "audio-sun.h"

class SUNDBRIAudio : public SUNAudio {
    public:
	SUNDBRIAudio(const char* device);
	virtual u_char* Read();
	virtual int NeedsSilence() const { return 1; }
    protected:

	int lastmean_[4];
};

class SUNXAudio : public SUNDBRIAudio {
    public:
	SUNXAudio(const char* device);
	virtual u_char* Read();
	virtual void Write(u_char*);
    protected:
	virtual void setup_device();

	u_char* ubuf_;
};

static class SunAudioClass : public TclClass {
    public:
	SunAudioClass() : TclClass("Audio/Sun") {}
	TclObject* create(int, const char*const*) {
		return (mk());
	}
	Audio* mk();
} sunaudio_class;

Audio* SunAudioClass::mk()
{
	Tcl& tcl = Tcl::instance();
	/* FIXME */
	tcl.evalf("[Application instance] get_option audioFileName");
	char device[256];
	strcpy(device, tcl.result());

	int i = 0;
	if (strcmp(device, "/dev/audio") == 0 &&
	    (i = open("/dev/audioctl", O_RDONLY, 0)) >= 0) {
		audio_info_t info;
		AUDIO_INITINFO(&info);
		int j = ioctl(i, AUDIO_GETINFO, &info);
		close(i);
		if (j != -1 &&
		    info.play.avail_ports > 3 &&
		    info.play.avail_ports < 32) {
			/*
			 * if the driver is advanced enough to let us set
			 * the blocksize, use the 'sunx' driver which reads
			 * in linear to avoid the ulaw quantization noise
			 * from the dc offset on the mike & line inputs.
			 * If we can't set the blocksize, use the 'dbri'
			 * driver which just reads in ulaw (if we can't
			 * set the blocksize & switch to linear, sun's
			 * braindead driver will supply a weird, huge
			 * blocksize).
			 */
#ifndef __svr4__
			int bs = info.record._xxx[2];
#else
			int bs = info.record.buffer_size;
#endif
			return ((bs != -1 && bs != 0)?
				  new SUNXAudio(device) :
				  new SUNDBRIAudio(device));
		}
	}
	return (new SUNAudio(device));
}

SUNAudio::SUNAudio(const char* device)
{
	audio_info_t* i = new audio_info_t;
	state_ = i;
	AUDIO_INITINFO(i);

	rgain_ = 0;
	rbalance_ = 32;
	pgain_ = 0;
	pbalance_ = 32;
	mgain_ = 0;

	bufcur_ = buf_ = new u_char[blksize_];
	bufend_ = buf_ + blksize_;
	device_ = new char[strlen(device) + 1];
	strcpy(device_, device);

	input_names_ = "mike linein";
	output_names_ = "speaker headphone lineout";
}

int SUNAudio::command(int argc, const char*const* argv)
{
	Tcl& tcl = Tcl::instance();
	if (argc == 2) {
		/*
		 * <otcl> Audio/Sun public get_input_ports {}
		 * Return the list of available input ports.
		 * The list consists of a sequence of name, value
		 * pairs, where the name is the nick name of
		 * the port (e.g., "mike") and the value is
		 * the integer number to be used to manipulate
		 * the port in methods that take a port number.
		 */
		if (strcmp(argv[1], "get_input_ports") == 0) {
			tcl.result(input_names_);
			return (TCL_OK);
		}
		/*
		 * <otcl> Audio/Sun public get_output_ports {}
		 * Return the list of available output ports.
		 * The list consists of a sequence of name, value
		 * pairs, where the name is the nick name of
		 * the port (e.g., "speaker") and the value is
		 * the integer number to be used to manipulate
		 * the port in methods that take a port number.
		 */
		if (strcmp(argv[1], "get_output_ports") == 0) {
			tcl.result(output_names_);
			return (TCL_OK);
		}
	} else if (argc == 3) {
		/*
		 * <otcl> Audio/Sun public set_input_balance bal
		 * Set the input balance when stereo inputs
		 * are available.  (mash handles only
		 * mono processing for now).  The balance must
		 * be reset from Tcl when the input port is changed.
		 */
		if (strcmp(argv[1], "set_input_balance") == 0) {
			SetRBalance(atoi(argv[2]));
			return (TCL_OK);
		}
		/*
		 * <otcl> Audio/Sun public set_output_balance bal
		 * Set the output balance when stereo inputs
		 * are available.  (mash handles only
		 * mono processing for now).  The balance must
		 * be reset from Tcl when the output port is changed.
		 */
		if (strcmp(argv[1], "set_output_balance") == 0) {
			SetPBalance(atoi(argv[2]));
			return (TCL_OK);
		}
		/*
		 * <otcl> Audio/Sun public set_monitor_gain val
		 * Set the monitor gain for the currently enabled input
		 * port to <i>level</i>, where level is a linear gain
		 * factor from 0 to 255. FIXME should change this?
		 */
		if (strcmp(argv[1], "set_monitor_gain") == 0) {
			SetMGain(atoi(argv[2]));
			return (TCL_OK);
		}
	}
	return (Audio::command(argc, argv));
}

int SUNAudio::OPort(int p)
{
	switch(p) {
	default:
	case 0:
		return (AUDIO_SPEAKER);
	case 1:
		return (AUDIO_HEADPHONE);
	case 2:
		return (AUDIO_LINE_OUT);
	}
}

int SUNAudio::IPort(int p)
{
	switch(p) {
	default:
	case 0:
		return (AUDIO_MICROPHONE);
	case 1:
		return (AUDIO_LINE_IN);
	}
}

void SUNAudio::Obtain()
{
	if (!haveaudio()) {
		/*
		 * try to save the current audio state (we have to
		 * do this before opening /dev/audio because the open
		 * will wipe out the sample rate & encoding).
		 */
		int ctl = open("/dev/audioctl", O_RDONLY, 0);
		if (ctl >= 0) {
			audio_info_t* i = (audio_info_t*)state_;
			AUDIO_INITINFO(i);
			ioctl(ctl, AUDIO_GETINFO, i);
			close(ctl);
		} else {
			perror("audio");
		}
		fd_ = open(device_, O_RDWR|O_NDELAY);
		if (fd_ >= 0) {
			setup_device();
			Audio::Obtain();
		} else
			perror("open");
	}
}

void SUNAudio::Release()
{
	if (haveaudio()) {
		/* restore the audio state to what it was on entry */
		setinfo((audio_info_t*)state_);
		Audio::Release();
	}
}

void SUNAudio::Write(u_char *cp)
{
	register int len = blksize_;
	int cc = write(fd_, (char *)cp, len);
	if ((len -= cc) != 0) {
		do {
			if (cc < 0) {
				if (errno != EPERM)
					perror("audio write");
				return;
			}
			cp += cc;
			cc = write(fd_, (char *)cp, len);
			len -= cc;
		} while (len > 0);
	}
}

int SUNAudio::FrameReady()
{
	register int len = bufend_ - bufcur_;
	while (len > 0) {
		int cc = read(fd_, (char *)bufcur_, len);
		if (cc < 0) {
			switch (errno) {
			case EINVAL:
				/* probably wrapped file pos. */
				lseek(fd_, 0, SEEK_SET);
				cc = 0;
				break;

			case EPERM:
				/* probably lost audio */
				Release();
				Obtain();
				goto out;

#ifndef __svr4__
			case EWOULDBLOCK:
#endif
			case EAGAIN:
				/* should warn about audio_bsize here */
				goto out;

			default:
				perror("audio read");
				Release();
				Obtain();
				return (0);
			}
		}
		bufcur_ += cc;
		len -= cc;
	}
    out:
	return (bufcur_ >= bufend_);
}

u_char* SUNAudio::Read()
{
	bufcur_ = buf_;
	return (buf_);
}

void SUNAudio::setup_device()
{
	audio_info_t info;
	if (fcntl(fd_, F_SETFL, O_NONBLOCK|O_NDELAY) < 0)
		perror("audio O_NONBLOCK");
	int on = 1;
	ioctl(fd_, FIONBIO, &on);
#ifndef __svr4__
	int bsddrvr = 0;
	AUDIO_INITINFO(&info);
	getinfo(&info);
	if (info._yyy[0] == 0)
		bsddrvr = 1;
#endif
	AUDIO_INITINFO(&info);
	info.play.sample_rate = 8000; /*FIXME*/
	info.play.channels = 1; /*FIXME*/
	info.play.precision = 8; /*FIXME*/
	info.play.encoding = AUDIO_ENCODING_ULAW; /*FIXME*/
	info.play.gain = pgain_;
	info.play.port = OPort(oport_);
	info.play.balance = pbalance_;

	info.record.sample_rate = 8000; /*FIXME*/
	info.record.channels = 1; /*FIXME*/
	info.record.precision = 8; /*FIXME*/
	info.record.encoding = AUDIO_ENCODING_ULAW; /*FIXME*/
	info.record.gain = rgain_;
	info.record.port = IPort(iport_);
	info.record.balance = rbalance_;
#ifndef __svr4__
	/* solaris 2.2 hack: set buffer size.
	 * NB- this changes to _xxx[3] for os<4.1.3 */
	info.record._xxx[2] = blksize_;
	if (bsddrvr) {
		/* bsd driver hack - set buffer size */
		info._yyy[0] = blksize_;
	} else {
		/* 4.1.3 bug workaround: pause then resume */
		info.record.pause = 1;
		setinfo(&info);
		info.record.pause = 0;
	}
#else
	info.record.buffer_size = blksize_;
#endif
	info.monitor_gain = mgain_;
	setinfo(&info);
	/* flush input to get rid of any data fragments */
	ioctl(fd_, I_FLUSH, FLUSHR);
	bufcur_ = buf_;
}


int SUNAudio::GainClip(int level)
{
        if (level < AUDIO_MIN_GAIN)
                return AUDIO_MIN_GAIN;
        else if (level > AUDIO_MAX_GAIN)
                return AUDIO_MAX_GAIN;
        else
                return level;
}

int SUNAudio::getinfo(audio_info_t* info)
{
	int sts;
	if (fd_ < 0)
		sts = 0;
	else
		sts = ioctl(fd_, AUDIO_GETINFO, info);
	return (sts);
}

int SUNAudio::setinfo(audio_info_t* info)
{
	int sts;
	if (fd_ < 0)
		sts = 0;
	else
		sts = ioctl(fd_, AUDIO_SETINFO, info);
	return (sts);
}

void SUNAudio::SetRGain(int level)
{
	audio_info_t info;

	rgain_ = GainClip(level);
	AUDIO_INITINFO(&info);
	info.record.gain = rgain_;
	setinfo(&info);
}

void SUNAudio::SetPGain(int level)
{
	audio_info_t info;

	pgain_ = GainClip(level);
	AUDIO_INITINFO(&info);
	info.play.gain = pgain_;
	setinfo(&info);
}

void SUNAudio::SetRBalance(int level)
{
	audio_info_t info;

	rbalance_ = GainClip(level);
	AUDIO_INITINFO(&info);
	info.record.balance = rbalance_;
	setinfo(&info);
}

void SUNAudio::SetMGain(int level)
{
	audio_info_t info;

	mgain_ = GainClip(level);
	AUDIO_INITINFO(&info);
	info.monitor_gain = mgain_;
	setinfo(&info);
}

void SUNAudio::SetPBalance(int level)
{
	audio_info_t info;

	pbalance_ = GainClip(level);
	AUDIO_INITINFO(&info);
	info.play.balance = pbalance_;
	setinfo(&info);
}

void SUNAudio::OutputPort(int p)
{
	audio_info_t info;

	oport_ = p;
	AUDIO_INITINFO(&info);
	info.play.port = OPort(p);
	setinfo(&info);
}

void SUNAudio::InputPort(int p)
{
	audio_info_t info;

	iport_ = p;
	AUDIO_INITINFO(&info);
	info.record.port = IPort(p);
	setinfo(&info);
}


extern const short mulawtolin[];
extern const unsigned char lintomulaw[];
extern const unsigned char lintomulawX[];

SUNDBRIAudio::SUNDBRIAudio(const char* device) : SUNAudio(device)
{
	lastmean_[0] = 0;
	lastmean_[1] = 0;
	lastmean_[2] = 0;
	lastmean_[3] = 0;
	input_names_ = "mike linein";
	output_names_ = "speaker headphone lineout";
}

u_char* SUNDBRIAudio::Read()
{
	register u_char* cp = buf_;

	/*
	 * for some reason, SUN didn't bother to filter out the
	 * mike 'phantom power' DC signal so we end up with a
	 * large DC offset that screws up the lin-to-mu conversion
	 * and the speakerphone power calculations.  So, all the
	 * hair in the following loop is a low-pass filter with ~1Hz
	 * passband to estimate the DC offset.
	 */
	register u_int* ip = (u_int*)cp;
	register u_int* ep = (u_int*)(cp + blksize_);
	register int smean = lastmean_[iport_];
	register const short* u2l = mulawtolin;
	register const u_char* l2u = lintomulaw;
	if (iport_ != 0) {
		while (ip < ep) {
			register int mean, dif;
			register u_int res;
			register u_int dat = *ip;
			register int s0 = u2l[dat >> 24];
			register int s1 = u2l[(dat >> 16) & 0xff];
			register int s2 = u2l[(dat >> 8) & 0xff];
			register int s3 = u2l[dat & 0xff];

			mean = smean >> 13;
			dif = s0 - mean;
			smean += dif;
			res = l2u[dif & 0xffff] << 24;

			mean = smean >> 13;
			dif = s1 - mean;
			smean += dif;
			res |= l2u[dif & 0xffff] << 16;

			mean = smean >> 13;
			dif = s2 - mean;
			smean += dif;
			res |= l2u[dif & 0xffff] << 8;

			mean = smean >> 13;
			dif = s3 - mean;
			smean += dif;
			res |= l2u[dif & 0xffff];

			*ip++ = res;
		}
	} else {
		while (ip < ep) {
			register int mean, dif;
			register u_int res;
			register u_int dat = *ip;
			register int s0 = u2l[dat >> 24];
			register int s1 = u2l[(dat >> 16) & 0xff];
			register int s2 = u2l[(dat >> 8) & 0xff];
			register int s3 = u2l[dat & 0xff];

			mean = smean >> 13;
			dif = s0 - mean;
			smean += dif;
			res = l2u[(dif >> 1) & 0xffff] << 24;

			mean = smean >> 13;
			dif = s1 - mean;
			smean += dif;
			res |= l2u[(dif >> 1) & 0xffff] << 16;

			mean = smean >> 13;
			dif = s2 - mean;
			smean += dif;
			res |= l2u[(dif >> 1) & 0xffff] << 8;

			mean = smean >> 13;
			dif = s3 - mean;
			smean += dif;
			res |= l2u[(dif >> 1) & 0xffff];

			*ip++ = res;
		}
	}
	lastmean_[iport_] = smean;
	bufcur_ = cp;
	return (cp);
}


SUNXAudio::SUNXAudio(const char* device) : SUNDBRIAudio(device)
{
	ubuf_ = bufcur_;
	bufcur_ = buf_ = new u_char[blksize_ << 1];
	bufend_ = buf_ + (blksize_ << 1);
}

u_char* SUNXAudio::Read()
{
	/*
	 * for some reason, SUN didn't bother to filter out the
	 * mike 'phantom power' DC signal so we end up with a
	 * large DC offset that screws up the lin-to-mu conversion
	 * and the speakerphone power calculations.  So, all the
	 * hair in the following loop is a low-pass filter with ~1Hz
	 * passband to estimate the DC offset.  To avoid the
	 * quantization noise from unbiasing a mulaw signal, we
	 * have to read & write 8KHz linear.
	 */
	register short* ip = (short*)buf_;
	register short* ep = ip + blksize_;
	register u_int* op = (u_int*)ubuf_;
	register int smean = lastmean_[iport_];
	register const u_char* l2u = lintomulawX;
	while (ip < ep) {
		register int mean, dif;
		register u_int res;
		register int s0 = ip[0];
		register int s1 = ip[1];
		register int s2 = ip[2];
		register int s3 = ip[3];

		mean = smean >> 13;
		dif = s0 - mean;
		smean += dif;
		res = l2u[dif & 0x1ffff] << 24;

		mean = smean >> 13;
		dif = s1 - mean;
		smean += dif;
		res |= l2u[dif & 0x1ffff] << 16;

		mean = smean >> 13;
		dif = s2 - mean;
		smean += dif;
		res |= l2u[dif & 0x1ffff] << 8;

		mean = smean >> 13;
		dif = s3 - mean;
		smean += dif;
		res |= l2u[dif & 0x1ffff];

		*op++ = res;
		ip += 4;
	}
	lastmean_[iport_] = smean;
	bufcur_ = buf_;
	return (ubuf_);
}


void SUNXAudio::Write(u_char *cp)
{
	register u_int len = blksize_;
	u_int samps[MAXAUDIOSIZE/2];
	register u_int* sp = samps;
	register u_int* ep = sp + len / 2;
	register const u_short* u2l = (u_short*)mulawtolin;
	register u_int* ip = (u_int*)cp;
	for ( ; sp < ep; sp += 4) {
		register u_int s = *ip++;
		sp[0] = (u2l[(s >> 24) & 0xff] << 16) | u2l[(s >> 16) & 0xff];
		sp[1] = (u2l[(s >> 8) & 0xff] << 16) | u2l[s & 0xff];
		s = *ip++;
		sp[2] = (u2l[(s >> 24) & 0xff] << 16) | u2l[(s >> 16) & 0xff];
		sp[3] = (u2l[(s >> 8) & 0xff] << 16) | u2l[s & 0xff];
	}
	write(fd_, (char *)samps, len << 1);
}

void SUNXAudio::setup_device()
{
	SUNAudio::setup_device();

	audio_info_t info;
	AUDIO_INITINFO(&info);
	info.record.precision = 16;
	info.record.encoding  = AUDIO_ENCODING_LINEAR;
#ifndef __svr4__
	info.record._xxx[2]   = blksize_ << 1;
#else
	info.record.buffer_size = blksize_ << 1;
#endif
	info.play.precision   = 16;
	info.play.encoding    = AUDIO_ENCODING_LINEAR;
	setinfo(&info);
	/* flush input again to get rid of any data with wrong encoding */
	ioctl(fd_, I_FLUSH, FLUSHR);
	bufcur_ = buf_;
}
