/*
 * audio-voxware.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1991-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

static const char rcsid[] =
    "@(#) $Header: /usr/mash/src/repository/mash/mash-1/audio/audio-voxware.cc,v 1.18 2002/02/03 03:10:46 lim Exp $";

#include <string.h>
#include <sys/fcntl.h>
#include <errno.h>
#if defined(sco) || defined(__bsdi__)
#include <sys/socket.h>
#endif
#include <unistd.h>
#if defined(__FreeBSD__)
#include <sys/types.h>
#include <sys/uio.h>
#include <machine/soundcard.h>
#else
#include <sys/soundcard.h>
#endif
#include "audio.h"
#include "tclcl.h"

#define ULAW_ZERO 0x7f
#define ABUFLOG2 7
#define ABUFLEN (1 << ABUFLOG2)
#define NFRAG 5

#if defined(__bsdi__)
#define BSD_SB_FIX
#endif
class VoxWareAudio : public Audio {
    public:
	VoxWareAudio();
	virtual int FrameReady();
	virtual u_char* Read();
	virtual	void Write(u_char *);
	virtual void SetRGain(int);
	virtual void SetPGain(int);
	virtual void OutputPort(int);
	virtual void InputPort(int);
	virtual void Obtain();
	virtual void Release();
    protected:

	u_char* readptr;
	u_char* readbufend;
	u_char* readbuf;

	u_char* ubufptr;
	u_char* ubufend;
	u_char* ubuf;

	u_char* writeptr;
	u_char* writebufend;
	u_char* writebuf;

	int mixer;
#ifdef BSD_SB_FIX
#define NONE 0
#define READ 1
#define WRITE 2
	int lastop;       /* last operation: read or write */
#endif /* BSD_SB_FIX */
};

static class VoxWareAudioClass : public TclClass {
    public:
	VoxWareAudioClass() : TclClass("Audio/VoxWare") {}
	TclObject* create(int, const char*const*) {
		return (new VoxWareAudio);
	}
} voxware_audio_class;

VoxWareAudio::VoxWareAudio()
	:mixer(-1)
{
	/*FIXME assumes GUS PnP (or similar); should query this */
	input_names_ = "mike linein";
	output_names_ = "speaker";

	readbuf = new u_char[ABUFLEN];
	readptr = readbufend = readbuf + ABUFLEN;

	writeptr = writebuf = new u_char[ABUFLEN];
	writebufend = writebuf + ABUFLEN;

	ubufptr = ubuf = new u_char[blksize_];
	ubufend = ubuf + blksize_;
	memset(ubuf, ULAW_ZERO, blksize_);

	/*
	 * The only way to determine if the device is full duplex
	 * or not is by actually opening it.  Unfortunately, we might
	 * not be able to open it because some other process is
	 * using it.  Assume half-duplex.  Obtain() will override
	 * if appropriate.
	 */
	duplex_ = 0;
#ifdef BSD_SB_FIX
	lastop = NONE;
#endif /* BSD_SB_FIX */
}

#if defined(__bsdi__)

void VoxWareAudio::Obtain()
{
	if (haveaudio())
		abort();

	fd_ = open("/dev/audio", O_RDWR|O_NDELAY);
	if (fd_ >= 0) {
		int on = 1;
		ioctl(fd_, FIONBIO, &on);

		int frag = (NFRAG << 16) | ABUFLOG2;
		ioctl(fd_, SNDCTL_DSP_SETFRAGMENT, &frag);
#ifdef fullduplex
		Audio::Obtain();
#else
		notify();
#endif
	}
}

#else
void VoxWareAudio::Obtain()
{
	if (haveaudio())
		abort();

	fd_ = open("/dev/audio", O_RDWR|O_NDELAY);
	if (fd_ >= 0) {
		mixer = open("/dev/mixer", O_RDWR);
		if (mixer < 0) {
			/*FIXME*/
			perror("/dev/mixer");
		}
		/*
		 * Set non-blocking mode.  Have to do the driver
		 * version as well.  FIXME is the FIONBIO really needed?
		 */
		int on = 1;
		ioctl(fd_, FIONBIO, &on);
		on = 1;
		ioctl(fd_,SNDCTL_DSP_NONBLOCK , &on);

		/*
		 * Set up the DMA capture buffers.  The value in the
		 * upper word is the maximum backlog in blocks
		 * and the lower word is the (base 2 log of the)
		 * block size.  Code in this module assumes this
		 * block size is smaller than the frame size
		 * used in the rest of vat (usually 160 samples so
		 * ABUGLOG2 should be no larger than 7).
		 */
		int frag = (NFRAG << 16) | ABUFLOG2;
		ioctl(fd_, SNDCTL_DSP_SETFRAGMENT, &frag);

		/*
		 * (re)set the speed in case someone else changed it
		 */
		int speed = 8000;
		ioctl(fd_, SNDCTL_DSP_SPEED, &speed);

#if defined(SNDCTL_DSP_GETCAPS)
		int info;
		if (ioctl(fd_, SNDCTL_DSP_GETCAPS, &info) < 0)
			duplex_ = 0;
		else
			duplex_ = (info & DSP_CAP_DUPLEX) ? 1 : 0;
#else
		duplex_ = 0;
#endif

		/*
		 * Set the line input level to 0 to shut
		 * off the analog side-tone gain between
		 * the line-in and line-out.  This would otherwise
		 * wreak havoc on an echo canceler, for example,
		 * plugged into the audio adaptor.
		 */
		int v = 0;
		if (mixer > 0)
			(void)ioctl(mixer, MIXER_WRITE(SOUND_MIXER_LINE), &v);

		/*
		 * Restore the hardware settings in case
		 * some other vat changed them.
		 */
		SetRGain(rgain_);
		SetPGain(pgain_);
		InputPort(iport_);

		/*FIXME*/
		if (duplex_)
			Audio::Obtain();
		else
			notify();
	}
}
#endif

void VoxWareAudio::Release()
{
	if (haveaudio()) {
		if (mixer > 0)
			(void)close(mixer);
		mixer = -1;
		Audio::Release();
	}
}


void VoxWareAudio::Write(u_char *cp)
{
	register u_char *cpend = cp + blksize_;
	register u_char *wbuf = writeptr;
	register u_char *wend = writebufend;
	for ( ; cp < cpend; cp += 4) {
		wbuf[0] = cp[0];
		wbuf[1] = cp[1];
		wbuf[2] = cp[2];
		wbuf[3] = cp[3];
		wbuf += 4;
		if (wbuf >= wend) {
			wbuf = writebuf;
			if (write(fd_, (char*)wbuf, ABUFLEN) != ABUFLEN)
				perror("voxware aud write");
		}
#ifdef BSD_SB_FIX
		/*
		 * close and reopen audio device when switching
		 * from read to write.
		 */
		if (lastop == READ) {
			Release();
			Obtain();
		}
		lastop = WRITE;
#endif /* BSD_SB_FIX */
	}
	writeptr = wbuf;
}

int VoxWareAudio::FrameReady()
{
	register u_char* cp = ubufptr;
	register u_char* cpend = ubufend;
	register u_char* rbuf = readptr;
	register u_char* rend = readbufend;

	/*
	 * This loop assumes the output block size is
	 * larger than the read size (ABUFLEN).
	 */
#ifdef BSD_SB_FIX
	/*
	 * close and reopen audio device when switching
	 * from read to write.
	 */
	if (lastop == WRITE) {
		Release();
		Obtain();
	}
	lastop = READ;
#endif /* BSD_SB_FIX */
	for ( ; cp < cpend; cp += 4) {
		if (rbuf >= rend) {
			rbuf = readbuf;
			int cc = read(fd_, (char*)rbuf, ABUFLEN);
			if (cc <= 0) {
				ubufptr = cp;
				readbufend = rbuf;
				if (cc == -1 && errno != EAGAIN) {
					Release();
					Obtain();
				}
				return (0);
			}
			readbufend = rend = rbuf + cc;
		}
		cp[0] = rbuf[0];
		cp[1] = rbuf[1];
		cp[2] = rbuf[2];
		cp[3] = rbuf[3];
		rbuf += 4;
	}
	readptr = rbuf;
	return (1);
}

u_char* VoxWareAudio::Read()
{
	u_char* cp = ubuf;
	ubufptr = cp;
	return (cp);
}

void VoxWareAudio::SetRGain(int level)
{
        rgain_ = level;
	level = int(level / 2.56);
	int v = level << 8 | level;
	if (mixer >= 0 &&
	    ioctl(mixer, MIXER_WRITE(SOUND_MIXER_IGAIN), &v) < 0)
		perror("SOUND_MIXER_IGAIN");
}

void VoxWareAudio::SetPGain(int level)
{
	pgain_ = level;
	level = int(level / 2.56);
	int v = level << 8 | level;
	if (mixer >= 0)
		(void)ioctl(mixer, MIXER_WRITE(SOUND_MIXER_PCM), &v);
}

void VoxWareAudio::OutputPort(int p)
{
	oport_ = p;
}

void VoxWareAudio::InputPort(int p)
{
	if (mixer >= 0) {
		/*
		 * the port number is the positional index of the list
		 * of input ports in input_names_
		 */
		int mask = (p == 0) ?
			SOUND_MASK_MIC : SOUND_MASK_LINE;
		if (ioctl(mixer, MIXER_WRITE(SOUND_MIXER_RECSRC), &mask) < 0)
			perror("voxware ioctl: SOUND_MIXER_RECSRC");
	}
	iport_ = p;
}
