/*
 * vidreps.h --
 *
 *      Video frame memory representation class definitions.
 *
 *      Defined in this file:
 *
 *      VidRep - base class for all video representations.
 *      Uncompressed - YUV uncompressed representation.
 *      Semicompressed - YUV representation at DCT coefficient level.
 *      JPEGCompressed - Fully compressed JPEG frame.
 *
 * Copyright (c) 1998-2001 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/* @(#) $Header: /usr/mash/src/repository/mash/mash-1/fx/vidreps.h,v 1.1 2002/02/07 04:17:01 chema Exp $ */


#ifndef VIDREP_H
#define VIDREP_H

#include "config.h"
#include "tclcl.h"
#include "module.h"
#include "tclDvmBasic.h"


// defines for the init and dealloc methods. Effects should only use the 
//	first two flags, and the first one only if performance is needed through 
//	buffer copying avoidance
// a. use another frame's memory buffer
#define VIDREP_OVERLOAD_MEMORY 0x01
// b. allocate memory for both luma and chroma
#define VIDREP_ALLOC_MEMORY 0x02
// c. allocate memory only for chroma buffers
#define VIDREP_ALLOC_CHROMAS 0x04
// d. register in Dali (for compatibility purposes)
#define VIDREP_REGISTER 0x08
//
// ISSET used to check flags
// 
#define ISSET(type,flag) ((type & flag) == flag)


class VidRep : public TclObject, public Buffer {
public:
	VidRep();
	virtual ~VidRep();

	int w_;
	int h_;

	// chroma subsampling
	int css_h_;
	int css_v_;

	// in case this is a part a bigger frame, this one's size and relative 
	//	position
	int true_w_;
	int true_h_;
	int xoff_;
	int yoff_;

	int resized_;
	u_int32_t ssrc_;
	u_int32_t ts_;
	char *source_id_;

	virtual void copy_geometry(VidRep *temp); 
	virtual int command(int argc, const char*const* argv);
	virtual Buffer *copy() {
		/* FIXME This needs to be changed, but we'll do that later. */
		return 0;
	};
	inline void convert_csss (int csss, int &css_h, int &css_v) {
		switch (csss) {
		case 444:
			css_h = 1; css_v = 1; break;
		case 422:
			css_h = 2; css_v = 1; break;
		case 420:
			css_h = 2; css_v = 2; break; 
		case 411:
			css_h = 4; css_v = 1; break; 
		default:
			fprintf (stderr, "convert_csss: don't understand %d csss\n", csss);
			abort();
		}
	};

	inline int convert_csss (int css_h, int css_v) {
		if ((css_h == 2) && (css_v == 1)) {
			return (422);
		} else if ((css_h == 2) && (css_v == 2)) {
			return (420);
		} else if ((css_h == 4) && (css_v == 1)) {
			return (411);
		} else if ((css_h == 1) && (css_v == 1)) {
			return (444);
		} else {
			fprintf (stderr, "convert_csss: unknown csss (%d,%d)\n", css_h, css_v);
			abort();
		}
		return (-1);
	};

	inline int get_csss () {
		return convert_csss (css_h_, css_v_);
	};
};

class Uncompressed : public VidRep {
public:
	Uncompressed();
	Uncompressed(int type);
	virtual ~Uncompressed();

	void init(int w, int h, int css_h, int css_v, int alt_css_h, int alt_css_v,
			int true_w, int true_h, int xoff, int yoff, 
			u_char *ldata, u_char *crdata, u_char *cbdata, int type);
	void init(Uncompressed *fr, int csss, int type);
	void dealloc();

	virtual inline int samesize(Uncompressed *fr) {
		// TBD: currently we just check width and height, which probably 
		//	isn't enough
		return ((w_ == fr->w_) && (h_ == fr->h_));
	};

	void byte_copy(Uncompressed *frame);
	void check_chroma();
	void add_chroma(int csss);
	void save_disk(char* prefix);
	virtual int command(int argc, const char*const* argv);

	// memory allocation description: may be one of the following:
	//	- VIDREP_OVERLOAD_MEMORY, if no memory was self-allocated,
	//	- VIDREP_ALLOC_MEMORY, if the full memory was self-allocated, and
	//	- VIDREP_ALLOC_CHROMAS if two alternative chromas were allocated.
	int memory_allocation_;

	// main frame representation (Y, Cr, and Cb). In the case of frames 
	//	with multiple chroma representations, the usual memory arrangement 
	//	is that there are two chunks of memory, one with one luma and 
	//	two chromas and the second with the two alternative chromas. The 
	//	first one has normally been allocated elsewhere, and is pointed 
	//	by lum_->firstByte, alt_cr_->firstByte, and alt_cb_->firstByte. The 
	//	second chunk has been allocated here and is pointed by cr_->firstByte 
	//	and cb_->firstByte
	ByteImage *lum_;
	ByteImage *cr_;
	ByteImage *cb_;
	// Dali registration ids for the three main buffers
	int lum_id_;
	int cr_id_;
	int cb_id_;

	// alternative chroma representation. This is only used when chroma 
	//	subsampling upgrading or downgrading is needed.
	int alt_css_h_;
	int alt_css_v_;
	ByteImage *alt_cr_;
	ByteImage *alt_cb_;
	int alt_cr_id_;
	int alt_cb_id_;

	// main byte pointer. it points always to the self-allocated memory
	u_char *bp_;
};

class JPEGCompressed : public VidRep {
public:
	JPEGCompressed();
	virtual ~JPEGCompressed();

	int self_allocated_;
	int q_;
	int len_;

	u_char *bp_;

	void set(int w, int h, int decimation, int q, int len, 
			u_int32_t ssrc, u_int32_t ts, u_char *bp);
	void dealloc();
	virtual int command(int argc, const char*const* argv);
};

class Semicompressed : public VidRep {
public:
	Semicompressed();
	virtual ~Semicompressed();

	int self_allocated_;
 
	ScImage *lum_;
	ScImage *cr_;
	ScImage *cb_;

	int lum_id_;
	int cr_id_;
	int cb_id_;

	void allocate();
	void dealloc();

	void print();
	void print(int x, int y);

	virtual int command(int argc, const char*const* argv);

	void set(int w, int h, int h_subsample, int v_subsample, 
			int tw, int th, int xoff, int yoff);
};
#endif

