/*
 * mbv2-src.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1998-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef lint
static const char rcsid[] = "@(#) $Header: /usr/mash/src/repository/mash/mash-1/mbv2/mbv2-src.cc,v 1.12 2002/02/03 03:17:08 lim Exp $";
#endif


#include "mbv2-obj.h"
#include "mbv2-cmd.h"
#include "mtrace.h"
#include "mbv2-listsort.h"

// the following include is required for definition of LookupLocalAddr()
#include "net/net.h"

DEFINE_OTCL_CLASS(MBv2Source, "MBv2Source") {
	INSTPROC_PUBLIC(source_id);
	INSTPROC_PUBLIC(session);
}


MBv2Source::MBv2Source()
	: session_(NULL), srm_source_(NULL)
{
	Tcl_InitHashTable(&htPages_, TCL_ONE_WORD_KEYS);
	Tcl_InitHashTable(&htPageIds_, sizeof(MBv2PageId)/sizeof(int));
}


MBv2Source::~MBv2Source()
{
	// FIXME: must do cleanup here
	Tcl_DeleteHashTable(&htPages_);
	Tcl_DeleteHashTable(&htPageIds_);
}


int
MBv2Source::init(int argc, const char * const *argv)
{
	TclObject *session;
	int srm_source_token=0;
	const char *cname;
	BEGIN_PARSE_ARGS(argc, argv);
	ARG(session);
	ARG(cname);
	ARG_DEFAULT(srm_source_token, 0);
	END_PARSE_ARGS;

	session_ = (MBv2Session*)session;
	if (srm_source_token==0) {
		char *default_srcid = NULL;
		if (*cname != '\0') {
			default_srcid = new char [24 + strlen(cname)];
			sprintf(default_srcid, "%s:%s",
				intoa(LookupLocalAddr()), cname);
		}
		srm_source_ = srm_create_source(session_->srm_session(),
						default_srcid);
		if (!default_srcid) delete [] default_srcid;
	} else {
		srm_source_ = (srm_source_t) srm_source_token;
	}

	session_->source(srm_source_, this);
	return TCL_OK;
}


int
MBv2Source::source_id(int argc, const char * const *argv)
{
	BEGIN_PARSE_ARGS(argc, argv);
	END_PARSE_ARGS;

	char srcid[40];
	srm_srcid2str(srm_get_source_id(srm_source_), srcid);
	Tcl::instance().resultf("%s", srcid);
	return TCL_OK;
}


int
MBv2Source::session(int argc, const char * const *argv)
{
	BEGIN_PARSE_ARGS(argc, argv);
	END_PARSE_ARGS;

	Tcl::instance().resultf("%s", session_->name());
	return TCL_OK;
}


void
MBv2Source::recv(unsigned int cid, unsigned int seqno,
		 const unsigned char *data, int len, const srm_adu_info *info)
{
	MBv2Page *page = find_page(cid);
	if (!page) {
		MTrace(trcMB, ("huh? a mapping for this page (cid=%u) ought "
			       "to exist", cid));
		return;
	}

	if (page->have_cmd((MBv2CmdId)seqno)) {
		// we already have this command
		// don't bother parsing it out
		return;
	}

	MBv2Cmd *cmd = MBv2Cmd::create((MBv2CmdId)seqno, info, data, len);
	if (!cmd) return;

	page->insert(cmd);
}


Bool
MBv2Source::dispatch(MBv2Page *page, MBv2Cmd *cmd)
{
	// packetize the new command
	int size = cmd->adu_size(), adu_buf_size;
	unsigned char *adu_buf = session_->adu_buf(adu_buf_size);
	if (size > adu_buf_size) {
		adu_buf = session_->resize_adu_buf(size);
	}
	if (!cmd->packetize(adu_buf)) {
		return FALSE;
	}

	// send the ADU
	srm_adu_info adu_info;
	adu_info.type = cmd->type();
	adu_info.timestamp = cmd->timestamp();
	unsigned int seqno = srm_send(srm_source_, page->cid(), adu_buf, size,
				      &adu_info);
	MTrace(trcMB|trcVerbose, ("sending cmd %u", seqno));

	// assign the cmd ID
	cmd->id((MBv2CmdId)seqno);

	// apply the command
	page->insert(cmd);
	return TRUE;
}


void
MBv2Source::read_adu(unsigned int cid, unsigned int seqno,
		     unsigned char **data_ptr, unsigned int *len_ptr,
		     srm_free_proc *free_proc_ptr,
		     srm_adu_info *info)
{
	// find the cmd
	MBv2Page *page = find_page(cid);
	if (!page) return;
	MBv2Cmd *cmd = page->cmd(seqno);
	if (!cmd) return;

	// packetize the new command
	int size = cmd->adu_size(), adu_buf_size;
	unsigned char *adu_buf = session_->adu_buf(adu_buf_size);
	if (size > adu_buf_size) {
		adu_buf = session_->resize_adu_buf(size);
	}
	if (!cmd->packetize(adu_buf)) {
		return;
	}

	info->type = cmd->type();
	info->timestamp = cmd->timestamp();

	*data_ptr = adu_buf;
	*len_ptr  = (unsigned int)size;
	*free_proc_ptr = NULL;
}


void
MBv2Source::recv_cid(unsigned int cid, const unsigned char *name,
		     int name_len)
{
	MTrace(trcMB, ("Creating page object for cid %u", cid));
	if (name_len != sizeof(MBv2PageId)) return;
	MBv2PageId *net = (MBv2PageId*)name, id;
	id = ntoh(*net);
	MBv2Page *page = create_page(cid, id);
	// if there are any deferred commands waiting for this cid mapping,
	// move them to the page's internal list now
	flush_deferred(cid, page);
}


MBv2Page *
MBv2Source::create_page(u_int32_t cid, const MBv2PageId &pageid)
{
	MBv2Page *page;
	int created;
	Tcl_HashEntry *entry=Tcl_CreateHashEntry(&htPages_, (char*)cid,
						 &created);
	if (created) {
		page = new MBv2Page(this, cid, pageid);
		page->canvas(session_->get_canvas(pageid));
		Tcl_SetHashValue(entry, page);

		entry = Tcl_CreateHashEntry(&htPageIds_, (char*)&pageid,
					    &created);
		Tcl_SetHashValue(entry, page);
	} else {
		page = (MBv2Page*) Tcl_GetHashValue(entry);
	}
	return page;
}


MBv2Page *
MBv2Source::create_page(const MBv2PageId &pageid)
{
	MBv2PageId net = hton(pageid);
	unsigned int cid = srm_calloc(srm_source_, SRMv2_ROOT_CID,
				      (const unsigned char *)&net,
				      sizeof(MBv2PageId));
	return create_page(cid, pageid);
}


MBv2Page *
MBv2Source::find_page(u_int32_t cid)
{
	Tcl_HashEntry *entry = Tcl_FindHashEntry(&htPages_, (char*)cid);
	if (entry) return (MBv2Page*) Tcl_GetHashValue(entry);
	else return NULL;
}


// FIXME: when we have epochs, this should find a page in the current epoch
MBv2Page *
MBv2Source::find_page(const MBv2PageId &pageid)
{
	Tcl_HashEntry *entry = Tcl_FindHashEntry(&htPageIds_, (char*)&pageid);
	if (entry) return (MBv2Page*) Tcl_GetHashValue(entry);
	else return NULL;
}


void
MBv2Source::defer_cmd(MBv2Page *page, MBv2Cmd *cmd, MBv2Dependency *dep)
{
	MBv2Page *dependsOnPage = find_page(dep->cid);
	if (dependsOnPage) {
		dependsOnPage->defer_cmd(page, cmd, dep);
	} else {
		// add this to the per-source defer list
		MBv2SaveDependency *d = new MBv2SaveDependency;
		d->page = page;
		d->cmd  = cmd;
		d->dep  = *dep;

		::insert_sorted(&deferredCmds_, d);
	}
}


void
MBv2Source::flush_deferred(u_int32_t cid, MBv2Page *page)
{
	ListIndex idx=deferredCmds_.getFirst(), tmp;

	// do the easy cases first
	if (deferredCmds_.IsEmpty() ||
	    deferredCmds_.PeekAtTail()->dep.cid < cid) {
		// there is no match here
		return;
	}

	while (!deferredCmds_.IsDone(idx)) {
		MBv2SaveDependency *d = deferredCmds_.getData(idx);
		tmp = idx;
		idx = deferredCmds_.getNext(idx);
		if (d->dep.cid > cid) break;
		if (d->dep.cid ==cid) {
			page->defer_cmd(d->page, d->cmd, &d->dep);
			deferredCmds_.Remove(tmp);
			delete d;
		}
	}
}

