/*
 * rgb-converter.cc --
 *
 *      RGB to YUV converters. It supports 32-, 24-, 16-, 8-, 4-, and
 *      1-bit color RGB. You have to provide both the RGB and the YUV
 *      frames, and RGB_Converter does the conversion. Check VideoCaptureTest
 *      (mash/video/video-test.cc) for a conversion example.
 *
 * Copyright (c) 1996 Isidor Kouvelas (University College London)
 * Copyright (c) 1995-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef lint
static const char rcsid[] =
    "@(#) $Header: /usr/mash/src/repository/mash/mash-1/render/rgb-converter.cc,v 1.16 2002/02/03 04:15:15 lim Exp $";
#endif

#include "stdio.h"
#include "rgb-converter.h"
#include "bsd-endian.h"

u_int32_t RGB_Converter::r2yuv_[256];
u_int32_t RGB_Converter::g2yuv_[256];
u_int32_t RGB_Converter::b2yuv_[256];

RGB_Converter_422 RGB_Converter_422::instance_;
RGB_Converter_411 RGB_Converter_411::instance_;

RGB_Converter::RGB_Converter(int bpp, u_int8_t *map, int ncol, int useBGR) :
	bpp_(bpp), ncol_(0), useBGR_(useBGR), ymap_(0), umap_(0), vmap_(0), method_(0)
{
	static int init = 0;
	if (init == 0) {
		for (int rgb = 0; rgb < 256; ++rgb) {
			/* can't have overflow in this direction */
			int y = int(0.299 * rgb);
			int u = int(-0.1687 * rgb) & 0xff;
			int v = int(0.5 * rgb);
			r2yuv_[rgb] = y | u << 10 | v << 20;

			y = int(0.587 * rgb);
			u = int(-0.3313 * rgb) & 0xff;
			v = int(-0.4187 * rgb) & 0xff;
			g2yuv_[rgb] = y | u << 10 | v << 20;

			y = int(0.114 * rgb);
			u = int(0.5 * rgb);
			v = int(- 0.0813 * rgb) & 0xff;
			b2yuv_[rgb] = y | u << 10 | v << 20;
		}
		init = 1;
	}

	if(!useBGR) {
		switch (bpp) {
		case 32 :
		        // had to add the &RGB_Converter stuff, does this mess up
			//   virtual functions?  It shouldn't, the old compiler assumed
			//   &RGB_Converter, so it should be ok
			method_ = &RGB_Converter::convert32;
			break;
		case 24:
			method_ = &RGB_Converter::convert24;
			break;
		case 16:
			method_ = &RGB_Converter::convert16;
			ncol_ = 65536;
			break;
		case 8:
			method_ = &RGB_Converter::convert8;
			ncol_ = 256;
			break;
		case 4:
			method_ = &RGB_Converter::convert4;
			ncol_ = 16;
			break;
		case 1:
			method_ = &RGB_Converter::convert1;
			ncol_ = 2;
			break;
		default:
			fprintf(stderr, "RGB_Converter: Unsuported bpp!\n");
			abort();
		}
	}
	else {
		switch (bpp) {
		case 32 :
			method_ = &RGB_Converter::convert32bgr;
			break;
		case 24:
			method_ = &RGB_Converter::convert24bgr;
			break;
		case 16:
			method_ = &RGB_Converter::convert16bgr;
			break;
		case 8:
		case 4:
		case 1:
		default:
			fprintf(stderr, "RGB_Converter: Unsuported bgr bpp!\n");
			abort();
		}
	}

	if (ncol_ > 0) {
		if (ncol > ncol_) {
			fprintf(stderr, "RGB_Converter too many colors %d > %d\n", ncol, ncol_);
			ncol = ncol_;
		}

		u_int32_t	yuv;
		u_int8_t	*yp = ymap_ = new u_int8_t[ncol_];
		u_int8_t	*up = umap_ = new u_int8_t[ncol_];
		u_int8_t	*vp = vmap_ = new u_int8_t[ncol_];

		for (int i = 0; i < ncol; i++) {
			yuv = b2yuv_[*map++];
			yuv += g2yuv_[*map++];
			yuv += r2yuv_[*map++];
			map++;

			*yp++ = yuv;
			*up++ = (yuv >> 10) ^ 0x80;
			*vp++ = (yuv >> 20) ^ 0x80;
		}
	}
}

RGB_Converter::~RGB_Converter()
{
	if (ncol_ > 0) {
		delete [] ymap_;
		delete [] umap_;
		delete [] vmap_;
	}
}

void
RGB_Converter::convert(u_int8_t* p, int w, int h, u_int8_t* frm, int outw, int outh, int invert)
{
	if (outw == 0)
		outw = w;
	if (outh == 0)
		outh = h;
	(this->*method_)(p, w, h, frm, outw, outh, invert);
}


#ifndef min
#define min(a, b) ((a) < (b) ? (a) : (b))
#endif

#define INIT422 \
	if (outw == 0) \
		outw = inw; \
	if (outh == 0) \
		outh = inh; \
\
	u_int8_t* yp = (u_int8_t*)frm; \
	int off = outw * outh; \
	u_int8_t* up = (u_int8_t*)(frm + off); \
	off += off >> 1; \
	u_int8_t* vp = (u_int8_t*)(frm + off); \
\
	u_int32_t* R; \
	u_int32_t* G; \
	u_int32_t* B; \
\
	R = r2yuv_; \
	G = g2yuv_; \
	B = b2yuv_; \
\
	int h = min(inh, outh); \
	int w = min(inw, outw); \
	int inpad = (inw - w) * bpp_ / 8; \
	int outpad = outw - w; \
\
	int stride = inw * bpp_ / 8; \
	if (invert) \
		p += stride * (inh - 1 - (inh - h) / 2); \
	else \
		p += stride * ((inh - h) / 2); \
\
	int outvpad = (outh - h) / 2; \
	yp += outw * outvpad; \
	up += outw / 2 * outvpad; \
	vp += outw / 2 * outvpad; \
\
	/* right side of equal is the multiple of bpp_/8 closest to inpad/2 */ \
	p += ((int)((inw - w) / 2)) * bpp_ / 8; \
\
	yp += outpad >> 1; \
	up += outpad >> 2; \
	vp += outpad >> 2;

#define ADVANCE422 \
	p += inpad; \
	yp += outpad; \
	up += outpad >> 1; \
	vp += outpad >> 1; \
	if (invert) \
		p -= stride << 1;

#define INIT411 \
	if (outw == 0) \
		outw = inw; \
	if (outh == 0) \
		outh = inh; \
\
	u_int8_t* yp = (u_int8_t*)frm; \
	int off = outw * outh; \
	u_int8_t* up = (u_int8_t*)(frm + off); \
	off += off >> 2; \
	u_int8_t* vp = (u_int8_t*)(frm + off); \
\
	u_int32_t* R; \
	u_int32_t* G; \
	u_int32_t* B; \
\
	R = r2yuv_; \
	G = g2yuv_; \
	B = b2yuv_; \
\
	int h = min(inh, outh); \
	int w = min(inw, outw); \
	int inpad = (inw - w) * bpp_ / 8; \
	int outpad = outw - w; \
\
	int stride = inw * bpp_ / 8; \
	int next_line = stride; \
	if (invert) { \
		p += stride * (inh - 1 - (inh - h) / 2); \
		next_line = -next_line; \
	} else \
		p += stride * ((inh - h) / 2); \
\
	/* Hack to get NTSC image to sit at right position within CIF */ \
	int outvpad = ((outh - h) / 2) & (outw > 176? ~0xf: ~0x3); \
	yp += outw * outvpad; \
	up += outw / 4 * outvpad; \
	vp += outw / 4 * outvpad; \
\
	/* right side of equal is the multiple of bpp_/8 closest to inpad/2 */ \
	p += ((int)((inw - w) / 2)) * bpp_ / 8; \
\
	yp += outpad >> 1; \
	up += outpad >> 2; \
	vp += outpad >> 2;

#define ADVANCE411 \
		p += inpad; \
		yp += outw + outpad; \
		up += outpad >> 1; \
		vp += outpad >> 1; \
		if (invert) \
			p -= 3 * stride; \
		else \
			p += stride;

// 32-bit color is defined as follows:
//
//	xBGR xBGR xBGR xBGR // BIG_ENDIAN
//
//	BGRx BGRx BGRx BGRx // ELSE
//
//	FIXME: This seems a bug, but things work.
//
/*FIXME*/
#if BYTE_ORDER == BIG_ENDIAN
#define ROFF 3
#define GOFF 2
#define BOFF 1
#else
#define ROFF 2
#define GOFF 1
#define BOFF 0
#endif

void
RGB_Converter_422::convert32(u_int8_t* p, int inw, int inh, u_int8_t* frm, int outw, int outh, int invert)
{
	INIT422
	while (--h >= 0) {
		for (int x = 0; x < w; x += 2) {
			/*
			 * We use the linearity of the colorspace conversion
			 * to use a compact table for each of the R, G, and
			 * B color components.  Note that we do not have
			 * to fix up overflow because the transform does
			 * not escape the YUV cube in the RGB->YUV direction.
			 * Moreover, we do not need to worry about overflow
			 * between the y,u, and v components affecting
			 * eachother (in the parallel add) because there
			 * are two empty bits between each component
			 * so we can survive two overflows.
			 */
			u_int32_t yuv = R[p[ROFF]];
			yuv += G[p[GOFF]];
			yuv += B[p[BOFF]];
			p += 4;

			/*
			 * Flip the high bit on the chrominance because
			 * the encoder expects them in the range 0-255.
			 */
			*yp++ = yuv;
			*up++ = (yuv >> 10) ^ 0x80;
			*vp++ = (yuv >> 20) ^ 0x80;

			yuv = R[p[ROFF]];
			yuv += G[p[GOFF]];
			yuv += B[p[BOFF]];
			p += 4;

			*yp++ = yuv;
		}
		ADVANCE422
	}
}

void
RGB_Converter_422::convert32bgr(u_int8_t* p, int inw, int inh, u_int8_t* frm, int outw, int outh, int invert)
{
	INIT422
	while (--h >= 0) {
		for (int x = 0; x < w; x += 2) {
			/*
			 * We use the linearity of the colorspace conversion
			 * to use a compact table for each of the R, G, and
			 * B color components.  Note that we do not have
			 * to fix up overflow because the transform does
			 * not escape the YUV cube in the RGB->YUV direction.
			 * Moreover, we do not need to worry about overflow
			 * between the y,u, and v components affecting
			 * eachother (in the parallel add) because there
			 * are two empty bits between each component
			 * so we can survive two overflows.
			 */
			u_int32_t yuv = R[p[BOFF]];
			yuv += G[p[GOFF]];
			yuv += B[p[ROFF]];
			p += 4;

			/*
			 * Flip the high bit on the chrominance because
			 * the encoder expects them in the range 0-255.
			 */
			*yp++ = yuv;
			*up++ = (yuv >> 10) ^ 0x80;
			*vp++ = (yuv >> 20) ^ 0x80;

			yuv = R[p[BOFF]];
			yuv += G[p[GOFF]];
			yuv += B[p[ROFF]];
			p += 4;

			*yp++ = yuv;
		}
		ADVANCE422
	}
}

void RGB_Converter_411::convert32(u_int8_t* p, int inw, int inh, u_int8_t* frm, int outw, int outh, int invert)
{
	INIT411
	for (h >>= 1; --h >= 0; ) {
		for (int x = 0; x < w; x += 2) {
			u_int32_t yuv = R[p[ROFF]];
			yuv += G[p[GOFF]];
			yuv += B[p[BOFF]];

			yp[0] = yuv;
			*up++ = (yuv >> 10) ^ 0x80;
			*vp++ = (yuv >> 20) ^ 0x80;

			yuv = R[p[next_line + ROFF]];
			yuv += G[p[next_line + GOFF]];
			yuv += B[p[next_line + BOFF]];
			yp[outw] = yuv;
			p += 4;
			++yp;

			yuv = R[p[ROFF]];
			yuv += G[p[GOFF]];
			yuv += B[p[BOFF]];
			yp[0] = yuv;

			yuv = R[p[next_line + ROFF]];
			yuv += G[p[next_line + GOFF]];
			yuv += B[p[next_line + BOFF]];
			yp[outw] = yuv;
			p += 4;
			++yp;
		}
		ADVANCE411
	}
}

void RGB_Converter_411::convert32bgr(u_int8_t* p, int inw, int inh, u_int8_t* frm, int outw, int outh, int invert)
{
	INIT411
	for (h >>= 1; --h >= 0; ) {
		for (int x = 0; x < w; x += 2) {
			u_int32_t yuv = R[p[BOFF]];
			yuv += G[p[GOFF]];
			yuv += B[p[ROFF]];

			yp[0] = yuv;
			*up++ = (yuv >> 10) ^ 0x80;
			*vp++ = (yuv >> 20) ^ 0x80;

			yuv = R[p[next_line + BOFF]];
			yuv += G[p[next_line + GOFF]];
			yuv += B[p[next_line + ROFF]];
			yp[outw] = yuv;
			p += 4;
			++yp;

			yuv = R[p[BOFF]];
			yuv += G[p[GOFF]];
			yuv += B[p[ROFF]];
			yp[0] = yuv;

			yuv = R[p[next_line + BOFF]];
			yuv += G[p[next_line + GOFF]];
			yuv += B[p[next_line + ROFF]];
			yp[outw] = yuv;
			p += 4;
			++yp;
		}
		ADVANCE411
	}
}

#undef ROFF
#undef GOFF
#undef BOFF

// The input frame is u_int8_t*, so there's no place for endianess.
//	24-bit color is defined as follows:
//
//	RGB RGB RGB RGB RGB
//
#define ROFF 0
#define GOFF 1
#define BOFF 2

void RGB_Converter_422::convert24(u_int8_t* p, int inw, int inh, u_int8_t* frm, int outw, int outh, int invert)
{
	INIT422
	while (--h >= 0) {
		for (int x = 0; x < w; x += 2) {
			u_int32_t yuv = R[p[ROFF]];
			yuv += G[p[GOFF]];
			yuv += B[p[BOFF]];
			p += 3;

			*yp++ = yuv;
			*up++ = (yuv >> 10) ^ 0x80;
			*vp++ = (yuv >> 20) ^ 0x80;

			yuv = R[p[ROFF]];
			yuv += G[p[GOFF]];
			yuv += B[p[BOFF]];
			p += 3;

			*yp++ = yuv;
		}
		ADVANCE422
	}
}

void RGB_Converter_422::convert24bgr(u_int8_t* p, int inw, int inh, u_int8_t* frm, int outw, int outh, int invert)
{
	INIT422
	while (--h >= 0) {
		for (int x = 0; x < w; x += 2) {
			u_int32_t yuv = R[p[BOFF]];
			yuv += G[p[GOFF]];
			yuv += B[p[ROFF]];
			p += 3;

			*yp++ = yuv;
			*up++ = (yuv >> 10) ^ 0x80;
			*vp++ = (yuv >> 20) ^ 0x80;

			yuv = R[p[BOFF]];
			yuv += G[p[GOFF]];
			yuv += B[p[ROFF]];
			p += 3;

			*yp++ = yuv;
		}
		ADVANCE422
	}
}

void RGB_Converter_411::convert24(u_int8_t* p, int inw, int inh, u_int8_t* frm, int outw, int outh, int invert)
{
	INIT411
	for (h >>= 1; --h >= 0; ) {
		for (int x = 0; x < w; x += 2) {
			u_int32_t yuv = R[p[ROFF]];
			yuv += G[p[GOFF]];
			yuv += B[p[BOFF]];

			yp[0] = yuv;
			*up++ = (yuv >> 10) ^ 0x80;
			*vp++ = (yuv >> 20) ^ 0x80;

			yuv = R[p[next_line + ROFF]];
			yuv += G[p[next_line + GOFF]];
			yuv += B[p[next_line + BOFF]];
			yp[outw] = yuv;
			p += 3;
			++yp;

			yuv = R[p[ROFF]];
			yuv += G[p[GOFF]];
			yuv += B[p[BOFF]];
			yp[0] = yuv;

			yuv = R[p[next_line + ROFF]];
			yuv += G[p[next_line + GOFF]];
			yuv += B[p[next_line + BOFF]];
			yp[outw] = yuv;
			p += 3;
			++yp;
		}
		ADVANCE411
	}
}

void RGB_Converter_411::convert24bgr(u_int8_t* p, int inw, int inh, u_int8_t* frm, int outw, int outh, int invert)
{
	INIT411
	for (h >>= 1; --h >= 0; ) {
		for (int x = 0; x < w; x += 2) {
			u_int32_t yuv = R[p[BOFF]];
			yuv += G[p[GOFF]];
			yuv += B[p[ROFF]];

			yp[0] = yuv;
			*up++ = (yuv >> 10) ^ 0x80;
			*vp++ = (yuv >> 20) ^ 0x80;

			yuv = R[p[next_line + BOFF]];
			yuv += G[p[next_line + GOFF]];
			yuv += B[p[next_line + ROFF]];
			yp[outw] = yuv;
			p += 3;
			++yp;

			yuv = R[p[BOFF]];
			yuv += G[p[GOFF]];
			yuv += B[p[ROFF]];
			yp[0] = yuv;

			yuv = R[p[next_line + BOFF]];
			yuv += G[p[next_line + GOFF]];
			yuv += B[p[next_line + ROFF]];
			yp[outw] = yuv;
			p += 3;
			++yp;
		}
		ADVANCE411
	}
}

// 16-bit color is defined as follows:
//
//	           5-bit red             5-bit green         5-bit blue
//	    |---------------------|  |----------------|  |----------------|
//	x16 x15 x14 x13 x12 x11 x10  x9  x8  x7  x6  x5  x4  x3  x2  x1  x0
//
#define RED(x)		(((x) >> 7) & 0xf8)
#define GREEN(x)	(((x) >> 2) & 0xf8)
#define BLUE(x)		(((x) << 3) & 0xf8)

void RGB_Converter_422::convert16(u_int8_t* p, int inw, int inh, u_int8_t* frm, int outw, int outh, int invert)
{
	INIT422
	u_int16_t* sp;
	while (--h >= 0) {
		sp = (u_int16_t *)p;
		for (int x = 0; x < w; x += 2) {
			u_int32_t yuv = R[RED(*sp)];
			yuv += G[GREEN(*sp)];
			yuv += B[BLUE(*sp)];
			sp++;

			*yp++ = yuv;
			*up++ = (yuv >> 10) ^ 0x80;
			*vp++ = (yuv >> 20) ^ 0x80;

			yuv = R[RED(*sp)];
			yuv += G[GREEN(*sp)];
			yuv += B[BLUE(*sp)];
			sp++;

			*yp++ = yuv;
		}
		p = (u_int8_t *)sp;
		ADVANCE422
	}
}

void RGB_Converter_422::convert16bgr(u_int8_t* p, int inw, int inh, u_int8_t* frm, int outw, int outh, int invert)
{
	INIT422
	u_int16_t* sp;
	while (--h >= 0) {
		sp = (u_int16_t *)p;
		for (int x = 0; x < w; x += 2) {
			u_int32_t yuv = R[BLUE(*sp)];
			yuv += G[GREEN(*sp)];
			yuv += B[RED(*sp)];
			sp++;

			*yp++ = yuv;
			*up++ = (yuv >> 10) ^ 0x80;
			*vp++ = (yuv >> 20) ^ 0x80;

			yuv = R[BLUE(*sp)];
			yuv += G[GREEN(*sp)];
			yuv += B[RED(*sp)];
			sp++;

			*yp++ = yuv;
		}
		p = (u_int8_t *)sp;
		ADVANCE422
	}
}

void RGB_Converter_411::convert16(u_int8_t* p, int inw, int inh, u_int8_t* frm, int outw, int outh, int invert)
{
	INIT411
	next_line /= 2;
	u_int16_t* sp;
	for (h >>= 1; --h >= 0; ) {
		sp = (u_int16_t *)p;
		for (int x = 0; x < w; x += 2) {
			u_int32_t yuv = R[RED(*sp)];
			yuv += G[GREEN(*sp)];
			yuv += B[BLUE(*sp)];

			yp[0] = yuv;
			*up++ = (yuv >> 10) ^ 0x80;
			*vp++ = (yuv >> 20) ^ 0x80;

			yuv = R[RED(sp[next_line])];
			yuv += G[GREEN(sp[next_line])];
			yuv += B[BLUE(sp[next_line])];
			yp[outw] = yuv;
			sp++;
			++yp;

			yuv = R[RED(*sp)];
			yuv += G[GREEN(*sp)];
			yuv += B[BLUE(*sp)];
			yp[0] = yuv;

			yuv = R[RED(sp[next_line])];
			yuv += G[GREEN(sp[next_line])];
			yuv += B[BLUE(sp[next_line])];
			yp[outw] = yuv;
			sp++;
			++yp;
		}
		p = (u_int8_t *)sp;
		ADVANCE411
	}
}

void RGB_Converter_411::convert16bgr(u_int8_t* p, int inw, int inh, u_int8_t* frm, int outw, int outh, int invert)
{
	INIT411
	next_line /= 2;
	u_int16_t* sp;
	for (h >>= 1; --h >= 0; ) {
		sp = (u_int16_t *)p;
		for (int x = 0; x < w; x += 2) {
			u_int32_t yuv = R[BLUE(*sp)];
			yuv += G[GREEN(*sp)];
			yuv += B[RED(*sp)];

			yp[0] = yuv;
			*up++ = (yuv >> 10) ^ 0x80;
			*vp++ = (yuv >> 20) ^ 0x80;

			yuv = R[BLUE(sp[next_line])];
			yuv += G[GREEN(sp[next_line])];
			yuv += B[RED(sp[next_line])];
			yp[outw] = yuv;
			sp++;
			++yp;

			yuv = R[BLUE(*sp)];
			yuv += G[GREEN(*sp)];
			yuv += B[RED(*sp)];
			yp[0] = yuv;

			yuv = R[BLUE(sp[next_line])];
			yuv += G[GREEN(sp[next_line])];
			yuv += B[RED(sp[next_line])];
			yp[outw] = yuv;
			sp++;
			++yp;
		}
		p = (u_int8_t *)sp;
		ADVANCE411
	}
}

void RGB_Converter_422::convert8(u_int8_t* p, int inw, int inh, u_int8_t* frm, int outw, int outh, int invert)
{
	INIT422
	while (--h >= 0) {
		for (int x = 0; x < w; x += 2) {
			*yp++ = ymap_[*p];
			*up++ = umap_[*p];
			*vp++ = vmap_[*p];
			p++;

			*yp++ = ymap_[*p];
			p++;
		}
		ADVANCE422
	}
}

void RGB_Converter_411::convert8(u_int8_t* p, int inw, int inh, u_int8_t* frm, int outw, int outh, int invert)
{
	INIT411
	for (h >>= 1; --h >= 0; ) {
		for (int x = 0; x < w; x += 2) {
			yp[0] = ymap_[*p];
			*up++ = umap_[*p];
			*vp++ = vmap_[*p];
			yp[outw] = ymap_[p[next_line]];
			p++;
			++yp;
			yp[0] = ymap_[*p];
			yp[outw] = ymap_[p[next_line]];
			p++;
			++yp;
		}
		ADVANCE411
	}
}

void RGB_Converter_422::convert4(u_int8_t* p, int inw, int inh, u_int8_t* frm, int outw, int outh, int invert)
{
	INIT422
	while (--h >= 0) {
		for (int x = 0; x < w; x += 2) {
			int v = *p >> 4;
			*yp++ = ymap_[v];
			*up++ = umap_[v];
			*vp++ = vmap_[v];

			*yp++ = ymap_[*p & 0xf];
			p++;
		}
		ADVANCE422
	}
}

void RGB_Converter_411::convert4(u_int8_t* p, int inw, int inh, u_int8_t* frm, int outw, int outh, int invert)
{
	INIT411
	for (h >>= 1; --h >= 0; ) {
		for (int x = 0; x < w; x += 2) {
			int v = *p >> 4;
			yp[0] = ymap_[v];
			*up++ = umap_[v];
			*vp++ = vmap_[v];
			yp[outw] = ymap_[*(p + next_line) >> 4];

			++yp;
			yp[0] = ymap_[*p & 0xf];
			yp[outw] = ymap_[p[next_line] & 0xf];
			++yp;

			p++;
		}
		ADVANCE411
	}
}

void RGB_Converter_422::convert1(u_int8_t* p, int inw, int inh, u_int8_t* frm, int outw, int outh, int invert)
{
	INIT422
	while (--h >= 0) {
		for (int x = 0; x < w; x += 8) {
			int v = *p;
			for (int i = 8; i > 0 ; i -= 2) {
				if (v & 0x80) {
					*yp++ = *ymap_;
					*up++ = *umap_;
					*vp++ = *vmap_;
				} else {
					*yp++ = ymap_[1];
					*up++ = umap_[1];
					*vp++ = vmap_[1];
				}

				if (v & 0x40)
					*yp++ = *ymap_;
				else
					*yp++ = ymap_[1];

				v <<= 2;
			}
			p++;
		}
		ADVANCE422
	}
}

void RGB_Converter_411::convert1(u_int8_t* p, int inw, int inh, u_int8_t* frm, int outw, int outh, int invert)
{
	INIT411
	for (h >>= 1; --h >= 0; ) {
		for (int x = 0; x < w; x += 2) {
			int v = *p;
			int vn = p[next_line];
			for (int i = 8; i > 0 ; i -= 2) {
				if (v & 0x80) {
					*yp = *ymap_;
					*up++ = *umap_;
					*vp++ = *vmap_;
				} else {
					*yp = ymap_[1];
					*up++ = umap_[1];
					*vp++ = vmap_[1];
				}

				if (vn & 0x80)
					yp[outw] = *ymap_;
				else
					yp[outw] = ymap_[1];
				++yp;

				if (v & 0x40)
					*yp = *ymap_;
				else
					*yp = ymap_[1];

				if (vn & 0x40)
					yp[outw] = *ymap_;
				else
					yp[outw] = ymap_[1];
				++yp;

				v <<= 2;
				vn <<= 2;
			}
			p++;
		}
		ADVANCE411
	}
}

