# tgw-ui.tcl --
#
#       This file has the main ui component of the tgw application. Also, it
#       contains many of the slightly modified superclasses to make things
#       work... This is the main file with all the code in it.
#
# Copyright (c) 2000-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


import Switcher \
	ActiveSourceManager \
	CuesReceiver VisualFrame ActiveSource AudioAgent RealServerUI H261TranscoderAddrInput RedirUI RealConfigUI OutgoingReal OutgoingCombiner VideoAgent OutgoingH261 AddressBlock/Simple Network OutgoingRedir Observer

# this is the file that defines the UI and the member objects for TGW. Actually, everything is implemented in here.


# This is a subclass of ActiveSourceManager that basically just changes the UI that it uses to display the streams so that we can manipulate it to add new options.

Class ActiveSourceManagerTGW -superclass ActiveSourceManager

ActiveSourceManagerTGW public really_activate src {
	$self instvar grid_ curcol_ currow_ ui_ list_direction_ localChannel_

	set as [new ActiveSource $self $ui_ $grid_.$src $src $localChannel_]

	grid $grid_.$src -row $currow_ -column $curcol_ -sticky we

    #inserted by Tim to attach the stream and also make the GUI
	$ui_ attach_stream $src $self $grid_ $currow_
    #end of insertion

	if { $list_direction_ == "vertical" } {
		grid columnconfigure $grid_ $curcol_ -weight 1
	} else {
		grid rowconfigure $grid_ $currow_ -weight 1
	}

	$ui_ update_decoder $src

	$self bump

	#
	# Someone became activated, so we have to change
	# the switchable menu to include this participant
	#
	Switcher rebuild_switch_list_menu

	# XXX this is a temporary hack while cindy works on the VideoBox class...
	if {[$ui_ info class] == "MuiUI"} {
		$ui_ maybe_switch_in $src
	}
}

# Adds the "source" title at the top of the sources column
ActiveSourceManagerTGW public add_active { as src } {
        $self instvar active_ grid_ currow_ curcol_ tgwTitleCol_
	if { [array size active_] == 0 } {
	    $self next $as $src
	    frame $grid_.title(0) -relief sunken -borderwidth 1
	    label $grid_.title(0).label -text Sources -font [$self get_option smallfont] -relief flat
	    pack $grid_.title(0).label
	    grid $grid_.title(0) -row 0 -column 0 -sticky we
	    grid columnconfigure $grid_ 0 -weight 1
	    $self bump
	} else {
	    $self next $as $src
	}
}

# this is a subclass of AudioAgent that we need to create so that the app doesn't use the soundcard that is actually on the machine, but instead it uses the virtual soundcard that will give the sample to the RealAudio encoder.

Class AudioAgentTGW -superclass AudioAgent

AudioAgentTGW public init { app spec RealEnc {callback {}} } {
    $self instvar deviceName audioDevice_
    set deviceName "Audio/RealAudioVirtualDevice"
    $self next $app $spec $callback
    #link the output of the virtual device to go to the RealEncoder
    set temp [$audioDevice_ set audio_]
    $temp linkEncoder $RealEnc
}

# This is the main UI for TGW

Class TGWUI -superclass Observer -configuration {
	minwidth 200
	minheight 100
	filterGain 0.25
	geometry {}
	vain false
}

TGWUI instproc window-title { prefix name } {
	$self instvar name_ prefix_
	set name_ $name
	set prefix_ $prefix

	wm iconname . "$prefix_ $name_"
	wm title . "$prefix_ $name_"

}


# Build the user-interface.
TGWUI instproc init { w localChannel globalChannel videoAgent vpipe exitCmd useUI rServer rStream } {
	$self next 
	$self instvar localChannel_ globalChannel_ videoAgent_ vpipe_ exitCmd_ 
	set localChannel_ $localChannel
	set globalChannel_ $globalChannel
	set videoAgent_ $videoAgent
	set vpipe_ $vpipe
	set exitCmd_ "$exitCmd"

        # let this ui be an observer of the VideoAgent so it can catch "trigger_sdes"
	$videoAgent attach $self

	$self build_gui $w

	$self set-geometry

}

TGWUI instproc reset {} {
	$self new_hostspec
}

TGWUI instproc build_gui { w } {
	$self instvar videoAgent_ userwindows_ path_ vpipe_ asm_ localChannel_ globalChannel_

	$self set_rate_vars [$videoAgent_ set session_]

	$self instvar id_
	set id_ [after 1000 "$self periodic_update"]

	#
	# Configure the Switcher class to ignore attempts to switch
	# to the local source, since this is not interesting (unless
	# we set the "vain" configuration option to true).
	#
	if { ![$self yesno vain] && [$videoAgent_ have_network] } {
		Switcher set ignore_([$videoAgent_ local]) 1
	}

	$self layout_gui $w

	#
	# Local coordination bus events
	# - receive a focus speaker message from vat running on the
	#   same machine as this vic
	#
	set cb $localChannel_
	if { $cb != "" } {
		$cb register FOCUS_SPEAKER "$asm_ focus_speaker"
	}

	#
	# Global coordination bus events
	# - receive a aware {ear,hand,yes,no} message from vat
	#   running on others machine
	# - receive an unaware of the same types
	#
	# CuesReceiver handle these events, this is created only
	# when the user specified to do so on the command line
	#
	set cb $globalChannel_
	if { [$self yesno useCues] && $cb!="" }  {
		CuesReceiver set_cb $cb
	}
}

# Given widget <i>w</i>, create frame $w.top, initializing its visual and colormap.
# Within this frame, a frame ($w.top.f) is created for holding either a label (instvar label_)
# or another frame (instvar grid_).
# A few keybinbdings are assigned.
#

TGWUI public layout_gui {w} {
	$self instvar asm_ controlMenu_ videoAgent_ vpipe_ vframe_ exitCmd_ localChannel_

	set vframe_ [new VisualFrame $w.top]

	set asm_ [new ActiveSourceManagerTGW $self $w.top $videoAgent_ vertical $localChannel_]
        $vframe_ attach_observer $asm_
	foreach i { 1 2 3 4 } {
		bind . <Key-$i> "$asm_ redecorate $i"
	}

	pack $w.top -expand 1 -fill both

	$self build_menubar $w.top.bar
	pack $w.top.bar -fill x -side bottom

	$asm_ init_grid $w.top.f
	pack $w.top.f -expand 1 -fill both -side top
}

# Size the "." window using values from the resource database:
# <i>minwidth</i>, <i>minheight</i>, and <i>geometry</i> (all in
# pixels).  Specify <i>geometry</i> in the form wXh or set it to empty
# string, {}, to cancel the existing user-specified geometry and revert
# to the size requested internally by its widgets.
#
TGWUI instproc set-geometry {} {
	#
	# Withdraw window so that user-placement is deferred
	# until after initial geometry is computed
	#
	global mash
	if { ![info exists mash(environ)] || $mash(environ)!="mplug" } {
	    wm withdraw .
	}
	wm geometry . [$self get_option geometry]
	update idletasks
	set minwidth [winfo reqwidth .]
	set minheight [winfo reqheight .]
	#XXX
	if { $minwidth < [$self get_option minwidth] } {
		set minwidth [$self get_option minwidth]
	}
	if { $minheight < [$self get_option minheight] } {
		set minheight [$self get_option minheight]
	}
	wm minsize . $minwidth $minheight
	if { ![info exists mash(environ)] || $mash(environ)!="mplug" } {
		wm deiconify .
	}
}

#
# Every second, update rate variables.
#
TGWUI instproc periodic_update { } {
	$self instvar videoAgent_ vpipe_ id_
	if [$vpipe_ running] {
		update_rate [$videoAgent_ set session_]
	}
	update idletasks
	set id_ [after 1000 "$self periodic_update"]
}

#
TGWUI instproc clean_timers { } {
	$self cancel_periodic_update
}

#
# Cancels the periodic_update.
#
TGWUI instproc cancel_periodic_update { } {
        $self instvar id_
        if [info exists id_] {
                after cancel $id_
                unset id_
        }
}

# Update the source description by updating the element of the global
# arrays src_info(), src_nickname(), and src_name() indexed by the
# specified <i>src</i>.  If the src_name() is changed, then this is
# reflected in the UserWindows in which this src appears.
#
TGWUI instproc trigger_sdes src {
	$self instvar asm_

	global src_info src_nickname src_name
	#
	# Figure out best presentation from available information.
	#
	set name [$src sdes name]
	set cname [$src sdes cname]
	set addr [$src addr]
	if { $name == "" } {
		if { $cname == "" } {
			set src_nickname($src) $addr
			set info $addr/[$src format_name]

		} else {
			set src_nickname($src) $cname
			set info "$addr/[$src format_name]"
		}
	} elseif [cname_redundant $name $cname] {
		set src_nickname($src) $name
		set info $addr/[$src format_name]
	} else {
		set src_nickname($src) $name
		set info $cname/[$src format_name]
	}
	set msg [$src sdes note]
	if { $msg != "" } {
		set info $msg
	}
	set src_info($src) $info

	# only call change_name when name really changes
	if { ![info exists src_name($src)] || "$src_name($src)" != "$name" } {
		set src_name($src) $name
		$asm_ change_name $src
	}
}

#
TGWUI instproc trigger_media src {}

#
# For the source specified by <i>src</i>, update its rate variables and its source description.
#
TGWUI instproc update_decoder src {
	$self set_rate_vars $src
	$self trigger_sdes $src
}

TGWUI instproc set_rate_vars {src} {
	global fpshat bpshat lhat shat
	if [info exists fpshat($src)] {
		unset fpshat($src)
		unset bpshat($src)
		unset lhat($src)
		unset shat($src)
	}
	set gain [$self get_option filterGain]
	set fpshat($src) 0
	rate_variable fpshat($src) $gain
	set bpshat($src) 0
	rate_variable bpshat($src) $gain
	set lhat($src) 0
	rate_variable lhat($src) $gain
	set shat($src) 0
	rate_variable shat($src) $gain
}

# An accessor function for a ControlMenu tkvar value.
#
TGWUI public use_hw_decode {} {
	$self instvar controlMenu_
	#return [$controlMenu_ use-hw]
    return 0
}

#
# An accessor function for a ControlMenu tkvar value.
#
TGWUI public mute_new_sources {} {
	$self instvar controlMenu_
	#return [$controlMenu_ mute-new-sources]
    return 0
}

TGWUI instproc build_menubar { w } {
    $self instvar newRealScreen_ newH261Screen_ newRedirScreen_ configReal_
    set newRealScreen_ [new RealServerUI $self]
    set newH261Screen_ [new H261TranscoderAddrInput $self]
    set newRedirScreen_ [new RedirUI $self]
    set configReal_ [new RealConfigUI $self]

    frame $w -relief ridge -borderwidth 2
    label $w.title -text "[$self get_option appname] v[version]" -font [$self get_option smallfont] \
	    -relief flat -justify left
    button $w.quit -text Quit -relief raised \
	    -font [$self get_option smallfont] -command "$self shutdown" \
	    -highlightthickness 1
    menubutton $w.new -text " New Transmission " -relief raised \
	    -font [$self get_option smallfont] -highlightthickness 1 \
	    -menu $w.new.menu

    set tempmenu [menu $w.new.menu -tearoff 0]
    $tempmenu add command -label "New Bandwidth Limiter" -command "$self showNewH261screen"
    $tempmenu add command -label "New Redirector" -command "$self showNewRedirector"

    set classList [Class info instances]
    if { [lsearch -exact $classList "RealWindow"] == -1 } {
	$tempmenu add command -label "New Real Transcoded Broadcast" -command "$self showNewRealScreen" -state disabled
	$tempmenu add command -label "New Real Transcoded File" -command "$self newRealSession \"\" \"\" \"\" \"\"" -state disabled
	$tempmenu add command -label "New Stream Combiner" -command "$self showNewRealScreen combine" -state disabled
    } else {
	$tempmenu add command -label "New Real Transcoded Broadcast" -command "$self showNewRealScreen"
	$tempmenu add command -label "New Real Transcoded File" -command "$self newRealSession \"\" \"\" \"\" \"\""
	$tempmenu add command -label "New Stream Combiner" -command "$self showNewRealScreen combine"
    }
    pack $w.title -side left -fill both -expand 1
    pack $w.new $w.quit -side left -padx 1 -pady 1
    pack $w.new -fill y -expand 1
}

TGWUI instproc shutdown { } {
    $self instvar exitCmd_
# make sure all transcoding is stopped so files can be saved and post-processed
    eval $exitCmd_
}

#
# This is the function that gets called every time a new source appears (as input)
#

TGWUI instproc attach_stream { src sourceManager grid rownum } {
    $self instvar managers_ sources_ windows_ source_count_ grid_

    if ![info exists source_count_] {
	set source_count_ 0
    }
#add it to the arrays
    set sources_($source_count_) $src
    set managers_($src) $sourceManager
    set windows_($source_count_) $rownum
    set grid_ $grid
    incr source_count_

    $self updateUI
}

#
# This is the beginning of the code needed to transcode to Real.
# showNewRealScreen toggles the UI that asks for the server name, username, etc...
# newRealSession will just add a new column on the matrix.
#

TGWUI instproc showNewRealScreen { {combine {}} } {
    $self instvar realSessions_ newRealScreen_
    $newRealScreen_ toggle $combine
}

TGWUI instproc newRealSession { serverName serverPort userName password } {
    $self instvar outgoingSessions_

    if ![info exists outgoingSessions_] {
	set outgoingSessions_(0) [new OutgoingReal $serverName $serverPort $userName $password]
    } else {
	set nextCount [array size outgoingSessions_]
	set outgoingSessions_($nextCount) [new OutgoingReal $serverName $serverPort $userName $password]
    }
    $self addTitle "RN Transcode: $serverName"
    $self updateUI
}

# this is the class that makes a combiner stream
TGWUI instproc newCombiner { serverName serverPort userName password } {
    $self instvar outgoingSessions_

    set rend [new Combiner]

    if ![info exists outgoingSessions_] {
	set outgoingSessions_(0) [new OutgoingCombiner $rend $serverName $serverPort $userName $password]
    } else {
	set nextCount [array size outgoingSessions_]
	set outgoingSessions_($nextCount) [new OutgoingCombiner $rend $serverName $serverPort $userName $password]
    }
    $self addTitle "RN Combiner: $serverName"
    $self updateUI
}


# puts the title on top of the new column... used by all transcoders
TGWUI instproc addTitle { name } {
    $self instvar grid_ tgwTitleCol_
    if ![info exists tgwTitleCol_] {
	set tgwTitleCol_ 1
    }
    frame $grid_.title($tgwTitleCol_) -relief sunken -borderwidth 2
    label $grid_.title($tgwTitleCol_).label -text $name -font [$self get_option smallfont] -relief flat
    pack $grid_.title($tgwTitleCol_).label 
    grid $grid_.title($tgwTitleCol_) -row 0 -column $tgwTitleCol_ -sticky we
    incr tgwTitleCol_
}

# updateUI refreshes the matrix to make sure that there
# is a widget everywhere there should be. All transcoders use this.

TGWUI instproc updateUI { } {
    $self instvar windows_ sources_ outgoingSessions_ grid_

    if ![info exists outgoingSessions_] {
	return
    }

    foreach src [array names windows_] {
	for { set sessionNum 0 } { $sessionNum < [array size outgoingSessions_] } { incr sessionNum } {
	    if {[llength [info commands $grid_.$windows_($src)_outgoing($sessionNum)]] == 0} {
		frame $grid_.$windows_($src)_outgoing($sessionNum) -relief sunken -borderwidth 2
		$outgoingSessions_($sessionNum) makeUI $grid_.$windows_($src)_outgoing($sessionNum) $self $sources_($src) $windows_($src)
		grid $grid_.$windows_($src)_outgoing($sessionNum) -row $windows_($src) -column [expr $sessionNum + 1] -sticky nsew
	    }
	}
    }
}

#
# This will set up a new session on which we transcode to H261
#

TGWUI instproc showNewH261screen { } {
    $self instvar newH261Screen_
    $newH261Screen_ toggle
}

TGWUI instproc newH261Session { addr maxBW } {
    $self instvar outgoingSessions_ auction_houses_ out_vidAgents_

    if [info exists auction_houses_($addr)] {
	puts "we already transcode to that address"
    } else {
	set out_vidAgents_($addr) [new VideoAgent $self $addr]
	set osess [$out_vidAgents_($addr) get_transmitter]
	$osess loopback-layer -1
#	        $osess set rate_control_ 1
#	        $osess set txonly_ 1
#	        $osess data-bandwidth 128000
	set auction_houses_($addr) [new BidderHouse $maxBW]
	$auction_houses_($addr) hold_bid
	$auction_houses_($addr) update_bw

	if ![info exists outgoingSessions_] {
	    set outgoingSessions_(0) [new OutgoingH261 $osess $maxBW $auction_houses_($addr) $addr]
	} else {
	    set nextCount [array size outgoingSessions_]
	    set outgoingSessions_($nextCount) [new OutgoingH261 $osess $maxBW $auction_houses_($addr) $addr]
	}
    } 
    $self addTitle "B/W limiter: $addr"
    $self updateUI    
}


# for redirection, or tunnels

TGWUI instproc showNewRedirector { } {
    $self instvar newRedirScreen_
    $newRedirScreen_ toggle
}

TGWUI instproc newRedirSession { fwVid fwAud vidDest audSrc audDest } {
    $self instvar outgoingSessions_ videoAgent_

    if { $fwVid == 1 } {
	set vidRedir [new Redirector 1024]
	set ab [new AddressBlock/Simple $vidDest]
	set addr [$ab addr 0]
	set sport [$ab sport 0]
	set rport [$ab rport 0]
	set ttl [$ab ttl 0]
	set n [new Network]
	$n open $addr $sport $rport $ttl
	$n loopback 1
	$vidRedir add-sink $n
	delete $ab

	set n [new Network]
	set src_net [$videoAgent_ network]
	$n open [$src_net addr] [$src_net sport] [$src_net rport] [$src_net ttl]
	$n loopback 1    
	$vidRedir add-source-net $n
    } else {
	set vidRedir ""
    }

    if { $fwAud == 1 } {
	# currently audio forwarding is not supported... couldn't get it to work.
	set audRedir [new Redirector 1024]
	set ab [new AddressBlock/Simple $audDest]
	set addr [$ab addr 0]
	set sport [$ab sport 0]
	set rport [$ab rport 0]
	set ttl [$ab ttl 0]
	set n [new Network]
	$n open $addr $sport $rport $ttl
	$n loopback 1
	set temp [$n addr]
	$audRedir add-sink $n
	delete $ab

	set ab [new AddressBlock/Simple $audSrc]
	set addr [$ab addr 0]
	set sport [$ab sport 0]
	set rport [$ab rport 0]
	set ttl [$ab ttl 0]
	set n [new Network]
	$n open $addr $sport $rport $ttl
	$n loopback 1    
	set temp [$n addr]
	$audRedir add-source-net $n
	delete $ab
    } else {
	set audRedir ""
    }

    if ![info exists outgoingSessions_] {
	set outgoingSessions_(0) [new OutgoingRedir $fwVid $fwAud $vidRedir $vidDest $audRedir $audDest]
    } else {
	set nextCount [array size outgoingSessions_]
	set outgoingSessions_($nextCount) [new OutgoingRedir $fwVid $fwAud $vidRedir $vidDest $audRedir $audDest]
    }
    $self addTitle "Redirect: $vidDest"
    $self updateUI    
}

    
