# desc-stream.tcl --
#
#       Provides base abstraction of a stream in the Broadcast Description
#       (broadcast on/off, duration, etc).  Derived classes specialized for
#       RVC and RealNetworks streams.
#
# Copyright (c) 2000-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

Import enable

Class BroadcastDescription_Stream

# base class does nothing

BroadcastDescription_Stream instproc init {name} {
    $self instvar broadcast_ source_ clock_ startTime_ callback_ name_

    set broadcast_ "false"
    set source_ "null"
    set startTime_ 0
    set callback_ ""
    set name_ $name
}

BroadcastDescription_Stream public broadcast {{val default}} {
    return "false"
}

BroadcastDescription_Stream public duration {{val default}} {
    return 0
}

BroadcastDescription_Stream public source {{val default}} {
    return "null"
}

BroadcastDescription_Stream public notifyCallback {val} {
    $self instvar callback_

    set callback_ $val
}

BroadcastDescription_Stream public name {{val default}} {
    $self instvar name_

    if {$val != "default"} {
	set name_ $val
    }
    return $name_
}

################################################

Class BroadcastDescription_ReplayStream -superclass BroadcastDescription_Stream

################################################

Class BroadcastDescription_RvcStream -superclass BroadcastDescription_Stream

BroadcastDescription_RvcStream instproc init {name hostname} {
    $self instvar broadcast_ source_ sourceStartTime_
    $self instvar hostname_ shortHostname_

    $self next $name
    set hostname_ $hostname

    set parts [split $hostname_ "."]
    set shortHostname_ [lindex $parts 0]
    set source_ [$BroadcastDescription::405Control matrix_getInputSource $shortHostname_]
    set callback "$self switchCallback"
    set filter [$self getFilter $shortHostname_]
    $BroadcastDescription::405Control callback_register $callback $filter
    $BroadcastDescription::405Control callback_enable $callback
}

BroadcastDescription_RvcStream public destroy {} {
    set callback "$self switchCallback"
    $BroadcastDescription::405Control callback_unregister $callback
}

BroadcastDescription_RvcStream public switchCallback {amxMsg} {
    $self instvar shortHostname_ source_ sourceStartTime_ callback_

    if {[llength $amxMsg] != 0} {
	set eventInfo [lindex $amxMsg 0]
	set type [lindex $eventInfo 0]
	set eventData [lindex $amxMsg 1]
    } else {
	return ""
    }

    if {$type == "matrixSwitch"} {
	set input [lindex $eventData 0]
	set output [lindex $eventData 1]
	if {$output == $shortHostname_} {
	    if {$input != $source_} {
		# the stream was switched
		set source_ $input
		set sourceStartTime_ $BroadcastDescription::broadcastClock
		# do the notification callback if one exists
		if {$callback_ != ""} {
		    eval "$callback_ $source_"
		}
	    }
	}
    }

    return ""
}

# FIXME - this state can become corrupted if the director removes the rvc
#    from the broadcast window
#
# The proper way to do it is for the DC to know who is watching, and send
#   messages to those apps when something changes
#
# At the very least, we should check to make sure that the state we currently
#   have is the same as what is in the DC
BroadcastDescription_RvcStream public broadcast {{val default}} {
    $self instvar broadcast_ sourceStartTime_ hostname_

    switch -exact -- $val {
	true {
	    if {$broadcast_ == "false"} {
		set sourceStartTime_ $BroadcastDescription::broadcastClock
		set result [$BroadcastDescription::DcControl broadcastHost $hostname_]
		if {!$result} {
		    set msg "Warning: unable to broadcast for $hostname_"
		    tk_messageBox -message $msg -type ok -title "BroadcastDescription_RvcStream: Warning" -icon "warning"
	    return -1
		}
	    }
	    set broadcast_ $val
	}
	false {
	    set broadcast_ $val
	    $BroadcastDescription::DcControl unbroadcastHost $hostname_
	}
    }

    return $broadcast_
}

BroadcastDescription_RvcStream public duration {{val default}} {
    $self instvar sourceStartTime_ broadcast_

    if {$broadcast_} {
	return [expr $BroadcastDescription::broadcastClock - $sourceStartTime_]
    } else {
	return 0
    }
}

BroadcastDescription_RvcStream public source {{val default}} {
    $self instvar source_ sourceStartTime_ hostname_ shortHostname_

    if {$val == "default"} {
	# it's a query on the source
	return "$source_"
    }

    if {$source_ != $val} {
	# it's a new source, so switch
	switch -exact -- $val {
	    mbonePC -
	    sgiPC -
	    frontPC -
	    liveboard -
	    laptop -
	    rackVCR -
	    frontVCR -
	    speakerCamera -
	    wideCamera -
	    audienceCamera -
	    elmo {
		$BroadcastDescription::405Control matrix_switchVideoStream $val $shortHostname_
	    }
	    default {
		puts stderr "BroadcastDescription_RvcStream::source: source \"$source_\" is invalid.  Ignoring."
	    }
	}
    }

    return $source_
}

BroadcastDescription_RvcStream instproc getFilter {watchOutput} {
    # This filters out all events except for matrixSwitch events that have an
    #   output of $watchOutput
    set filter ""
    set filter "$filter set doCallback 0\n"
    set filter "$filter set info \[lindex \$arg 0\]\n"
    set filter "$filter set type \[lindex \$info 0\]\n"
    set filter "$filter set data \[lindex \$arg 1\]\n"
    set filter "$filter if \{\$type == \"matrixSwitch\"\} \{\n"
    set filter "$filter set output \[lindex \$data 1\]\n"
    set filter "$filter if \{\$output == \"$watchOutput\"\} \{\n"
    set filter "$filter set doCallback 1\n"
    set filter "$filter \}\n"
    set filter "$filter \}\n"
    return $filter
}

################################################

Class BroadcastDescription_ReplayStream -superclass BroadcastDescription_Stream

################################################

Class BroadcastDescription_RealNetworksStream -superclass BroadcastDescription_Stream

BroadcastDescription_RealNetworksStream instproc init {name} {
    $self instvar broadcast_ source_ sourceStartTime_

    # get the current input
    set source_ [$BroadcastDescription::405Control matrix_getInputSource "realNetworks"]

    $self next $name

    set callback "$self switchCallback"
    set filter [$self getFilter "realNetworks"]
    $BroadcastDescription::405Control callback_register $callback $filter
    $BroadcastDescription::405Control callback_enable $callback
}

BroadcastDescription_RealNetworksStream public destroy {} {
    set callback "$self switchCallback"
    $BroadcastDescription::405Control callback_unregister $callback
}

BroadcastDescription_RealNetworksStream public switchCallback {amxMsg} {
    $self instvar source_ sourceStartTime_ callback_

    if {[llength $amxMsg] != 0} {
	set eventInfo [lindex $amxMsg 0]
	set type [lindex $eventInfo 0]
	set eventData [lindex $amxMsg 1]
    } else {
	return ""
    }

    if {$type == "matrixSwitch"} {
	set input [lindex $eventData 0]
	set output [lindex $eventData 1]
	if {$output == "realNetworks"} {
	    if {$input != $source_} {
		# the stream was switched
		set source_ $input
		set sourceStartTime_ $BroadcastDescription::broadcastClock
		# do the notification callback if one exists
		if {$callback_ != ""} {
		    eval $callback_ $source_
		}
	    }
	}
    }

    return ""
}

# FIXME - this doesn't really mean much since we can't control RealNetworks
#   broadcasts from here, but we need it to maintain the stream abstraction
#   and all the timings work the same way
#
# Eventually, you would like to be able to remotely control the Real Server
#   and this function would enable/disable broadcasting
BroadcastDescription_RealNetworksStream public broadcast {{val default}} {
    $self instvar broadcast_ sourceStartTime_

    switch -exact -- $val {
	true {
	    if {$broadcast_ == "false"} {
		set sourceStartTime_ $BroadcastDescription::broadcastClock
	    }
	    set broadcast_ $val
	}
	false {
	    set broadcast_ $val
	}
    }

    return $broadcast_
}

BroadcastDescription_RealNetworksStream public duration {{val default}} {
    $self instvar sourceStartTime_ broadcast_

    if {$broadcast_} {
	return [expr $BroadcastDescription::broadcastClock - $sourceStartTime_]
    } else {
	return 0
    }
}

BroadcastDescription_RealNetworksStream public source {{val default}} {
    $self instvar source_ sourceStartTime_

    if {$val == "default"} {
	# it's a query on the source
	return "$source_"
    }

    if {$source_ != $val} {
	# it's a new source, so switch
	switch -exact -- $val {
	    mbonePC -
	    sgiPC -
	    frontPC -
	    liveboard -
	    laptop -
	    rackVCR -
	    frontVCR -
	    speakerCamera -
	    wideCamera -
	    audienceCamera -
	    elmo {
		$BroadcastDescription::405Control matrix_switchVideoStream $val realNetworks
	    }
	    default {
		puts stderr "BroadcastDescription_RealNetworksStream::source: source \"$val\" is invalid.  Ignoring."
	    }
	}
    }

    return $source_
}

BroadcastDescription_RealNetworksStream instproc getFilter {watchOutput} {
    # This filters out all events except for matrixSwitch events that have an
    #   output of $watchOutput
    set filter ""
    set filter "$filter set doCallback 0\n"
    set filter "$filter set info \[lindex \$arg 0\]\n"
    set filter "$filter set type \[lindex \$info 0\]\n"
    set filter "$filter set data \[lindex \$arg 1\]\n"
    set filter "$filter if \{\$type == \"matrixSwitch\"\} \{\n"
    set filter "$filter set output \[lindex \$data 1\]\n"
    set filter "$filter if \{\$output == \"$watchOutput\"\} \{\n"
    set filter "$filter set doCallback 1\n"
    set filter "$filter \}\n"
    set filter "$filter \}\n"
    return $filter
}

################################################

Class BroadcastDescription_StreamStatus

BroadcastDescription_StreamStatus instproc init {streamObj base} {
    $self instvar widget_

    label $base.name -text "[$streamObj name]:"
    pack $base.name -side left
    set widget_ [label $base.value -text "[$streamObj source]"]
    pack $base.value -side right
    set callback "$self sourceChange"
    $streamObj notifyCallback $callback
}

BroadcastDescription_StreamStatus public sourceChange {newSource} {
    $self instvar widget_

    $widget_ configure -text "$newSource"
}
