# analyze.tcl --
#
#       FIXME: This file needs a description here.
#
# Copyright (c) 2000-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

global g_markFile g_simFile g_stats g_outfile

proc showUsage {} {
    puts stdout "analyze: <mark file> <sim file> <outfile> <W1 W2 W3 W4 W5>"
    puts stdout "\tW1 ="
    puts stdout "\tW2 ="
    puts stdout "\tW3 ="
    puts stdout "\tW4 ="
    puts stdout "\tW5 ="
    exit
}

proc init {argv} {
    global g_markFile g_simFile g_stats g_outfile

    if {[llength $argv] != 3} {
	showUsage
    }

    set g_markFile [lindex $argv 0]
    set g_simFile [lindex $argv 1]
    set g_outfile [lindex $argv 2]

    # clear stats
    set g_stats(all,total) 0
    set g_stats(all,same) 0
    set g_stats(all,different) 0

    set g_stats(speaker,total) 0
    set g_stats(speaker,same) 0
    set g_stats(speaker,different) 0
    set g_stats(speaker,different,speaker) 0
    set g_stats(speaker,different,left) 0
    set g_stats(speaker,different,center) 0
    set g_stats(speaker,different,right) 0

    set g_stats(left,total) 0
    set g_stats(left,same) 0
    set g_stats(left,different) 0
    set g_stats(left,different,speaker) 0
    set g_stats(left,different,left) 0
    set g_stats(left,different,center) 0
    set g_stats(left,different,right) 0

    set g_stats(center,total) 0
    set g_stats(center,same) 0
    set g_stats(center,different) 0
    set g_stats(center,different,speaker) 0
    set g_stats(center,different,left) 0
    set g_stats(center,different,center) 0
    set g_stats(center,different,right) 0

    set g_stats(right,total) 0
    set g_stats(right,same) 0
    set g_stats(right,different) 0
    set g_stats(right,different,speaker) 0
    set g_stats(right,different,left) 0
    set g_stats(right,different,center) 0
    set g_stats(right,different,right) 0
}

proc go {} {
    global g_markFile g_simFile g_stats g_outfile

    if {[catch {open $g_markFile r} mark]} {
	puts stdout "Cannot open file $g_markFile for reading"
	return 0
    }

    if {[catch {open $g_simFile r} sim]} {
	puts stdout "Cannot open file $g_simFile for reading"
	return 0
    }

    set go 1
    while {$go} {
	set ok(mark) [gets $mark line(mark)]
	set ok(sim) [gets $sim line(sim)]

	if {$ok(mark) < 0 || $ok(sim) < 0} {
	    set go 0
	} else {
	    set temp [split $line(mark) " "]
	    set index(mark) [lindex $temp 0]
	    set mode(mark) [lindex $temp 1]
	    set temp [split $line(sim) " "]
	    set index(sim) [lindex $temp 0]
	    set mode(sim) [lindex $temp 1]
	    if {$index(sim) != $index(mark)} {
		puts stdout "weird failure!!!"
		exit
	    }
	    processEntry $index(sim) $mode(mark) $mode(sim)
	}
    }

    report
    genOutfile

    close $sim
    close $mark
}

proc processEntry {index mark sim} {
    global g_stats g_outfile

    #puts stdout "$index $mark $sim"
    incr g_stats(all,total)
    incr g_stats($mark,total) 1
    if {$mark == $sim} {
	incr g_stats(all,same) 1
	incr g_stats($mark,same) 1
    } else {
	incr g_stats(all,different) 1
	incr g_stats($mark,different) 1
	incr g_stats($mark,different,$sim) 1
    }

}

proc report {} {
    global g_stats g_outfile

    set modes [list all speaker left center right]
    foreach mode $modes {
	puts stdout "$mode"
	if {$mode != "all"} {
	    set pct [expr (double($g_stats($mode,total)) / double($g_stats(all,total))) * 100]
	    puts stdout "\tpct $mode: $pct"
	}
	puts stdout "\ttotal: $g_stats($mode,total)"
	puts stdout "\tsame: $g_stats($mode,same)"
	puts stdout "\tdifferent: $g_stats($mode,different)"
	if {$mode != "all"} {
	    if {$g_stats($mode,different) != 0} {
		foreach wrong $modes {
		    if {$wrong != "all" && $wrong != $mode} {
			set pct [expr (double($g_stats($mode,different,$wrong)) / double($g_stats($mode,different))) * 100]
			puts stdout "\t\t$wrong: $g_stats($mode,different,$wrong) == $pct %"
		    }
		}
	    }
	}
	if {$g_stats($mode,total) > 0} {
	    set pctSame [expr double($g_stats($mode,same)) / double($g_stats($mode,total))]
	    puts stdout "\t% same: [expr $pctSame * 100.0]"
	} else {
	}
    }
}

proc genOutfile {} {
    global g_stats g_outfile

    set outfile "stdout"

    set modes [list speaker left center right]
    foreach mode $modes {
	# pct(total,$mode) - what % of total time is this mic
	set pct(total,$mode) [expr (double($g_stats($mode,total)) / double($g_stats(all,total)))]

	# pct(correct,$mode) - what % are correct
	if {$g_stats($mode,total) > 0} {
	    set pct(correct,$mode) [expr double($g_stats($mode,same)) / double($g_stats($mode,total))]
	} else {
	    set pct(correct,$mode) 0
	}

	# pct(missed,mode) - of the incorrect ones, what % is speaker
	if {($mode != "speaker") && ($g_stats($mode,different) != 0)} {
	    set pct(missed,$mode) [expr (double($g_stats($mode,different,speaker)) / double($g_stats($mode,different)))]
	} else {
	    set pct(missed,$mode) 0
	}

	# pct(wrong,$mode) - of the incorrect ones, what % is another mic
	set pct(wrong,$mode) [calculatePercentWrong $mode]

	puts $outfile "$mode:"
	puts $outfile "\t% total: $pct(total,$mode)"
	puts $outfile "\t% missed: $pct(missed,$mode)"
	puts $outfile "\t% wrong: $pct(wrong,$mode)"
	puts $outfile "\t% correct: $pct(correct,$mode)"
    }

    set pct(correct,all) 0
    set pct(missed,all) 0
    set pct(wrong,all) 0
    foreach mode $modes {
	set pct(correct,all) [expr $pct(correct,all) + [expr $pct(total,$mode) * $pct(correct,$mode)]]
	if {$mode != "speaker"} {
	    set pct(missed,all) [expr $pct(missed,all) + [expr $pct(total,$mode) * [expr 1 - $pct(correct,$mode)] * $pct(missed,$mode)]]
	    set pct(wrong,all) [expr $pct(wrong,all) + [expr $pct(total,$mode) * [expr 1 - $pct(correct,$mode)] * $pct(wrong,$mode)]]
	}
    }

    set pct(falsePositive,all) [expr $pct(total,speaker) * [expr 1 - $pct(correct,speaker)]]


    set mode "all"
    puts $outfile "$mode:"
    puts $outfile "\t% correct: $pct(correct,$mode)"
    # of the times it was the speaker, what % was correctly identified
    set pct(speakerCorrect,$mode) [expr $pct(total,speaker) * $pct(correct,speaker)]
    puts $outfile "\t% speaker correct: $pct(speakerCorrect,$mode)"
    # of the times it was a mic, what % was correctly identified
    set pct(questionsCorrect,$mode) [expr $pct(correct,$mode) - [expr $pct(total,speaker) * $pct(correct,speaker)]]
    puts $outfile "\t% questions correct: $pct(questionsCorrect,$mode)"
    # should be speaker, but selected mic
    puts $outfile "\t% false positive: $pct(falsePositive,$mode)"
    # should be mic, but selected speaker
    puts $outfile "\t% missed: $pct(missed,$mode)"
    # should be mic X, but selected mic Y
    puts $outfile "\t% wrong: $pct(wrong,$mode)"

    set total [expr $pct(correct,$mode) + $pct(falsePositive,$mode) + $pct(missed,$mode) + $pct(wrong,$mode)]
    puts $outfile "total is $total"




#    set mode "questions"
#    puts $outfile "$mode:"

#    puts $outfile "\t% total: $pct(total,$mode)"
#    puts $outfile "\t% correct: $pct(correct,$mode)"

    doSeparateStats

}


proc calculatePercentWrong {mode} {
    global g_stats

    if {$g_stats($mode,different) != 0} {
	set numerator [expr double($g_stats($mode,different,left)) + double($g_stats($mode,different,center)) + double($g_stats($mode,different,right))]
	set pct [expr $numerator / double($g_stats($mode,different))]
	return $pct
    } else {
	return 0
    }
}

proc doSeparateStats {} {
    global g_stats g_outfile

    if {[catch {open $g_outfile w} outfile]} {
	puts stdout "Cannot open file $g_outfile for writing"
	return 0
    }

    set g_stats(questions,total) [expr $g_stats(left,total) + $g_stats(center,total) + $g_stats(right,total)]
    puts stdout "total questions: $g_stats(questions,total)"


    # do pcts in terms of questions
    set modes [list left center right]
    foreach mode $modes {
	# pct(total,$mode) - what % of questions are this mic
	set pct(total,$mode) [expr (double($g_stats($mode,total)) / double($g_stats(questions,total)))]

	# pct(correct,$mode) - when it's this mic, what % are correct
	if {$g_stats($mode,total) > 0} {
	    set pct(correct,$mode) [expr double($g_stats($mode,same)) / double($g_stats($mode,total))]
	} else {
	    set pct(correct,$mode) 0
	}

	# pct(missed,mode) - when it's this mic, what % is speaker
	if {($mode != "speaker") && ($g_stats($mode,different) != 0)} {
	    set pct(missed,$mode) [expr (double($g_stats($mode,different,speaker)) / double($g_stats($mode,total)))]
	} else {
	    set pct(missed,$mode) 0
	}

	# pct(wrong,$mode) - when it's this mic, what % is another mic
	if {$g_stats($mode,different) != 0} {
	    set numerator [expr double($g_stats($mode,different,left)) + double($g_stats($mode,different,center)) + double($g_stats($mode,different,right))]
	    set pct(wrong,$mode) [expr $numerator / double($g_stats($mode,total))]
	} else {
	    set pct(wrong,$mode) 0
	}

    }

    set mode "speaker"
    set pct(total,$mode) [expr (double($g_stats($mode,total)) / double($g_stats(all,total)))]
    set pct(correct,$mode) [expr $g_stats($mode,same) / double($g_stats($mode,total))]
    set pct(falsePositives,$mode) [expr 1 - $pct(correct,$mode)]

    # break into questions and normal and do stats for each
    set modes [list left center right]

    set pct(correct,questions) 0
    set pct(missed,questions) 0
    set pct(wrong,questions) 0
    set pct(total,questions) [expr double($g_stats(questions,total)) / double($g_stats(all,total))]
    foreach mode $modes {
  	set pct(correct,questions) [expr double($pct(correct,questions)) + [expr $pct(total,$mode) * $pct(correct,$mode)]]
	set pct(missed,questions) [expr double($pct(missed,questions)) + $pct(total,$mode) * $pct(missed,$mode)]
	set pct(wrong,questions) [expr double($pct(wrong,questions)) + ($pct(total,$mode) * $pct(wrong,$mode))]
    }

    # output in a form easy to parse
    set mode "questions"
    puts $outfile "percentAllAreQuestions $pct(total,$mode)"
    puts $outfile "percentQuestionsCorrect $pct(correct,$mode)"
    puts $outfile "percentQuestionsMissed $pct(missed,$mode)"
    puts $outfile "percentQuestionsWrong $pct(wrong,$mode)"
    set mode "speaker"
    puts $outfile "percentAllAreSpeaker $pct(total,$mode)"
    puts $outfile "percentSpeakerCorrect $pct(correct,$mode)"
    puts $outfile "percentSpeakerFalsePositives $pct(falsePositives,$mode)"

    puts $outfile ""

    set names(left) "Left"
    set names(center) "Center"
    set names(right) "Right"
    set modes [list left center right]
    foreach mode $modes {
	set modeName $names($mode)
	puts $outfile "percentQuestionsAre${modeName} $pct(total,$mode)"
	puts $outfile "percent${modeName}QuestionsMissed $pct(missed,$mode)"
	puts $outfile "percent${modeName}QuestionsWrong $pct(wrong,$mode)"
	puts $outfile "percent${modeName}QuestionsCorrect $pct(correct,$mode)"
    }

    close $outfile

}


# main program

init $argv

go

