
/******************************************************************************
**
**  This program is free software; you can redistribute it and/or
**  modify it, however, you cannot sell it.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
**  You should have received a copy of the license attached to the
**  use of this software.  If not, visit www.shmoo.com/osiris for
**  details.
**
******************************************************************************/

/*****************************************************************************
**
**  File:    mod_groups.c
**  Date:    January 1, 2004
**
**  Author:  Brian Wotring
**  Purpose: platform specific methods for reading group file information.
**
******************************************************************************/

#include "libosiris.h"
#include "libfileapi.h"
#include "rootpriv.h"
#include "common.h"
#include "version.h"

#include "scanner.h"
#include "logging.h"

#ifdef HAVE_GETGRENT
#include <grp.h>
#endif

#define GROUP_FILE "/etc/group"

static const char *MODULE_NAME = "mod_groups";

#ifdef WIN32

#include <lm.h>

void process_windows_groups( SCANNER *scanner )
{
    /* we need to use 2, not 3, to support NT and 2K. */

    LOCALGROUP_INFO_1 *pBuf = NULL;
    LOCALGROUP_INFO_1 *pTmpBuf;
    DWORD dwLevel = 1;
    DWORD dwPrefMaxLen = -1;
    DWORD dwEntriesRead = 0;
    DWORD dwTotalEntries = 0;
    DWORD dwResumeHandle = 0;
    DWORD i;
    DWORD dwTotalCount = 0;
    NET_API_STATUS nStatus;

    SCAN_RECORD_TEXT_1 record;

    do 
    {
        nStatus = NetGroupEnum( NULL,
                               dwLevel,
                               (LPBYTE*)&pBuf,
                               dwPrefMaxLen,
                               &dwEntriesRead,
                               &dwTotalEntries,
                               &dwResumeHandle);

        if( ( nStatus == NERR_Success) || ( nStatus == ERROR_MORE_DATA ) )
        {
            if( ( pTmpBuf = pBuf ) != NULL )
            {
                for( i = 0; (i < dwEntriesRead); i++ )
                {
                    if( pTmpBuf == NULL )
                    {
                        log_error( "error reading group entry." );
                        break;
                    }

                    initialize_scan_record( (SCAN_RECORD *)&record,
                                            SCAN_RECORD_TYPE_TEXT_1 );

                    osi_strlcpy( record.module_name, MODULE_NAME,
                                 sizeof( record.module_name ) );

                    _snprintf( record.name, sizeof( record.name ),
                               "group:%ls", pTmpBuf->lgrpi1_name );


                    _snprintf( record.data, sizeof( record.data ),
                               "name:%ls;", pTmpBuf->lgrpi1_name );

                    send_scan_data( scanner, (SCAN_RECORD *)&record );

                    pTmpBuf++;
                    dwTotalCount++;
                }
            }
        }
         
        else
        {
      
        }

        if( pBuf != NULL )
        {
            NetApiBufferFree( pBuf );
            pBuf = NULL;
        }

    } while( nStatus == ERROR_MORE_DATA );

    if( pBuf != NULL )
    {
        NetApiBufferFree(pBuf);
    }
}

#else

void process_local_etc_group_file( SCANNER *scanner )
{
    char temp_line[MAX_TEXT_RECORD_DATA_LENGTH];

    FILE *group_file;
    SCAN_RECORD_TEXT_1 record;

    /* open the local group database file for read only. */

#ifdef USE_PRIVSEP
    group_file = rootpriv_fopen( GROUP_FILE );
#else
    group_file = fopen( GROUP_FILE, "r" );
#endif

    if( group_file == NULL )
    {
        log_error( "unable to open local group file (%s).", GROUP_FILE );
        return;
    }
   
    /* now read each uncommented line */ 

    for(;;)
    {
        char *line;
        char *groupname_end;

        line = fgets( temp_line, sizeof( temp_line ), group_file );

        if( line == NULL)
        {
            break;
        }

        line = trim_white_space( line );

        /* skip commented and empty lines. */
    
        if( ( line == NULL ) || ( line[0] == '#' ) )
        {
            continue;
        }

        /* locate the groupname, this is the first item in the colon list. */

        if( ( groupname_end = strchr( line, ':' ) ) == NULL )
        {
            continue;
        }

        initialize_scan_record( (SCAN_RECORD *)&record,
                                SCAN_RECORD_TYPE_TEXT_1 );

        osi_strlcpy( record.module_name, MODULE_NAME,
                     sizeof( record.module_name ) );

        /* user the groupname as a key/path for this record. */

        (*groupname_end) = '\0';
        osi_strlcpy( record.name, "group:", sizeof( record.name ) );
        osi_strlcat( record.name, line, sizeof( record.name ) );
        (*groupname_end) = ':'; 

        /* now copy in the entire group entry into the data portion. */
        /* and send this record on its way.                          */

        osi_strlcpy( record.data, line, sizeof( record.data ) );
        send_scan_data( scanner, (SCAN_RECORD *)&record );
    }    

    fclose( group_file );
}

void process_group_entries( SCANNER *scanner )
{
    struct group *gr;
    SCAN_RECORD_TEXT_1 record;

    gr = getgrent();

    while( gr != NULL )
    {
        int index = 0;

        if( gr->gr_name != NULL )
        {
            char buf[10];

            initialize_scan_record( (SCAN_RECORD *)&record,
                                    SCAN_RECORD_TYPE_TEXT_1 );

            osi_strlcpy( record.module_name, MODULE_NAME,
                         sizeof( record.module_name ) );

            /* user the group name as a key/path for this record. */

            osi_strlcpy( record.name, "group:", sizeof(record.name) );
            osi_strlcat( record.name, gr->gr_name, sizeof(record.name) );

            /* now assemble the group entry string. */

            osi_strlcpy( record.data, gr->gr_name, sizeof(record.data) );
            osi_strlcat( record.data, ":", sizeof(record.data));

            osi_strlcat( record.data, gr->gr_passwd, sizeof(record.data) );
            osi_strlcat( record.data, ":", sizeof(record.data));

            osi_snprintf( buf, sizeof(buf), "%d", gr->gr_gid );
            osi_strlcat( record.data, buf, sizeof(record.data));
            osi_strlcat( record.data, ":", sizeof(record.data));

            /* list of group members. */

            if( gr->gr_mem != NULL )
            {
                while( (gr->gr_mem)[index] != NULL )
                {
                    osi_strlcat( record.data, gr->gr_mem[index++],
                                 sizeof(record.data) );

                    if( gr->gr_mem[index] )
                    {
                        osi_strlcat( record.data, ",", sizeof(record.data));
                    }
                }
            }

            send_scan_data( scanner, (SCAN_RECORD *)&record );
        }

        gr = getgrent();

    } /* while gr != NULL */
}

#endif /* not WIN32 */

void mod_groups( SCANNER *scanner )
{
#ifdef WIN32
    process_windows_groups( scanner );
#else

#ifdef HAVE_GETGRENT
    process_group_entries( scanner );
#else
    process_local_etc_group_file( scanner );
#endif
#endif /* WIN32 */
}

