/*****************************************************************************
 *                                                                           *
 * Program:   paul                                                           *
 *            (P)rogramm zur (A)uswertung und (U)mformung von                *
 *            (L)aserbildern                                                 *
 * Modul:     pngsave.c                                                      *
 *            Write PNG-files                                                *
 * Uses:      LibPNG (works best with 1.0.0, may be fail with 1.0.2)         *
 * Author:    Andreas Tille                                                  *
 * Date:      01.04.1998                                                     *
 * Copyright: Andreas Tille, 1999; GNU Public License                        *
 *                                                                           *
 *****************************************************************************/

#include <png.h>
#include <stdlib.h>
#include "paul.h"

int WritePNGImage(PICTURE *bild, char *filename)
/* Write PNG-file
 * --- Parameter: ---
 * PICTURE *bild       : image data to write
 * char    *filename   : PNG file name (possibly different from bild->file!!)
 * --- Return: ---
 * int WritePNGImage() : RET_ERR or RET_OK
 */
{
  register   unsigned char *ap, *bp, *fip, **app, **fipp;
  int                       rowbytes, nc;
  png_info                 *pngi;
  png_struct               *png;
  unsigned   char          *png_pixels, **scanlines;
  FILE                     *fp;
  png_text                  textp[1];
  CHUNK                    *cp;
  char                    **cbufs, **cbp, *zw;
  png_time                  zeit;

  g_return_val_if_fail ( IS_PICTURE(bild), RET_ERR );
  g_return_val_if_fail ( (fp = fopen(filename,"w")), RET_ERR );

  g_return_val_if_fail ( (png = png_create_write_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL)), 
                         RET_ERR ) ;
  g_return_val_if_fail ( (pngi = png_create_info_struct(png)), RET_ERR ) ;
  
  if ( setjmp(png->jmpbuf) ) {
    png_destroy_write_struct(&png,&pngi);
    fclose(fp);
    return RET_ERR;
  }

  /* Prepare PNG for writing. */
  png_init_io(png, fp);
  png_set_IHDR(png, pngi, bild->W, bild->H, 8,
               IsMonochrom(bild) ? PNG_COLOR_TYPE_GRAY : PNG_COLOR_TYPE_RGB,
	PNG_INTERLACE_NONE, PNG_COMPRESSION_TYPE_BASE, PNG_FILTER_TYPE_BASE);

  if ( bild->spec ) {
    pngi->max_text = nc = NumValidChunks(bild->spec);
    pngi->text     = g_new0(png_text, nc);
    cbufs          = g_new0(char *, nc);

    for ( cbp = cbufs, cp = bild->spec; cp->key != NULL; cp++ ) 
      if ( (zw = Chunk2Char(cp)) ) {
        textp->key         = (char *)cp->key;
        textp->text        = zw;
        textp->compression = (textp->text_length = strlen(textp->text)) < 256 ? PNG_TEXT_COMPRESSION_NONE 
                                                                              : PNG_TEXT_COMPRESSION_zTXt;
        png_set_text(png, pngi, textp, 1);
        *(cbp++) = zw;
      }
    g_return_val_if_fail ( nc == cbp - cbufs, RET_ERR );
  } else {
    nc    = 0;
    cbufs = NULL;
  }

  if ( bild->zeit > 0 ) png_convert_from_time_t(&zeit, bild->zeit);
  else                  png_convert_from_time_t(&zeit, time(NULL));
  png_set_tIME(png, pngi, &zeit);
  if ( bild->res ) 
    png_set_pHYs(png, pngi, bild->res, bild->res, PNG_RESOLUTION_UNKNOWN);
  if ( bild->x_offset != (unsigned long)-1 )
    png_set_oFFs(png, pngi, bild->x_offset, bild->y_offset, PNG_OFFSET_PIXEL);
  if ( bild->n_gamma && bild->gamma )
    png_set_PLTE(png, pngi, bild->gamma, bild->n_gamma);

  png_write_info(png, pngi);
  png_set_packing(png);

  /* Allocate memory. */
  rowbytes = bild->spp * bild->W;
  png_pixels = g_malloc(rowbytes * (bild->H));
  scanlines  = g_malloc(4 * bild->H);
      /*  4* to read alpha data! */

  /* Initialize image scanlines. */
  for ( fipp = (app = scanlines) + bild->H, ap = png_pixels; app < fipp; 
        app++, ap += rowbytes )
    *app = ap;

  if ( IsMonochrom(bild) && bild->storepix == 3 )
    for ( fip = (ap = png_pixels) + rowbytes * (bild->H), bp = bild->DATA + 1;
          ap < fip; ap++, bp += 3)
       *ap = *bp;
  else
    memcpy(png_pixels, bild->DATA, rowbytes * bild->H);

  png_write_image(png, scanlines);

  FREE(scanlines);
  FREE(png_pixels);
   
  png_write_end(png,pngi);
  png_destroy_write_struct(&png, &pngi);
  if ( cbufs ) {
    for ( nc--; nc >= 0; nc-- ) FREE(cbufs[nc]);
    FREE(cbufs);
  }
  
  fclose(fp);
  return RET_OK;
}


