// --------------------------------------------------------------------
// Output device writing XML stream
// --------------------------------------------------------------------

#include <stdio.h>
#include "ocfile.h"
#include <stddef.h>
#include <stdarg.h>
#include "gstring.h"
#include "config.h"
#include "globalparams.h"
#include "object.h"
#include "error.h"
#include "gfx.h"
#include "gfxstate.h"
#include "gfxfont.h"
#include "charcodetounicode.h"
#include "unicodemap.h"
#include "fontfile.h"
#include "catalog.h"
#include "page.h"
#include "stream.h"
#include "xmloutputdev.h"

#include <vector>

//------------------------------------------------------------------------
// XmlOutputDev
//------------------------------------------------------------------------

XmlOutputDev::XmlOutputDev(char *fileName, XRef *xrefA, Catalog *,
			   int /* firstPage */, int /* lastPage */,
			   GBool isMath, GBool noText)
{
  OCFILE *f;

  if (!(f = ocfopen(fileName, "w"))) {
    error(-1, "Couldn't open output file '%s'", fileName);
    ok = gFalse;
    return;
  }
  outputStream = f;

  // initialize
  ok = gTrue;
  xref = xrefA;
  inText = false;
  iIsMath = isMath;
  iNoText = noText;
  iUnicode = false;

  writePSFmt("<ipe creator=\"pdftoipe %s\">\n", PDFTOIPE_VERSION);
  // initialize sequential page number
  seqPage = 1;
}

XmlOutputDev::~XmlOutputDev()
{
  if (ok) {
    finishText();
    writePS("</ipe>\n");
  }
  fclose(outputStream);
}

// ----------------------------------------------------------

void XmlOutputDev::startPage(int pageNum, GfxState *state)
{
  int x1, y1, x2, y2;

  writePSFmt("<-- Page: %d %d -->\n", pageNum, seqPage);
  
  // rotate, translate, and scale page
  x1 = (int)(state->getX1() + 0.5);
  y1 = (int)(state->getY1() + 0.5);
  x2 = (int)(state->getX2() + 0.5);
  y2 = (int)(state->getY2() + 0.5);
  writePSFmt("<page mediabox=\"%d %d %d %d\">\n", x1, y1, x2, y2);
  ++seqPage;
}

void XmlOutputDev::endPage()
{
  finishText();
  writePS("</page>\n");
}

// --------------------------------------------------------------------

void XmlOutputDev::stroke(GfxState *state)
{
  finishText();
  GfxRGB rgb;
  state->getStrokeRGB(&rgb);
  writePSFmt("<path stroke=\"%g %g %g\"", rgb.r, rgb.g, rgb.b);
  writePSFmt(" pen=\"%g\"", state->getTransformedLineWidth());

  double *dash;
  double start;
  int length, i;

  state->getLineDash(&dash, &length, &start);
  if (length) {
    writePS(" dash=\"[");
    for (i = 0; i < length; ++i)
      writePSFmt("%g%s", state->transformWidth(dash[i]), 
		 (i == length-1) ? "" : " ");
    writePSFmt("] %g\"", state->transformWidth(start));
  }
    
  if (state->getLineJoin() > 0)
    writePSFmt(" join=\"%d\"", state->getLineJoin());
  if (state->getLineCap())
    writePSFmt(" cap=\"%d\"", state->getLineCap());

  writePS(">\n");
  doPath(state);
  writePS("</path>\n");
}

void XmlOutputDev::fill(GfxState *state)
{
  finishText();
  GfxRGB rgb;
  state->getFillRGB(&rgb);
  writePSFmt("<path fill=\"%g %g %g\" fillrule=\"wind\">\n",
	     rgb.r, rgb.g, rgb.b);
  doPath(state);
  writePS("</path>\n");
}

void XmlOutputDev::eoFill(GfxState *state)
{
  finishText();
  GfxRGB rgb;
  state->getFillRGB(&rgb);
  writePSFmt("<path fill=\"%g %g %g\">\n", rgb.r, rgb.g, rgb.b);
  doPath(state);
  writePS("</path>\n");
}

void XmlOutputDev::doPath(GfxState *state)
{
  GfxPath *path = state->getPath();
  GfxSubpath *subpath;
  int n, m, i, j;

  n = path->getNumSubpaths();

  double x, y, x1, y1, x2, y2;
  for (i = 0; i < n; ++i) {
    subpath = path->getSubpath(i);
    m = subpath->getNumPoints();
    state->transform(subpath->getX(0), subpath->getY(0), &x, &y);
    writePSFmt("%g %g m\n", x, y);
    j = 1;
    while (j < m) {
      if (subpath->getCurve(j)) {
	state->transform(subpath->getX(j), subpath->getY(j), &x, &y);
	state->transform(subpath->getX(j+1), subpath->getY(j+1), &x1, &y1);
	state->transform(subpath->getX(j+2), subpath->getY(j+2), &x2, &y2);
	writePSFmt("%g %g %g %g %g %g c\n", x, y, x1, y1, x2, y2);
	j += 3;
      } else {
	state->transform(subpath->getX(j), subpath->getY(j), &x, &y);
	writePSFmt("%g %g l\n", x, y);
	++j;
      }
    }
    if (subpath->isClosed()) {
      writePS("h\n");
    }
  }
}

// --------------------------------------------------------------------

void XmlOutputDev::updateTextPos(GfxState *)
{
  finishText();
}

void XmlOutputDev::drawChar(GfxState *state, double /*x*/, double /*y*/,
			    double /*dx*/, double /*dy*/,
			    double /*originX*/, double /*originY*/,
			    CharCode, Unicode *u, int uLen)
{
  // check for invisible text -- this is used by Acrobat Capture
  if ((state->getRender() & 3) == 3) {
    return;
  }

  // get the font
  GfxFont *font;
  if (!(font = state->getFont())) {
    return;
  }
  (void) font; // placate compiler
  
  if (iNoText) // discard text objects
    return;

  // begin text object
  if (!inText) {
    double *mat;
    mat = state->getTextMat();
    double x = state->getLineX();
    double y = state->getLineY();
    // writePSFmt("%g %g %g %g %g %g %g %g\n", 
    // mat[0], mat[1], mat[2], mat[3], mat[4], mat[5], x, y);
    double tx = mat[0] * x + mat[2] * y + mat[4];
    double ty = mat[1] * x + mat[3] * y + mat[5];

    GfxRGB rgb;
    state->getFillRGB(&rgb);

    writePSFmt("<text stroke=\"%g %g %g\" pos=\"%g %g\" size=\"normal\">", 
	       rgb.r, rgb.g, rgb.b, tx, ty);
    if (iIsMath)
      writePS("$");
  }
  inText = true;

  for (int i = 0; i < uLen; ++i) {
    int ch = u[i];
    if (ch == '\\') {
      writePS("$\\backslash$");
    } else if (ch == '^' || ch == '~') {
      writePS("\\");
      writePSChar(ch);
      writePS("{}");
    } else {
      if (ch == '&' || ch == '$' || ch == '#' || ch == '%'
	  || ch == '_' || ch == '{' || ch == '}')
	writePS("\\");
      writePSChar(ch);
    }
  }
}

void XmlOutputDev::finishText()
{
  if (inText) {
    if (iIsMath)
      writePS("$");
    writePS("</text>\n");
  }
  inText = false;
}

// --------------------------------------------------------------------

void XmlOutputDev::drawImageMask(GfxState *, Object */*ref*/, Stream */*str*/,
				 int /*width*/, int /*height*/, 
				 GBool /*invert*/, GBool /*inlineImg*/)
{
  // mask is a one-bit image
  // not yet implemented
}

void XmlOutputDev::drawImage(GfxState *state, Object * /*ref*/, Stream *str,
			     int width, int height, GfxImageColorMap *colorMap,
			     int * /*maskColors*/, GBool inlineImg)
{
  finishText();

  ImageStream *imgStr;
  Guchar *p;
  GfxRGB rgb;
  int x, y;
  int c;

  writePSFmt("<image width=\"%d\" height=\"%d\"", width, height);

  double *mat;
  mat = state->getCTM();
  double tx = mat[0] + mat[2] + mat[4];
  double ty = mat[1] + mat[3] + mat[5];
  writePSFmt(" rect=\"%g %g %g %g\"", mat[4], mat[5], tx, ty);
  
  if (str->getKind() == strDCT && !inlineImg &&
      3 <= colorMap->getNumPixelComps() && colorMap->getNumPixelComps() <= 4) {
    // dump JPEG stream
    std::vector<char> buffer;
    // initialize stream
    str = ((DCTStream *)str)->getRawStream();
    str->reset();
    // copy the stream
    while ((c = str->getChar()) != EOF)
      buffer.push_back(char(c));
    str->close();

    if (colorMap->getNumPixelComps() == 3)
      writePS(" ColorSpace=\"DeviceRGB\"");
    else 
      writePS(" ColorSpace=\"DeviceCMYK\"");
    writePS(" BitsPerComponent=\"8\"");
    writePS(" Filter=\"DCTDecode\"");
    writePSFmt(" length=\"%d\"", buffer.size());
    writePS(">\n");
    
    for (unsigned int i = 0; i < buffer.size(); ++i)
      writePSFmt("%02x", buffer[i] & 0xff);

#if 0
  } else if (colorMap->getNumPixelComps() == 1 && colorMap->getBits() == 1) {
    // 1 bit depth -- not implemented in Ipe

    // initialize stream
    str->reset();
    // copy the stream
    size = height * ((width + 7) / 8);
    for (i = 0; i < size; ++i) {
      writePSFmt("%02x", (str->getChar() ^ 0xff));
    }
    str->close();
#endif
  } else if (colorMap->getNumPixelComps() == 1) {
    // write as gray level image
    writePS(" ColorSpace=\"DeviceGray\"");
    writePS(" BitsPerComponent=\"8\"");
    writePS(">\n");
    
    // initialize stream
    imgStr = new ImageStream(str, width, colorMap->getNumPixelComps(),
			     colorMap->getBits());
    imgStr->reset();
    
    // for each line...
    for (y = 0; y < height; ++y) {
      
      // write the line
      p = imgStr->getLine();
      for (x = 0; x < width; ++x) {
	double gray;
	colorMap->getGray(p, &gray);
	writePSFmt("%02x", (int)(gray * 255 + 0.5));
	p += colorMap->getNumPixelComps();
      }
    }
    delete imgStr;

  } else {
    // write as RGB image
    writePS(" ColorSpace=\"DeviceRGB\"");
    writePS(" BitsPerComponent=\"8\"");
    writePS(">\n");
    
    // initialize stream
    imgStr = new ImageStream(str, width, colorMap->getNumPixelComps(),
			     colorMap->getBits());
    imgStr->reset();
    
    // for each line...
    for (y = 0; y < height; ++y) {
      
      // write the line
      p = imgStr->getLine();
      for (x = 0; x < width; ++x) {
	colorMap->getRGB(p, &rgb);
	writePSFmt("%02x", (int)(rgb.r * 255 + 0.5));
	writePSFmt("%02x", (int)(rgb.g * 255 + 0.5));
	writePSFmt("%02x", (int)(rgb.b * 255 + 0.5));
	p += colorMap->getNumPixelComps();
      }
    }
    delete imgStr;
  }
  writePS("\n</image>\n");
}

// --------------------------------------------------------------------

void XmlOutputDev::writePSChar(int code)
{
  if (code == '<')
    writePS("&lt;");
  else if (code == '>')
    writePS("&gt;");
  else if (code == '&')
    writePS("&amp;");
  else if (code < 0x80)
    writePSFmt("%c", code);
  else if (code < 0x800) {
    iUnicode = true;
    writePSFmt("%c%c", 
	       (((code & 0x7c0) >> 6) | 0xc0),
	       ((code & 0x03f) | 0x80));
  } else {
    iUnicode = true;
    // Do we never need to write UCS larger than 0x10000?
    writePSFmt("%c%c%c", 
	       (((code & 0x0f000) >> 12) | 0xe0),
	       (((code & 0xfc0) >> 6) | 0x80),
	       ((code & 0x03f) | 0x80));
  }
}

void XmlOutputDev::writePS(char *s)
{
  fwrite(s, 1, strlen(s), outputStream);
}

void XmlOutputDev::writePSFmt(const char *fmt, ...)
{
  va_list args;
  char buf[512];

  va_start(args, fmt);
  vsprintf(buf, fmt, args);
  va_end(args);
  fwrite(buf, 1, strlen(buf), outputStream);
}

// --------------------------------------------------------------------
