
/*
 * Copyright (c) Abraham vd Merwe <abz@blio.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *	  notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *	  notice, this list of conditions and the following disclaimer in the
 *	  documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of other contributors
 *	  may be used to endorse or promote products derived from this software
 *	  without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/time.h>
#include <time.h>
#include <event.h>
#include <netinet/in.h>

#include <abz/typedefs.h>
#include <debug/log.h>

#include "flow.h"
#include "signal.h"
#include "capture.h"
#include "gui.h"
#include "io.h"

#define PUTC_AT(c,x,y) do { \
		out_gotoxy(x,y); out_putc (c); \
	} while (0)

#define PUTS_AT(s,x,y) do { \
		out_gotoxy(x,y); out_puts (s); \
	} while (0)

struct gui
{
   struct event event;
   struct event timer;
   void (*display) (void);
   int width,height,y;
   int column[3];
   size_t flow;
};

static struct gui gui;

static void hline (int x1,int x2,int y)
{
   int x;

   out_gotoxy (x1,y);

   for (x = x1; x <= x2; x++)
	 out_putc (line.hl);
}

static void vline (int x,int y1,int y2)
{
   int y;

   for (y = y1; y <= y2; y++)
	 {
		out_gotoxy (x,y);
		out_putc (line.vl);
	 }
}

static __inline__ void drawrate (int x,int y,int w,double rate)
{
   if (w > 8)
	 {
		char buf[w];
		int i,n = 0;
		const char *suffix[] = { "", "k", "m", "g" };

		out_setattr (ATTR_BOLD);
		out_setcolor (COLOR_GREEN,COLOR_BLACK);

		rate *= 8.0;

		for (i = 0; i < ARRAYSIZE (suffix); i++)
		  if (rate > 1000.0)
			rate /= 1000.0, n++;

		snprintf (buf,w,"%.1f%s",rate,suffix[n]);

		if (strlen (buf) >= w)
		  strcpy (buf,"overflow");

		out_gotoxy (x,y);
		out_printf ("%*s",w,buf);
	 }
}

static void drawaddr (int x,int y,int w,in_addr_t addr,in_port_t port)
{
   char buf[64];
   size_t len;

   len = sprintf (buf,"%u.%u.%u.%u",HIPQUAD (addr));

   if (port)
	 len += sprintf (buf + len,":%u",port);

   if (w > strlen (buf))
	 {
		out_setattr (ATTR_OFF);
		out_setcolor (COLOR_CYAN,COLOR_BLACK);
		out_gotoxy (x,y);
		out_puts (buf);
	 }
}

static __inline__ void drawproto (int x,int y,int w,in_port_t port,uint8_t proto,int isfrag)
{
   static const struct
	 {
		uint8_t value;
		const char *name;
	 } list[] =
	 {
		{ IPPROTO_TCP, "tcp" },			/* Transmission Control Protocol		*/
		{ IPPROTO_UDP, "udp" },			/* User Datagram Protocol				*/
		{ IPPROTO_ICMP, "icmp" },		/* Internet Control Message Protocol	*/
		{ IPPROTO_IPIP, "ipip" },		/* IPIP tunnels							*/
		{ IPPROTO_GRE, "gre" },			/* General Routing Encapsulation		*/
		{ IPPROTO_SCTP, "sctp" },		/* Stream Control Transmission Protocol	*/
		{ IPPROTO_HOPOPTS, "ipv6" },	/* IPv6 Hop-by-Hop options				*/
		{ IPPROTO_IPV6, "ipv6" },		/* IPv6 header							*/
		{ IPPROTO_ROUTING, "ipv6" },	/* IPv6 routing header					*/
		{ IPPROTO_FRAGMENT, "ipv6" },	/* IPv6 fragmentation header			*/
		{ IPPROTO_ICMPV6, "ipv6" },		/* ICMPv6								*/
		{ IPPROTO_NONE, "ipv6" },		/* IPv6 no next header					*/
		{ IPPROTO_DSTOPTS, "ipv6" }		/* IPv6 destination options				*/
	 };
   size_t i;
   char buf[64];

   for (i = 0; i < ARRAYSIZE (list); i++)
	 if (proto == list[i].value)
	   break;

   strcpy (buf,i < ARRAYSIZE (list) ? list[i].name : "ip");

   if (isfrag)
	 strcat (buf," [frag]");

   if (w > strlen (buf))
	 {
		out_setattr (ATTR_OFF);
		out_setcolor (COLOR_CYAN,COLOR_BLACK);
		out_gotoxy (x + (w - strlen (buf)) / 2,y);
		out_puts (buf);
	 }
}

static void flow_draw (const struct flow *flow)
{
   time_t now = time (NULL);

   gui.flow++;

   if (gui.y > gui.height - 4 || now <= flow->timestamp)
	 return;

   drawaddr (2,
			 gui.y,
			 gui.column[0] - 2,
			 flow->saddr,flow->sport);

   drawaddr (gui.column[0] + 2,
			 gui.y,
			 gui.column[1] - gui.column[0] - 2,
			 flow->daddr,flow->dport);

   drawproto (gui.column[1] + 2,
			  gui.y,
			  gui.column[2] - gui.column[1] - 2,
			  flow->dport,flow->proto,flow->flags & FLOW_FRAG);

   drawrate (gui.column[2] + 1,
			 gui.y,
			 gui.width - gui.column[2] - 3,
			 flow->bitrate);

   gui.y++;
}

static void gui_flows (void)
{
   int i;

   out_setattr (ATTR_BOLD);
   out_setcolor (COLOR_BLUE,COLOR_BLACK);
   hline (1,gui.width - 2,2);
   PUTC_AT (line.lt,0,2);
   PUTC_AT (line.rt,gui.width - 1,2);

   gui.column[2] = gui.width - 12;
   gui.column[1] = gui.column[2] - 14;
   gui.column[0] = gui.column[1] >> 1;

   out_setattr (ATTR_BOLD);
   out_setcolor (COLOR_BLUE,COLOR_BLACK);

   for (i = 0; i < ARRAYSIZE (gui.column); i++)
	 {
		vline (gui.column[i],1,gui.height - 4);
		PUTC_AT (line.tt,gui.column[i],0);
		PUTC_AT (line.ct,gui.column[i],2);
		PUTC_AT (line.bt,gui.column[i],gui.height - 3);
	 }

   out_setcolor (COLOR_MAGENTA,COLOR_BLACK);
   PUTS_AT ("Source",2,1);
   PUTS_AT ("Destination",gui.column[0] + 2,1);
   PUTS_AT ("Protocol",gui.column[1] + 3,1);
   PUTS_AT ("Avg Rate",gui.column[2] + 2,1);

   gui.y = 3, gui.flow = 0;
   flow_process (flow_draw);
}

static void gui_help (void)
{
   int i,j;
   static const struct
	 {
		const char *key;
		const char *msg;
	 } help[] =
	 {
		{ "^L", "Redraw the screen" },
		{ "h or ?", "Print this list" },
		{ "q", "Quit" }
	 };
   static const char *msg[] =
	 {
		"Rates are displayed in bits per second. Rates displayed in green are",
		"lower than the expected bitrate. Rates displayed in yellow exceed the",
		"expected bitrate, but are still lower than the peak bitrate. If rates",
		"exceed the peak (or if no peak was defined) they are coloured red."
	 };

   out_setcolor (COLOR_MAGENTA,COLOR_BLACK);
   PUTS_AT ("Interactive commands",2,2);

   for (i = 0; i < ARRAYSIZE (help); i++)
	 {
		out_setattr (ATTR_BOLD);
		out_setcolor (COLOR_WHITE,COLOR_BLACK);
		out_gotoxy (4,4 + i);
		out_printf ("%-9s",help[i].key);

		out_setattr (ATTR_OFF);
		out_setcolor (COLOR_CYAN,COLOR_BLACK);
		out_puts (help[i].msg);
	 }

   out_setattr (ATTR_BOLD);
   out_setcolor (COLOR_MAGENTA,COLOR_BLACK);
   PUTS_AT ("Caveats",2,5 + i);

   out_setattr (ATTR_OFF);
   out_setcolor (COLOR_CYAN,COLOR_BLACK);
   for (i += 7, j = 0; j < ARRAYSIZE (msg); j++, i++)
	 PUTS_AT (msg[j],4,i);
}

static __inline__ void drawtime (void)
{
   time_t tp = time (NULL);
   struct tm *tm = localtime (&tp);

   out_setcolor (COLOR_WHITE,COLOR_BLACK);
   out_gotoxy (gui.width - 10,gui.height - 2);
   out_printf ("%02u:%02u:%02u",tm->tm_hour,tm->tm_min,tm->tm_sec);
}

static __inline__ void drawframe (void)
{
   out_setattr (ATTR_BOLD);
   out_setcolor (COLOR_BLUE,COLOR_BLACK);

   hline (1,gui.width - 2,0);
   hline (1,gui.width - 2,gui.height - 3);
   hline (1,gui.width - 2,gui.height - 1);

   vline (0,1,gui.height - 2);
   vline (gui.width - 1,1,gui.height - 2);

   PUTC_AT (line.tl,0,0);
   PUTC_AT (line.tr,gui.width - 1,0);
   PUTC_AT (line.lt,0,gui.height - 3);
   PUTC_AT (line.rt,gui.width - 1,gui.height - 3);
   PUTC_AT (line.bl,0,gui.height - 1);
   PUTC_AT (line.br,gui.width - 1,gui.height - 1);

   out_setcolor (COLOR_MAGENTA,COLOR_BLACK);
   PUTS_AT ("Potion IP Flow Monitor (press any key to continue)",2,gui.height - 2);

   drawtime ();
}

static void gui_display (void)
{
   gui.width = out_width ();
   gui.height = out_height ();

   out_clear ();
   drawframe ();
   gui.display ();
   out_flush ();
}

static void gui_dispatch (int fd,short event,void *arg)
{
   struct timeval tv = { .tv_sec = 1, .tv_usec = 0 };

   if (event & EV_READ)
	 {
		int c = in_getc ();

		if (gui.display == gui_flows)
		  {
			 if (c == KEY_CTRL('l') || c == KEY_CTRL('L'))
			   out_refresh ();
			 else if (c == 'h' || c == '?')
			   gui.display = gui_help;
			 else if (c == 'q')
			   {
				  gui_close ();
				  return;
			   }
			 else out_beep ();
		  }
		else gui.display = gui_flows;
	 }

   gui_display ();

   evtimer_del (&gui.timer);
   evtimer_add (&gui.timer,&tv);
}

int gui_open (const struct config *config)
{
   struct timeval tv = { .tv_sec = 1, .tv_usec = 0 };

   signal_open ();

   if (capture_open (config))
	 {
		signal_close ();
		return (-1);
	 }

   event_set (&gui.event,STDIN_FILENO,EV_READ | EV_PERSIST,gui_dispatch,NULL);

   if (event_add (&gui.event,NULL))
	 {
		log_printf (LOG_ERROR,"failed to add event handler: %m\n");
		signal_close ();
		capture_close ();
		return (-1);
	 }

   evtimer_set (&gui.timer,gui_dispatch,NULL);
   evtimer_add (&gui.timer,&tv);

   gui.display = gui_flows;

   io_open ();
   gui_display ();

   return (0);
}

void gui_close (void)
{
   static volatile int called = 0;

   if (!called)
	 {
		called = 1;
		evtimer_del (&gui.timer);
		event_del (&gui.event);
		io_close ();
		capture_close ();
		signal_close ();
	 }
}

