/* Copyright (C) 1999-2000 Bernhard Trummer
 * Copyright (C) 2003      Benjamin Drieu
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 *
 * $Log: parameters.h,v $
 * Revision 1.6  2003/03/06 14:54:24  benj
 * Update copyrigth where it is necessary
 *
 * Revision 1.5  2003/02/26 12:28:34  benj
 * Clean up options a bit
 * Fix a crash when images are not found
 * Command option '-l' now has an optional argument
 * .tex files are now based on xml presentation name
 *
 * Revision 1.4  2002/05/02 09:32:32  slash
 * Changes to make PresTiMeL compilable with g++ 3.0.
 * (Thanks to Benjamin Drieu)
 *
 * Revision 1.3  2002/02/12 08:33:14  slash
 * Bugfix in htmlnode.cc.
 * Changed htmlnode's cell_padding to cellpadding.
 *
 * Revision 1.2  2001/09/02 12:48:00  slash
 * Make use of AM_PATH_XML and AM_PATH_XML2 in configure.in
 *
 * Revision 1.1.1.1  2001/01/30 18:36:53  slash
 * Initial release.
 *
 */

#ifndef PARAMETERS_H
#define PARAMETERS_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string>

#include "imagesize.h"


#define DEFAULT_THEME  "standard_blue"
#define DEFAULT_TARGET_DIRECTORY  "./" 
#define DEFAULT_RESOLUTION  800

//---------------------------------------------------------------------------
/**
 * The format of the output : HTML, LATEX or XML.
 */
enum Output_Format { HTML, LATEX, XML };


/**
 * This class holds all parameters for the presentation.
 * The default-values for these parameters can be overridden by a config-file,
 * by the presentation itself and by commandline-options.
 */
class Parameters {
  public:
    //-----------------------------------------------------------------------
    /**
     * The constructor initializes the parameters to default-values.
     */
    Parameters();

    //-----------------------------------------------------------------------
    /**
     * Destructor.
     */
    ~Parameters();


    //-----------------------------------------------------------------------
    /**
     * This method reads the config-file stored in the member-variable
     * config_file_ and initializes the member_variables, which names
     * start with cf_.
     *
     * @return -1, if the file doesn't exist,
     *         else the number of errors in the config-file
     */
    int readConfigFile();


    //-----------------------------------------------------------------------
    /**
     * This method initializes the path to the selected theme,
     * which is stored in a subdirectory of the theme-path.
     *
     * @return  0, if the theme exists.
     *         -1, if the theme-path wasn't initialized.
     *         -2, if the selected theme doesn't have the file "config".
     */
    int initPathToTheme();

    //-----------------------------------------------------------------------
    /**
     * This method initializes the sizes of the background-images.
     * It must be called after the path to the theme was initialized.
     */
    void initImageSizes();


    //-----------------------------------------------------------------------
    /**
     * This method is used to determine the name of the config-file.
     *
     * @return The name of the config-file.
     */
    std::string &getConfigFile();

    //-----------------------------------------------------------------------
    /**
     * This method is used to determine the path to the themes.
     *
     * @return The path to the themes.
     */
    std::string getPathToThemes() const;

    //-----------------------------------------------------------------------
    /**
     * This method is used to determine the selected theme.
     *
     * @return the selected theme.
     */
    std::string &getTheme();

    //-----------------------------------------------------------------------
    /**
     * This method is used to determine, if the settings in the XML-file
     * were be disabled.
     *
     * @return 1, if the XML-settings were disabled, else 0.
     */
    int isXMLSettingsDisabled() const;

    //-----------------------------------------------------------------------
    /**
     * This method is used to determine the resolution for the HTML-output.
     *
     * @return The resolution (800 or 1024).
     */
    int getResolution() const;

    //-----------------------------------------------------------------------
    /**
     * This method is used to determine, if the background-graphic for the
     * slide is enabled or disabled.
     *
     * @return 1, if the background-graphic is enabled, else 0.
     */
    int isSlideBackgroundEnabled() const;

    //-----------------------------------------------------------------------
    /**
     * This method is used to determine, if the background-graphic for the
     * header is enabled or disabled.
     *
     * @return 1, if the background-graphic is enabled, else 0.
     */
    int isHeaderBackgroundEnabled() const;

    //-----------------------------------------------------------------------
    /**
     * This method is used to determine, if the First- and Last-button is
     * enabled or disabled.
     *
     * @return 1, if the First- and Last-button is enabled, else 0.
     */
    int isFirstAndLastButtonEnabled() const;

    //-----------------------------------------------------------------------
    /**
     * This method is used to determine, if the Note-button is
     * enabled or disabled.
     *
     * @return 1, if the Note-button is enabled, else 0.
     */
    int isNoteButtonEnabled() const;

    //-----------------------------------------------------------------------
    /**
     * This method is used to determine, if the sloppy-button option is
     * enabled or disabled.
     *
     * @return 1, if the sloppy-button option is enabled, else 0.
     */
    int isSloppyButtonsEnabled() const;

    //-----------------------------------------------------------------------
    /**
     * This method returns the value of the cellpadding-attribute for the
     * HTML-table of the grid-environment.
     *
     * @return The cellpadding-value.
     */
    int getCellPadding() const;

    //-----------------------------------------------------------------------
    /**
     * This method is used to determine, if the navigation-buttons should be
     * placed on the top of each slide.
     *
     * @return 1, if the navigation-buttons should be on the top, else 0.
     */
    int isTopNavigationEnabled() const;

    //-----------------------------------------------------------------------
    /**
     * This method is used to determine, if the LaTeX-output should be an
     * outline.
     *
     * @return 1, if the LaTeX-output should be an outline, else 0.
     */
    int isOutlineEnabled() const;


  public:

    //-----------------------------------------------------------------------
    /**
     * The name of the XML-file.
     */
    std::string xml_file_;

    //-----------------------------------------------------------------------
    /**
     * The name of the LaTeX file.
     */
    std::string latex_file_;

    //-----------------------------------------------------------------------
    /**
     * The name of the LaTeX outline file.
     */
    std::string latex_outline_file_;

    /**
     * The format of the output.
     */
    Output_Format output_format_;

    //-----------------------------------------------------------------------
    /**
     * The dimensions of the previous-button.
     */
    ImageSize *prev_;

    /**
     * The dimensions of the next-button.
     */
    ImageSize *next_;

    /**
     * The dimensions of the first-button.
     */
    ImageSize *first_;

    /**
     * The dimensions of the last-button.
     */
    ImageSize *last_;

    /**
     * The dimensions of the note-button.
     */
    ImageSize *note_;

    /**
     * The dimensions of the background-graphic for the slide.
     */
    ImageSize *slide_;

    /**
     * The dimensions of the background-graphic for the header.
     */
    ImageSize *header_;


    //-----------------------------------------------------------------------
    /**
     * The config-file, which will be read and evaluated.
     */
    std::string default_config_file_;

    /**
     * The config-file given on the command-line.
     */
    std::string cl_config_file_;


    //-----------------------------------------------------------------------
    /**
     * The default path to the directory which contains all themes.
     */
    std::string default_path_to_themes_;

    /**
     * The path to the themes set in the config-file.
     */
    std::string cf_path_to_themes_;

    /**
     * The path to the themes set with the commandline-option -p.
     */
    std::string cl_path_to_themes_;

    //-----------------------------------------------------------------------
    /**
     * The default theme for the HTML-output.
     */
    std::string default_html_theme_;

    /**
     * The theme for the HTML-output set in the config-file.
     */
    std::string cf_html_theme_;

    /**
     * The theme for the HTML-output set in the XML-file.
     */
    std::string xml_html_theme_;

    /**
     * The theme for the HTML-output set with the commandline-option -t.
     */
    std::string cl_html_theme_;

    //-----------------------------------------------------------------------
    /**
     * The full path to the selected theme (path_to_themes_ + theme_).
     */
    std::string full_path_to_theme_;

    //-----------------------------------------------------------------------
    /**
     * 1, if the settings stored in the XML-file should be disabled, else 1.
     */
    int default_disable_xml_settings_;

    /**
     * The disable-xml-settings option is :
     *  1, if set to "yes" in the config-file.
     *  0, if set to "no" in the config-file.
     * -1, if not given in the config-file. 
     */
    int cf_disable_xml_settings_;

    /**
     * 1, if the commandline-option -x was given, else -1.
     */
    int cl_disable_xml_settings_;
    
    //-----------------------------------------------------------------------
    /**
     * The default screen-resolution for the HTML-output.
     */
    int default_html_resolution_;

    /**
     * The screen-resolution for the HTML-output set in the config-file,
     * or -1, if the screen-resolution is not given.
     */
    int cf_html_resolution_;

    /**
     * The screen-resolution for the HTML-output set in the XML-file,
     * or -1, if the screen-resolution is not given.
     */
    int xml_html_resolution_;

    /**
     * 1, if the screen-resolution for the HTML-output was set with the
     * commandline-option -R, else -1.
     */
    int cl_html_resolution_;

    //-----------------------------------------------------------------------
    /**
     * 1, if the background-graphic for the slides should be disabled, else 0.
     */
    int default_html_disable_background_graphic_;

    /**
     * 1, if the commandline-option -B was given, else -1.
     */
    int cl_html_disable_background_graphic_;

    //-----------------------------------------------------------------------
    /**
     * 1, if the background-graphic for the header should be disabled, else 0.
     */
    int default_html_disable_header_graphic_;

    /**
     * 1, if the commandline_option -H was given, else -1.
     */
    int cl_html_disable_header_graphic_;

    //-----------------------------------------------------------------------
    /**
     * 1, if buttons to the first and last slide should be disabled, else 0.
     */
    int default_html_disable_first_last_button_;

    /**
     * 1, if the commandline-option -F was given, else -1.
     */
    int cl_html_disable_first_last_button_;

    //-----------------------------------------------------------------------
    /**
     * 1, if the note-button should be disabled, else 0.
     */
    int default_html_disable_note_button_;

    /**
     * 1, if the commandline-option -N was given, else -1.
     */
    int cl_html_disable_note_button_;

    //-----------------------------------------------------------------------
    /**
     * 0, if the buttons should be drawn at a fixed position.
     * 1, if the buttons should be drawn below the content of the slide.
     */
    int default_html_sloppy_buttons_;

    /**
     * The sloppy-buttons option is :
     *  1, if set to "yes" in the config-file.
     *  0, if set to "no" in the config-file.
     * -1, if not given in the config-file. 
     */
    int cf_html_sloppy_buttons_;

    /**
     * The sloppy-buttons option is :
     *  1, if set to "yes" in the XML-file.
     *  0, if set to "no" in the XML-file.
     * -1, if not given in the XML-file.
     */
    int xml_html_sloppy_buttons_;

    /**
     * 1, if the commandline-option -S was given, else -1.
     */
    int cl_html_sloppy_buttons_;

    //-----------------------------------------------------------------------
    /**
     * The default-value of the table-attribute 'cellpadding'
     * used in the grid-environment.
     */
    int default_html_cellpadding_;

    /**
     * The cellpadding option set in the XML-file, or -1 if not given.
     */
    int xml_html_cellpadding_;

    //-----------------------------------------------------------------------
    /**
     * 1, if all navigation-buttons should be on the top of the slide, else 0.
     */
    int default_html_top_navigation_;

    /**
     * The top-navigation option is :
     *  1, if set to "yes" in the config-file.
     *  0, if set to "no" in the config-file.
     * -1, if not given in the config-file.
     */
    int cf_html_top_navigation_;

    /**
     * The top-navigation option is :
     *  1, if set to "yes" in the XML-file.
     *  0, if set to "no" in the XML-file.
     * -1, if not given in the XML-file.
     */
    int xml_html_top_navigation_;

    /**
     * 1, if the commandline-option -T was given, else -1.
     */
    int cl_html_top_navigation_;

    //-----------------------------------------------------------------------
    /**
     * 0, if the LaTeX-output should use the documentclass 'seminar'.
     * 1, if the documentclass 'article' should be used.
     */
    int default_latex_print_outline_;

    /**
     * The print-outline option is :
     *  1, if set to "yes" in the config-file.
     *  0, if set to "no" in the config-file.
     * -1, if not given in the config-file.
     */
    int cf_latex_print_outline_;

    /**
     * The outline option is :
     *  1, if set to "yes" in the XML-file.
     *  0, if set to "no" in the XML-file.
     * -1, if not given in the XML-file.
     */
    int xml_latex_print_outline_;

    /**
     * 1, if the commandline parameter -O was given, else -1.
     */
    int cl_latex_print_outline_;
};

#endif //PARAMETERS_H
