package AtomicData::Decimal;

# Copyright (c) 2000, FundsXpress Financial Network, Inc.
# This library is free software released under the GNU Lesser General
# Public License, Version 2.1.  Please read the important licensing and
# disclaimer information included below.

# $Id: Decimal.pm,v 1.1 2000/11/23 23:36:15 muaddib Exp $

use strict;

=head1 NAME

AtomicData::Decimal - data encapsulator for straight decimal numbers

=head1 SYNOPSIS

see AtomicData.pm

=head1 DESCRIPTION

Implements data encapsulation for simple base ten
numbers. Specifically, scientific notation is not supported. This is
the package to use to encapsulate floating point numbers.

=cut

use AtomicData::Text;
@AtomicData::Decimal::ISA = qw (AtomicData::Text);

=head1 METHODS

For further methods, see root class.

=head2 parameters

For base documentation, see root class.

Will accept the parameters 'min_value'/number, 'max_value'/number
which constrain the data between these values inclusively.

=head2 set_format

for base documentation, see root class.

Will accept the parameters: 

'commas'/bool which will add a comma every third digit right from the
decimal point or end of value. The defualt is false.

'significant'/+int which will pad/truncate and round so that the float
has the specified number of significant digits. Note that since
rounding may occur, it is necessary to call C<format> before operating
on value to stay within the bounds. The default value is infinite.

'truncate'/bool will cause the truncation (not rounding as is the
default) of non-significant digits; works in conjuction with above and
has no affect if the above is not set.

'lead_zero'/bool will cause a leading zero (before the decimal point)
to be added if there is no digit. The default is false.

=head2 _canonicalize

 instance/private
 (float $value) _canonicalize (float $raw_value)

DESCRIPTION:

This is a private fucntion that will attempt to canonicalize the
encapsulated data. Will return the canonical form if successful and
set C<$this->{_can_failed}> if there was a problem and return
undef.

=cut

sub _canonicalize {
  my ($this, $val) = @_;

  # perform text canonicalization.
  $val = $this->SUPER::_canonicalize($val);

  # blank is already canonical.
  $val eq "" and return $val;

  #remove all valid commas
  $val =~ s/,(\d{3})\b/$1/g;
  #remove trailing .
  $val =~ s/(?!.*\..*)\.$//;

  # there must be a digit for it to be a number.
  if ($val =~ /^[^\d]+$/) {
    $this->{_can_failed} = [$&.' is not a valid number.'];
    return undef;
  }

  # decimal should fit this form.
  if ( $val !~ /^([-\+]?)(\d*)\.?(\d*)?(e(\d+))?$/ ) {
    $this->{_can_failed} = ["'$val' is not a decimal."];
    return undef;
  }
  my ($sign,$int,$frac,$exp) = ($1,$2,$3,$5);

  #trim leading zero's
  $int =~ s/^0+//;
  #add leading zero if we backed up to end or decimal point
  $int or $int = "0";

  #trim trailing zero's
  $frac =~ s/0+$//;

  #trim sign if all we have is "-0"
  $sign = "" if (!$int and !$frac);

  my $out = $sign.$int;
  $frac and $out .= ".$frac";
  $exp and $out .= "e$exp";

  return $out;
}

=head2 _verify

 instance/private
 (bool $verified, string \@errors) _verify (variable $value)

DESCRIPTION:

Verifies that the data falls within the set parameters noted
above. See C<verify()> in root class for behavior.

=cut

sub _verify {
  my ($this) = @_;

  my ($bool,$problems) = $this->SUPER::_verify();
  $bool or return (0, $problems);

  my $value = $this->canonical_value();
  my $parameters = $this->parameters();

  my $min_value = $parameters->{min_value};
  my $max_value = $parameters->{max_value};

  if (defined($min_value) and ($value < $min_value)) {
    return (0, ['The value must be greater than or equal to '.$min_value]);
  }
  if (defined($max_value) and ($value > $max_value)) {
    return (0, ['The value must be less than or equal to '.$max_value]);
  }
  if ($parameters->{not_negative} and $value < 0) {
    return (0, ['The value cannot be negative.']);
  }

  return (1, []);
}

=head2 test_equality

 instance
 (bool $equal) test_equility (variable $val1, variable $val2)

DESCRIPTION:

Compares the values for numerical equivalency. This should be
sufficient for all children of this class.

=cut

sub test_equality {
  my ($this, $x, $y) = @_;  
  return ($x == $y);
}

=head2 _format

 instance/private
 (string $value) _format (float $canonical_value)

DESCRIPTION:

Will accept the canonical value and format it according to the values
of the format hash, as noted above in C<set_format>.

=cut

sub _format {
  my ($this, $value) = @_;

  $value =~ /^([-\+]?)(\d*)\.(\d*)?(e(\d+))?$/ or return;
  my ($sign,$int,$frac,$exp) = ($1,$2,$3,$5);

  if ($this->{_format}->{commas}) {
    $int = $this->commify($int);
  }

  my $sig;
  if (defined($sig = $this->{_format}->{significant})) {
    my $length = length($frac);

    if ($sig < 0) {
      ;
    } elsif ($length < $sig) {
      $frac .= '0' x ($sig - $length);
    } elsif ($length > $sig) {
      if (! $this->{_format}->{truncate}) { #we have to round
	my $first_ns = substr($frac,$sig,($sig + 1));
	if ($first_ns >= 5) {
	  if ($sig == 0) {
	    $int++;
	  } else {
	    substr($frac,($sig-1),$sig) = substr($frac,($sig-1),$sig) + 1;
	  }
	}
      }
      $frac = substr($frac,0,$sig);
    }
  }

  $int  = ""  if $this->{_format}->{strip_lead_zero} and !$int;
  $sign = "+" if $this->{_format}->{show_plus} and ! $sign;

  my $out = $sign.$int;
  $frac and $out .= ".$frac";
  $exp and $out .= "e$exp";

  return $out;
}

=head2 commify

 public
 (string $commified) commify (int $number)

DESCRIPTION:

Properly commifies C<$number> which must be an integer.

=cut

sub commify {
  my ($this,$number) = @_;

  #we have same number, first, we need what is before the decimal, if
  #there is any
    
  #this will insert a comma every third digit from the right form
  #either the end of the string or another comma, the [^,] asserts
  #that the character preceding the third digit from the right is not
  #a , and this casues the while loop to termintate
  while ($number =~ s/([^,])(\d{3}(\Z|,))/$1,$2/) {}

  return $number;
}

1;
__END__

=head1 BUGS

No known bugs, but this does not mean no bugs exist.

=head1 SEE ALSO

L<AtomicData>, L<HTMLIO>, L<Field>.

=head1 COPYRIGHT

 PSP - Perl Server Pages
 Copyright (c) 2000, FundsXpress Financial Network, Inc.

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2 of the License, or (at your option) any later version.

 BECAUSE THIS LIBRARY IS LICENSED FREE OF CHARGE, THIS LIBRARY IS
 BEING PROVIDED "AS IS WITH ALL FAULTS," WITHOUT ANY WARRANTIES
 OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING, WITHOUT
 LIMITATION, ANY IMPLIED WARRANTIES OF TITLE, NONINFRINGEMENT,
 MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, AND THE
 ENTIRE RISK AS TO SATISFACTORY QUALITY, PERFORMANCE, ACCURACY,
 AND EFFORT IS WITH THE YOU.  See the GNU Lesser General Public
 License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA

=cut
