package AtomicData::Email;

# Copyright (c) 2000, FundsXpress Financial Network, Inc.
# This library is free software released under the GNU Lesser General
# Public License, Version 2.1.  Please read the important licensing and
# disclaimer information included below.

# $Id: Email.pm,v 1.1 2000/11/23 23:36:15 muaddib Exp $

use strict;

=head1 NAME

AtomicData::Email - for storing email addresses

=head1 SYNOPSIS

see AtomicData.pm

=head1 DESCRIPTION

Implements data encapsulation for email strings

=cut

use AtomicData::Text;
@AtomicData::Email::ISA = qw(AtomicData::Text);

use vars qw($RE_specials);
$RE_specials = "[".quotemeta("()<>@,;:.[]\"\\")."]";

=head1 METHODS

see AtomicData.pm for further methods.

=head2 parameters

see parent class for additional documentation. This class directly
supports 'us'/bool which, if set, will require that the address be a
valid US hosted email address. The default is false.
Supports 'top_level'/bool which if set will require the address to contain
at least one top level domain. The default is false.

=head2 _canonicalize

 instance/private
 (int $value) _canonicalize (int $raw_value)

DESCRIPTION:

Will accept a value and return the cannonical form or undef and set
C<$this->{_can_failed}>.

This conforms to RFC 822 except it does not allow group addressing or 
routing.

=cut

sub _canonicalize {
  my ($this, $val) = @_;

  #this will destroy leading and trailing space
  $val = $this->SUPER::_canonicalize($val);
  $val eq "" and return $val;

  # this simple form is no longer used.
  # unless ($val =~ /^([\w\d\.!\#\$\%\&\'\*\+\-\/]+\@([\w\-]+\.)+\w+)?$/) {
  #   $this->{_can_failed} = ['Invalid email address.'];
  #   return undef;
  # }

  my $email = $val;

  # break the email into name and email components.
  my ($phrase,$qphrase,$addrspec);
  if ($email =~ /^"(.+?)"\s*<([^>]+)>$/) {
    $qphrase = $1;
    $addrspec = $2;
    if ($qphrase =~ /[\"\\]/) {
      $this->{_can_failed} = ["Email quoted phrase contains illegal characters."];
      return undef;
    }
  } elsif ($email =~ /^(.+?)\s*<([^>]+)>$/) {
    $phrase = $1;
    $addrspec = $2;
    if ($phrase =~ /$RE_specials/) {
      $this->{_can_failed} = ["Email phrase contains illegal characters"];
      return undef;
    }
  } elsif ($email =~ /^(.*?)\s*\[mailto:([^\]]+)\]$/) {
    $phrase = $1;
    $addrspec = $2;
    if ($phrase =~ /$RE_specials/) {
      $this->{_can_failed} = ["Email phrase contains illegal characters"];
      return undef;
    }
      
  } else {
    $addrspec = $email;
  }

  # make sure the address component contains 1 and only 1 @
  my $count = $addrspec =~ tr/@/@/;
  if ( $count > 1 ) {
    $this->{_can_failed} = ["Email address contains multiple '\@'s."];
    return undef;
  } elsif ( $count < 1 ) {
    $this->{_can_failed} = ["Email must contain at least one '\@'."];
    return undef;
  }

  # make sure the address contains a local-part and a domain.
  if ( $addrspec !~ /(.+)\@(.+)/ ) {
    $this->{_can_failed} = ["Email address does not contain a local-part and domain."];
    return undef;
  }
  my $localpart = $1;
  my $domain   = $2;

  # LOCAL-PART check

  # make sure the local-part begins and ends with a valid character.
  if ($localpart =~ /^\.|\.$/) { 
    $this->{_can_failed} = ['Email address begins or ends with an invalid character'];
    return undef;
  }

  # check components of the username
  my @userparts = split(/\./,$localpart);
  for my $part (@userparts) {

    # null entry means two dots in a row
    if ( $part eq "" ) {  
      $this->{_can_failed} = ['Email address contains ".."'];
      return undef;
    }

    # check for illegal characters.
    if ($part =~ /$RE_specials/) {
      $this->{_can_failed} = ['Email address contains illegal characters'];
      return undef;
    }

  }#for @userparts
    
  # DOMAIN check

  # make sure the domain begins and ends with a valid character.
  if ($domain =~ /^\.|\.$/) { 
    $this->{_can_failed} = ['Email domain begins or ends with an invalid character'];
    return undef;
  }

  # check components of the domain
  my @domainparts = split(/\./,$domain);
  for my $part (@domainparts){

    # null entry means two dots in a row
    if ( $part eq "" ) {  
      $this->{_can_failed} = ['Email domain contains ".."'];
      return undef;
    } 

    # check for illegal characters.
    if ( $part =~ /$RE_specials/ ) {
      $this->{_can_failed} = ['Email domain contains illegal characters'];
      return undef;
    }

    # check for illegal characters.
    if ( $part =~ /$RE_specials/ ) {
      $this->{_can_failed} = ['Email domain contains illegal characters'];
      return undef;
    }

    # checking for beginning or ending dashes.
    if ( $part =~ /^-|-$/ ) {
      $this->{_can_failed} = ['Email domain contains illegal characters'];
      return undef;
    }

  }# for @domainparts

  if ($qphrase) {
    $val = "\"$qphrase\" <$addrspec>";
  } elsif ($phrase) {
    $val = "$phrase <$addrspec>";
  } else {
    $val = $addrspec;
  }
  
  return $val;
}

=head2 _verify

 instance/private
 (bool, string \@errors) _verify ()

DESCRIPTION:

Will check an the parameters set above.

=cut

sub _verify {
  my ($this) = @_;

  my ($bool,$problems) = $this->SUPER::_verify();
  $bool or return (0, $problems);

  my $value = $this->canonical_value();
  my $parameters = $this->parameters();

  if ($parameters->{us} and 
      $value !~ /(\.com|\.org|\.net|\.gov|\.mil|\.edu|\.us)>?$/) {
    push @$problems, 'Must be a US hosted email address.';
  }

  #top level checks that there is at least one top level domain
  if ($parameters->{top_level} and $value !~ /@.*\..*/) {
    push @$problems, 'Must contain at least one top level domain';
  }

  @$problems and return (0, $problems);
  return (1, []);
}

1;
__END__

=head1 BUGS

No known bugs, but this does not mean no bugs exist.

=head1 SEE ALSO

L<AtomicData>, L<HTMLIO>, L<Field>.

=head1 COPYRIGHT

 PSP - Perl Server Pages
 Copyright (c) 2000, FundsXpress Financial Network, Inc.

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2 of the License, or (at your option) any later version.

 BECAUSE THIS LIBRARY IS LICENSED FREE OF CHARGE, THIS LIBRARY IS
 BEING PROVIDED "AS IS WITH ALL FAULTS," WITHOUT ANY WARRANTIES
 OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING, WITHOUT
 LIMITATION, ANY IMPLIED WARRANTIES OF TITLE, NONINFRINGEMENT,
 MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, AND THE
 ENTIRE RISK AS TO SATISFACTORY QUALITY, PERFORMANCE, ACCURACY,
 AND EFFORT IS WITH THE YOU.  See the GNU Lesser General Public
 License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA

=cut
