package AtomicData::RTNumber;

# Copyright (c) 2000, FundsXpress Financial Network, Inc.
# This library is free software released under the GNU Lesser General
# Public License, Version 2.1.  Please read the important licensing and
# disclaimer information included below.

# $Id: RTNumber.pm,v 1.1 2000/11/23 23:36:15 muaddib Exp $

use strict;

=head1 NAME

AtmoicData::RTNumber - data encapsulator for routing and transit numbers

=head1 SYNOPSIS

see AtomicData.pm

=head1 DESCRIPTION

Implements data encapsulation for simple base ten financial routing
and transit numbers. See AtomicData.pm for further details.

=cut

use AtomicData::Text;
@AtomicData::RTNumber::ISA = qw (AtomicData::Text);

=head1 METHODS

see AtomicData.pm for further methods.

=head2 parameters

see parent class for more documentation.

Will take note of parameters 'check_digit'/trinary which may be either
-1, 0 or 1 indicating none, either or required wrt the existance of
the check digit on the string. Zero is the defualt.

=head2 _canonicalize

 instance/private
 (int $value) _canonicalize (int $raw_value)

DESCRIPTION:

Will accept a value and return the cannonical form or undef and set
C<$this->{_can_failed}>.

=cut

sub _canonicalize {
  my ($this, $val) = @_;

  $val = $this->SUPER::_canonicalize($val);
  defined $val or return;
  $val eq "" and return $val;

  # should be only digits
  if ($val !~ /^(\d{8})(\d)?$/) {
    push @{$this->{_can_failed}}, 
      ('Routing and transit number must be either 8',
       ' or 9 (with check digit) digits long.');
    return undef;
  }
  $val = $1;
  my $check_digit = $2;

  #val must be 8 digits long at this point
  my $check = calc_check_digit($val);

  if ($check == -1) {
    push @{$this->{_can_failed}}, 
      "Invalid R&T number; cannot generate a check digit.";
    return undef;
  } elsif (defined($check_digit) and ($check != $check_digit)) {
    push @{$this->{_can_failed}}, 
      "Check digit is invalid or there is a typo in the R&T number.";
    return undef;
  }

  if (!defined $this->{_parameters}->{check_digit} or
      $this->{_parameters}->{check_digit} != -1) {
    $val .= $check;
  }
  
  return $val;
}

=head2 _verify

 instance/private
 (bool $verified, string \@errors) _verify (variable $value)

DESCRIPTION:

Private method used to verify that the data falls within the set
parameters.

=cut

sub _verify {
  my ($this) = @_;

  my ($bool,$problems) = $this->SUPER::_verify();
  $bool or return (0, $problems);

  my $raw_value = $this->raw_value();
  my $value = $this->canonical_value();
  my $parameters = $this->parameters();

  if ($parameters->{min_value} and $parameters->{min_value}) {
    ($parameters->{min_value} >= 8 and $parameters->{min_value} <= 9) and
      delete $parameters->{min_value};
    ($parameters->{max_value} >= 8 and $parameters->{max_value} <= 9) and
      delete $parameters->{max_value};
  }

  ($bool,$problems) = $this->SUPER::_verify();
  $bool or return ($bool, $problems);

  #we have to do verification here because
  #if we use _verify we have the canonicalize
  #value and the verify function can't check to 
  #see if we started with 8 or 9 digits.

  if (length($value)) {
    ##blank value might be ok.  let AtomicData.pm worry about that.
    ##only do verification if value is not blank

    if (! $parameters->{check_digit}) {
      if ((length($raw_value) != 8) && (length($raw_value) != 9)) {
	return (0, ['R&T Number must be 8 or 9 digits long.']);
      }
    }
    elsif ($parameters->{check_digit} == -1) {
      if (length($raw_value) != 8) {
	return (0, ['Check digit must not be included.']);
      }
    }
    else {
      if (length($raw_value) != 9) {
	return (0, ['Check digit must be included.']);
      }
    }
  }
  return (1, []);
}

=head2 _format

 instance/private
 (integer $value) _format (integer $canonical_value)

DESCRIPTION:

Will accept a routing and transit number in the canonical form and
format as indicated by the hash passed to set_format.

=cut

sub _format {
  my ($this, $val) = @_;

  if ($this->{_format}->{check_digit} == -1) {
    if (length($val) == 9) {
      chop $val;
    }
  }
  elsif ($this->{_format}->{check_digit} == 1) {
    if (length($val) == 8) {
      my $check_digit = calc_check_digit($val);
      $val .= $check_digit;
    }
  }

  return $val;
}

=head2 calc_check_digit

 private package
 (int $check_digit) calc_check_digit (int $rt_number)

DESCRIPTION:

C<$rt_number> must be 8 digits and this function will return a single
digit C<$check_digit>, which will be -1 if C<$rt_number> is invalid.

=cut

sub calc_check_digit {
  my ($szIDNum) = @_;
  my(@Weights) = (3, 7, 1);
  my($nChar, $n);
  my($nPos)=0;
  my($nSum)=0;
  my($nSumFactor10);
  if ($szIDNum !~ /\d{8}/) {
    return -1;
  }
  for($n = 0; $n < 8; $n += 1) {
    $nChar = substr($szIDNum, $n, 1);
    $nSum += ($nChar * $Weights[$nPos]);
    
    $nPos++;
    if ($nPos >2) {
      $nPos = 0;
    }
  }
  $nSumFactor10 = $nSum;
  while ($nSumFactor10 % 10) {
    $nSumFactor10++;
  }

  return ($nSumFactor10 - $nSum);
}

1;
__END__

=head1 BUGS

No known bugs, but this does not mean no bugs exist.

=head1 SEE ALSO

L<AtomicData>, L<HTMLIO>, L<Field>.

=head1 COPYRIGHT

 PSP - Perl Server Pages
 Copyright (c) 2000, FundsXpress Financial Network, Inc.

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2 of the License, or (at your option) any later version.

 BECAUSE THIS LIBRARY IS LICENSED FREE OF CHARGE, THIS LIBRARY IS
 BEING PROVIDED "AS IS WITH ALL FAULTS," WITHOUT ANY WARRANTIES
 OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING, WITHOUT
 LIMITATION, ANY IMPLIED WARRANTIES OF TITLE, NONINFRINGEMENT,
 MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, AND THE
 ENTIRE RISK AS TO SATISFACTORY QUALITY, PERFORMANCE, ACCURACY,
 AND EFFORT IS WITH THE YOU.  See the GNU Lesser General Public
 License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA

=cut
