package HTMLIO::Utils;

# Copyright (c) 2000, FundsXpress Financial Network, Inc.
# This library is free software released under the GNU Lesser General
# Public License, Version 2.1.  Please read the important licensing and
# disclaimer information included below.

# $Id: Utils.pm,v 1.2 2001/02/10 22:21:50 muaddie Exp $

use strict;

=head1 NAME

HTMLIO::Utils - provide utility functions for HTMLIO and its children

=head1 SYNOPSIS

this is a private library, see the children of HTMLIO for usage examples

=head1 DESCRIPTION

A library package.

=cut

use Exporter;
use URI::Escape;  # re-export uri_escape from this
use PSP::HTML::Entities;

BEGIN {
  @HTMLIO::Utils::ISA = qw(Exporter);
  @HTMLIO::Utils::EXPORT_OK = qw(
    uri_escape
    uri_unescape
    cgi_escape
    cgi_unescape
    html_escape
    html_unescape
    html_tag
    stnd_html_input
    con_html_input
   );
  $HTMLIO::Utils::EXPORT_TAGS{all} = \@HTMLIO::Utils::EXPORT_OK;
}

# This is the character class of unsafe chars in RFC 2396 ^uric
# ^;\/?:@&=+\$,A-Za-z0-9\-_.!~*'()
# This is the character class of unsafe chars in some previous version.
# \x00-\x20"#%;+<>?&{}|\\\\^~`\[\]\x7F-\xFF

use vars qw($unsafe_cgi_uri_characters);
$unsafe_cgi_uri_characters = '^;\/:@\$,A-Za-z0-9\-_.!~*\'()';

=head2 uri_escape

  (string) uri_escape(string)

Escape unsafe URI characters.  (Re-exported from URI::Escape)

=head2 uri_unescape

  (string) uri_unescape(string)

Unescape escaped URI characters.  (Re-exported from URI::Escape)

=head2 cgi_escape

  (string) cgi_escape(string).

Escape unsafe characters.  The unsafe characters are characters unsafe
according to RFC 2396 ^uric, and additionally, characters that have special
meaning in CGI parameter query strings.

=cut

sub cgi_escape ($) {
  return URI::Escape::uri_escape($_[0],$unsafe_cgi_uri_characters);
}

=head2 cgi_unescape

  (string) cgi_unescape(string)

Unescape URI escaped characters.

=cut

sub cgi_unescape ($) {
  URI::Escape::uri_unescape(@_);
}

=head2 html_escape

  (string) html_escape(string)

Return the HTML-escaped string.  This function wins over
PSP::HTML::Entities::encode (which it uses) in that it does not modify its
input string, so you can pass it a read-only value.

=cut

sub html_escape {
  return PSP::HTML::Entities::encode(join("",@_));
}

=head2 html_unescape

  (string) html_unescape(string)

Return the HTML-unescaped string.  This function wins over
PSP::HTML::Entities::decode (which it uses) in that it does not modify its
input string, so you can pass it a read-only value.

=cut

sub html_unescape {
  return PSP::HTML::Entities::decode(join("",@_));
}

=head2 stnd_html_input

global
(string) stnd_html_input (string $type, \Data[] $data,
		        \int[] $slice, string $join, \hash $attributes)

DESCRIPTION:

=cut

sub html_tag {
  my ($type,$orig_attr,$attr_order) = @_;
  $orig_attr ||= {};

  my %attr = %$orig_attr;
  $attr_order ||= [ sort keys %attr ];

  my $checked = delete $attr{checked} ? " checked" : "";
  my $selected = delete $attr{selected} ? " selected" : "";
  my $attr_string = "";
  for my $att (@$attr_order) {
    next unless defined $attr{$att};
    $attr_string .= " $att=\"".html_escape($attr{$att}).'"';
    delete $attr{$att};
  }

  for my $att (sort keys %attr) {
    $attr_string .= " $att=\"".html_escape($attr{$att}).'"';
  }
  $attr_string .= $checked.$selected;

  return "<$type$attr_string>";
}

=head2 stnd_html_input

global
(string) stnd_html_input (string $type, \Data[] $data,
		        \int[] $slice, string $join, \hash $attributes)

DESCRIPTION:

Standard HTML input; will accept the given arguments and return the
proper string of html input(s). The C<$type> is any valid value for
the HTML attribute 'type' when used in the input element. C<$data>
must be a reference to a list of HTMLIO compliant atomic datatype
encapsulators or a reference to an empty list. If $data references an
empty list, then a single, empty, control box will be created. See TOF
documentation on HTMLIO for more information. C<$slice> is a
reference to a list of the indexes of the list referenced by C<$data>
for which an HTML input should be generated. If there is more than
one, then each input element will be joined on the value of
C<$join>. Finally C<$attributes> is a reference to hash whose keys are
attribute names and whose values are the values for that key.

=cut

sub stnd_html_input {
  my ($name, $type, $data, $slice, $join, $orig_attr) = @_;

  defined $name or throw Error::Simple("stnd_html_input requires a \$name");
  $type      ||= "text";
  $join      ||= "\n";

  $orig_attr ||= {};
  $orig_attr->{name} = $name;
  $orig_attr->{type} = $type;

  # derive our data slice from $data and $slice.
  my $sliced_data;
  if ($slice) {
    $sliced_data = [];
    for my $index (@$slice) {
      if ($index > $#{$data} or !defined $data->[$index]) {
	push @$sliced_data, undef;
      } else {
	push @$sliced_data, $data->[$index];
      }
    }
    @$sliced_data or push @$sliced_data, "";
  } elsif ($data) {
    $sliced_data = @$data ? $data : [ "" ];
  } else {
    $sliced_data = [ "" ];
  }

  my @html;
  for my $data (@$sliced_data) {
    my %attr = %$orig_attr;
    $attr{value} = $data if defined $data;
    push @html, html_tag('input',\%attr,[qw(name type value)]);
  }

  return join $join, @html;
}

=head2 con_html_input

global
(string) con_html_input (string $type, \Data[] $data,
		        \int[] $slice, string $join, \hash $attributes)

DESCRIPTION:

Standard HTML input for constrained data; will accept the given
arguments and return the proper string of html input(s). The C<$type>
is any valid value for the HTML attribute 'type' when used in the
input element. C<$data> must be a reference to a list of HTMLIO
compliant atomic datatype encapsulators. See TOF documentation on
HTMLIO for more information. C<$slice> is a reference to a list of
the keys in the hash passed by C<set_possible>. If there is more than
one, then each input element will be joined on the value of
C<$join>. Finally C<$attributes> is a reference to hash whose keys are
attribute names and whose values are the values for that key.

=cut

sub con_html_input {
  my ($name, $type, $possible, $data, $slice, $join, $orig_attr) = @_;

  defined $name or throw Error::Simple("con_html_input requires a \$name");
  $type      ||= "text";
  $possible  ||= {};
  $data      ||= [];
  $slice     ||= [ sort keys %$possible ];
  $join      ||= "\n";
  $orig_attr ||= {};
  $orig_attr->{name} = $name;
  $orig_attr->{type} = $type;

  my %checked = map { $_ => $_ } @$data;

  my @html;
  for my $key (@$slice) {
    my %attr = %$orig_attr;
    $attr{value} = $key;
    $checked{$key} and $type =~ /checkbox|radio/i and
      $attr{checked} = "true";
    push @html, html_tag('input',\%attr,[qw(name type value)]);
    defined $possible->{$key} and $html[-1] .= $possible->{$key};
  }

  return join $join, @html;
}

1;
__END__

=head1 BUGS

No known bugs, but this does not mean no bugs exist.

=head1 SEE ALSO

L<AtomicData>, L<HTMLIO>, L<Field>.

=head1 COPYRIGHT

 PSP - Perl Server Pages
 Copyright (c) 2000, FundsXpress Financial Network, Inc.

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2 of the License, or (at your option) any later version.

 BECAUSE THIS LIBRARY IS LICENSED FREE OF CHARGE, THIS LIBRARY IS
 BEING PROVIDED "AS IS WITH ALL FAULTS," WITHOUT ANY WARRANTIES
 OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING, WITHOUT
 LIMITATION, ANY IMPLIED WARRANTIES OF TITLE, NONINFRINGEMENT,
 MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, AND THE
 ENTIRE RISK AS TO SATISFACTORY QUALITY, PERFORMANCE, ACCURACY,
 AND EFFORT IS WITH THE YOU.  See the GNU Lesser General Public
 License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA

=cut
