/*
 * rtstreams.stdlib.cpp
 * 
 * Copyright (c) 2000-2004 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

////////////// File Streams using the STL ////////////////////

#ifndef __LRT_STREAMS_STDLIB__
#define __LRT_STREAMS_STDLIB__

#include "rtstreams.h"

#include "rtstring.h"
#include "rtfile.h"

#include <stdio.h>

namespace lrt {

FileInputStream::FileInputStream(const String &filename, bool textMode) :
	open(false), markPos(0)
{
	String fname = File(filename).getName();
	handle = ::fopen(fname.cStr(), (textMode ? "rt" : "rb"));
	if(handle) open = true;
	else _fail = true;
}

FileInputStream::FileInputStream(const File &file, bool textMode) : 
	open(false), markPos(0)
{
	String fname = file.getName();
	handle = ::fopen(fname.cStr(), (textMode ? "rt" : "rb"));
	if(handle) open = true;
	else _fail = true;
}

bool FileInputStream::markSupported()
{
	return true;
}

void FileInputStream::mark()
{
	if(!open) return;
	markPos = ::ftell((FILE*) handle);
}

void FileInputStream::reset()
{
	if(!open) return;
	::fseek((FILE*) handle, markPos, SEEK_SET);
}

int FileInputStream::read()
{
	if(!open) return -1;
	unsigned char ret;
	int rval = ::fread(&ret, 1, 1, (FILE*) handle);
	if(rval == 0)
	{
		if(ferror((FILE*) handle)) _fail = true;
		return -1;
	
	}
	return ret;
}

bool FileInputStream::eos()
{
	if(!open) return true;
	return (feof((FILE*) handle) != 0);
}

void FileInputStream::close()
{
	if(open) {
		::fclose((FILE*) handle);
		open = false;
	}
}

FileInputStream::~FileInputStream()
{
	close();
    // delete handle;  // must not be deleted!
}



FileOutputStream::FileOutputStream(const String &fileName, bool append)
	: open(false), handle(0)
{
	String fname = File(fileName).getName();
	handle = ::fopen(fname.cStr(), (append ? "ab" : "wb"));
	if(handle) open = true;
	else _fail = true;
}

FileOutputStream::FileOutputStream(const File &file, bool append)
	: open(false), handle(0)
{
	String fname = file.getName();
	handle = ::fopen(fname.cStr(), (append ? "ab" : "wb"));
	if(handle) open = true;
	else _fail = true;
}

bool FileOutputStream::write(int b)
{
	if(!open) return false;
	char temp = (char)b;
	int rval = ::fwrite(&temp, 1, 1, (FILE*) handle);
	if(rval == 0)
	{
		_fail = true;
		return false;
	}
	else return true;
}

bool FileOutputStream::write(const Array<char> &b, int off, int len)
{
	if(!open) return false;
	const char *ptr = b.getData();
	int rval = ::fwrite(ptr + off, 1, len, (FILE*) handle);
	if(rval < len)
	{
		_fail = true;
		return false;
	}
	else return true;
}

void FileOutputStream::flush()
{
	if(!open) return;
	::fflush((FILE*) handle);
}

void FileOutputStream::close()
{
	if(open && handle) {
		::fclose((FILE*) handle);
		open = false;
	}
}

FileOutputStream::~FileOutputStream()
{
	close();
    // delete handle;  // must not be deleted!
}

} // namespace

#endif

