#  Copyright (C) 1999-2004
#  Smithsonian Astrophysical Observatory, Cambridge, MA, USA
#  For conditions of distribution and use, see copyright notice in "copyright"

package provide DS9 1.0

proc IISInit {} {
    global iis

    iis open $iis(ififo) $iis(ofifo) $iis(port) $iis(unix)
}

proc IISClose {} {
    iis close
}

proc IISDebug {} {
    global debug

    iis debug $debug(iis)
}

# Callbacks

proc IISInitializeCmd {w h} {
    global iis

    # default frame size
    set iis(width) $w
    set iis(height) $h
}

proc IISInitFrameCmd {which} {
}

proc IISSetDisplayFrameCmd {which w h} {
    global iis

    IISGotoFrame $which
    IISLoadFrame $which
}

proc IISSetRefFrameCmd {which} {
    global iis
    global ds9

    if {[lsearch $ds9(frames) Frame$which] == -1} {
	return {[NOSUCHFRAME]}
    }

    if {[Frame$which has iis]} {
	return {}
    }

    set filename [Frame$which get iis file name $iis(x) $iis(y)]

    if { [string compare [string index $filename 0] "/"] } {
        set filename [file join [pwd] $filename]
    }

    return "$filename 1. 0. 0. 1. 0. 0. 1. 32767. 1."
}

proc IISEraseFrameCmd {which} {
    Frame$which iis erase
    IISResetTimer $which
}

proc IISMessageCmd {message} {
    global current

    $current(frame) iis message "\"$message\""
    RefreshInfoBox $current(frame)
}

proc IISWritePixelsCmd {which ptr x y dx dy} {
    Frame$which iis set $ptr $x $y $dx $dy
    IISResetTimer $which
}

proc IISReadPixelsCmd {which ptr x y dx dy} {
    global current

    if {$which > 0} {
	Frame$which get iis $ptr $x $y $dx $dy
    } else {
	$current(frame) get iis $ptr $x $y $dx $dy
    }
    IISResetTimer $which
}

proc IISWCSCmd {which a b c d e f z1 z2 zt} {
    # if there is a change in config, we are not told until now
    if {$which > 0} {
	IISLoadFrame $which
	Frame$which iis wcs $a $b $c $d $e $f $z1 $z2 $zt
    }
}

proc IISSetCursorPosCmd {x y} {
    global current

    $current(frame) iis cursor $x $y image
}

proc IISGetCursorPosCmd {} {
    global current

    if {[$current(frame) has iis]} {
	set num [string range $current(frame) end end]
	return "[$current(frame) get iis cursor] $num"
    } else {
	# default to first frame
	return "1 1 0"
    }
}

proc IISCursorModeCmd {state} {
    global iis
    global canvas
    global current
    global cursor

    if {$state != $iis(state)} {
	if {$iis(frame) == {}} {
	    set f $current(frame)
	} else {
	    set f $iis(frame)
	}

	set iis(state) $state
	$f iis cursor mode $state

	if {$state} {
	    # set key bindings
	    bind $canvas(name) <Any-Key> [list IISCursorKey %A %x %y]
	    bind $canvas(name) <f> {}

	    $canvas(name) bind $iis(frame) <c> {}
	    $canvas(name) bind $iis(frame) <i> {}
	    $canvas(name) bind $iis(frame) <e> {}
	    $canvas(name) bind $iis(frame) <s> {}
	    $canvas(name) bind $iis(frame) <b> {}
	    $canvas(name) bind $iis(frame) <g> {}
	    $canvas(name) bind $iis(frame) <G> {}

	    $canvas(name) bind $iis(frame) <plus> {}
	    $canvas(name) bind $iis(frame) <minus> {}

	    if {$cursor(timer,abort)} {
		set cursor(timer,abort) 0
		set cursor(timer) 1
	    } else {
		set cursor(timer) 1
		CursorTimer
	    }
	} else {
	    # restore canvas key bindings
	    bind $canvas(name) <Any-Key> {}
	    bind $canvas(name) <f> {ToggleFreeze}

	    $canvas(name) bind $iis(frame) <c> \
		[list DisplayCoordDialog $iis(frame) %x %y canvas]
	    $canvas(name) bind $iis(frame) <i> \
		[list $iis(frame) marker property include 1 %x %y]
	    $canvas(name) bind $iis(frame) <e> \
		[list $iis(frame) marker property include 0 %x %y]
	    $canvas(name) bind $iis(frame) <s> \
		[list $iis(frame) marker property source 1 %x %y]
	    $canvas(name) bind $iis(frame) <b> \
		[list $iis(frame) marker property source 0 %x %y]
	    $canvas(name) bind $iis(frame) <g> GroupCreate
	    $canvas(name) bind $iis(frame) <G> GroupCreateSilent

	    $canvas(name) bind $iis(frame) <plus> [list CubeNext]
	    $canvas(name) bind $iis(frame) <minus> [list CubePrev]

	    set cursor(timer,abort) 1
	    set cursor(timer) 0
	    set iis(frame) [lindex [$canvas(name) gettags current] 0]
	}
    }
}

proc IISLoadFrame {which} {
    global iis

    if {$which > 0} {
	if {(![Frame$which has iis]) || \
		([Frame$which get iis width] != $iis(width)) || \
		([Frame$which get iis height] != $iis(height))} {
	    
	    StartLoad
	    Frame$which iis new $iis(width) $iis(height)
	    FinishLoad
	} else {
	    # make sure any previous data is rendered
	    Frame$which update now
	}
    }
}

proc IISGotoFrame {which} {
    global current

    if {$which > 0} {
	if {$current(frame) != "Frame$which"} {
	    CreateGotoFrame $which base
	}
    }
}

proc IISResetTimer {which} {
    global iis

    if {![info exists iis(timer$which)]} {
	after 500 IISTimer $which
    }
    set iis(timer$which) 1
}

proc IISTimer {which} {
    global iis

    if {$iis(timer$which)} {
	after 500 IISTimer $which
	set iis(timer$which) 0
    } else {
	if {$which > 0} {
	    Frame$which iis update
	}
	unset iis(timer$which)
    }
}

proc IISCursorKey {key x y} {
    global current
    global canvas
    global iis

    set iis(x) $x
    set iis(y) $y

    if {$key!={}} {
	set which [lindex [$canvas(name) gettags current] 0]
	if {$which != ""} {
	    set num [string range $which end end]
	    set coord [$which get coordinates canvas $x $y image]
	    if {$coord == ""} {
		switch -- $key {
		    : -
		    q {set coord "0 0"}
		    default {return}
		}
	    }
	    $which iis cursor $x $y canvas
	    iis retcur [lindex $coord 0] [lindex $coord 1] $key $num
	}
    }
}
