#  Copyright (C) 1999-2004
#  Smithsonian Astrophysical Observatory, Cambridge, MA, USA
#  For conditions of distribution and use, see copyright notice in "copyright"

package provide DS9 1.0

proc CenterCurrentFrame {} {
    global current

    CenterFrame $current(frame)
}

proc CenterAllFrame {} {
    global ds9

    foreach f $ds9(frames) {
	CenterFrame $f
    }
}

proc CenterFrame {which} {
    if {$which != ""} {
	SetWatchCursor
	$which center
	UpdateGraphXAxis
	UnsetWatchCursor

	UpdatePanZoomMenu
	UpdatePanZoomDialog
    }
}

proc Pan {x y sys sky} {
    global current

    if {$current(frame) != ""} {
	SetWatchCursor
	$current(frame) pan $sys $sky $x $y
	UpdateGraphXAxis
	UpdatePanZoomDialog
	UnsetWatchCursor
    }
}

proc PanTo {x y sys sky} {
    global current

    if {$current(frame) != ""} {
	SetWatchCursor
	$current(frame) pan to $sys $sky $x $y
	UpdateGraphXAxis
	UpdatePanZoomDialog
	UnsetWatchCursor
    }
}

proc ButtonPan {which x y} {
    global panzoom

    switch -- $panzoom(mode) {
	click {}
	drag {
	    $which pan motion begin $x $y
	}
	panzoom {}
    }
}

proc MotionPan {which x y} {
    global panzoom

    switch -- $panzoom(mode) {
	click {}
	drag {
	    $which pan motion $x $y
	}
	panzoom {}
    }
}

proc ReleasePan {which x y} {
    global panzoom
    global current

    switch -- $panzoom(mode) {
	click {
	    SetWatchCursor
	    $which pan to canvas $x $y
	}
	drag {
	    $which pan motion end $x $y

	    # we need to delay setting the cursor here, since set cursor causes
	    # an update, so the display will flash

	    SetWatchCursor
	}
	panzoom {
	    if {$panzoom(last) != " $x $y "} {
		set panzoom(state) 1
	    }
	    SetWatchCursor
	    switch -- $panzoom(state) {
		1 {
		    $which pan to canvas $x $y
		    $which update
		    set c [$which get cursor canvas]
		    $which warp to $c
		    set panzoom(last) $c

		    set z [$current(frame) get zoom]
		    if {$z < 2} {
			set panzoom(state) 2
		    } elseif {$z < 4} {
			set panzoom(state) 3
		    } elseif {$z < 8} {
			set panzoom(state) 4
		    } else {
			set panzoom(state) 5
		    }
		}
		2 {
		    $which zoom to 2 about canvas $x $y
		    set panzoom(state) 3
		}
		3 {
		    $which zoom to 4 about canvas $x $y
		    set panzoom(state) 4
		}
		4 {
		    $which zoom to 8 about canvas $x $y
		    set panzoom(state) 5
		}
		5 {
		    $which zoom to 1 about canvas $x $y
		    set panzoom(state) 2
		}
	    }
	    if {$which == $current(frame)} {
		set current(zoom) [$current(frame) get zoom]
	    }
	}
    }
    UpdateGraphXAxis
    UpdatePanZoomDialog
    UnsetWatchCursor
}

proc PreservePan {} {
    global current
    global panzoom

    if {$current(frame) != {}} {
	$current(frame) pan preserve $panzoom(preserve)
    }
}

proc ZoomToFit {} {
    global current
    global grid

    if {$current(frame) != ""} {
	SetWatchCursor

	# we need to update the grid because titles are zoom dependant
	if {$grid(view) && $grid(type) == "publication"} {
	    # recalculate to make room for labels
	    $current(frame) zoom to fit .8
	    set current(zoom) [$current(frame) get zoom]
	} else {
	    $current(frame) zoom to fit
	    set current(zoom) [$current(frame) get zoom]
	}
	GridUpdateZoom

	RefreshInfoBox $current(frame)
	UpdateGraphXAxis
	UpdatePanZoomDialog
	UnsetWatchCursor
    }
}

proc ChangeZoom {} {
    global current

    if {$current(frame) != ""} {
	SetWatchCursor
	$current(frame) zoom to $current(zoom)

	RefreshInfoBox $current(frame)
	UpdateGraphXAxis
	UpdatePanZoomDialog
	GridUpdateZoom

	UnsetWatchCursor
    }
}

proc Zoom {value} {
    global current

    if {$current(frame) != ""} {
	SetWatchCursor
	$current(frame) zoom $value
	set current(zoom) [$current(frame) get zoom]

	RefreshInfoBox $current(frame)
	UpdateGraphXAxis
	UpdatePanZoomDialog
	GridUpdateZoom

	UnsetWatchCursor
    }
}

proc ButtonZoom {which x y} {
    global current

    if {$current(frame) != ""} {
	SetWatchCursor
	$which zoom 2 about canvas $x $y
	if {$which == $current(frame)} {
	    set current(zoom) [$current(frame) get zoom]
	}

	RefreshInfoBox $current(frame)
	UpdateGraphXAxis
	UpdatePanZoomDialog
	GridUpdateZoom

	UnsetWatchCursor
    }
}

proc ShiftZoom {which} {
    global current

    if {$current(frame) != ""} {
	SetWatchCursor
	$which zoom .5
	if {$which == $current(frame)} {
	    set current(zoom) [$current(frame) get zoom]
	}

	RefreshInfoBox $current(frame)
	UpdateGraphXAxis
	UpdatePanZoomDialog
	GridUpdateZoom

	UnsetWatchCursor
    }
}

proc ChangeOrient {} {
    global current

    if {$current(frame) != ""} {
	SetWatchCursor
	$current(frame) orient $current(orient)
	UpdateGraphXAxis
	UpdatePanZoomDialog
	UnsetWatchCursor
    }
}

proc Rotate {value} {
    global current

    if {$current(frame) != ""} {
	SetWatchCursor
	$current(frame) rotate $value
	set current(rotate) [$current(frame) get rotate]

	RefreshInfoBox $current(frame)
	UpdateGraphXAxis
	UpdatePanZoomDialog
	UnsetWatchCursor
    }
}

proc ChangeRotate {} {
    global current

    if {$current(frame) != ""} {
	SetWatchCursor
	$current(frame) rotate to $current(rotate)

	RefreshInfoBox $current(frame)
	UpdateGraphXAxis
	UpdatePanZoomDialog
	UnsetWatchCursor
    }
}

proc ButtonRotate {which x y} {
    global panzoom

    $which rotate motion begin
    set panzoom(x) $x
}

proc MotionRotate {which x y} {
    global current
    global panzoom
    global cursor

    $which rotate motion [expr double($panzoom(x)-$x)/$panzoom(speed) * 180.]

    set current(rotate) [$which get rotate fixed]
    RefreshInfoBox $current(frame)
}

proc ReleaseRotate {which x y} {
    global current

    $which rotate motion end
    set current(rotate) [$which get rotate fixed]

    # we need to delay setting the cursor here, since set cursor causes
    # an update, so the display will flash

    SetWatchCursor

    RefreshInfoBox $which
    UpdateGraphXAxis
    UpdatePanZoomDialog

    UnsetWatchCursor
}

proc PanZoomDialog {} {
    global panzoom
    global menu
    global ds9

    # see if we already have a window visible
    if [winfo exist $panzoom(top)] {
	raise $panzoom(top)
	return
    }

    set w $panzoom(top)
    set title "Pan Zoom Rotate"

    # create the window
    toplevel $w -colormap $ds9(main)
    wm title $w $title
    wm iconname $w $title
    wm group $w $ds9(top)
    wm protocol $w WM_DELETE_WINDOW PanZoomDestroyDialog

    # init variables
    set panzoom(clabel) $panzoom(system)

    $w configure -menu $panzoom(mb)

    menu $panzoom(mb) -tearoff 0
    $panzoom(mb) add cascade -label File -menu $panzoom(mb).file
    $panzoom(mb) add cascade -label Edit -menu $panzoom(mb).edit
    $panzoom(mb) add cascade -label Coordinate -menu $panzoom(mb).coord
    $panzoom(mb) add cascade -label Pan -menu $panzoom(mb).pan
    $panzoom(mb) add cascade -label Zoom -menu $panzoom(mb).zoom
    $panzoom(mb) add cascade -label Orient -menu $panzoom(mb).orient
    $panzoom(mb) add cascade -label Rotate -menu $panzoom(mb).rotate

    menu $panzoom(mb).file -tearoff 0 -selectcolor $menu(selectcolor)
    $panzoom(mb).file add command -label "Apply" -command PanZoomApplyDialog
    $panzoom(mb).file add separator
    $panzoom(mb).file add command -label "Close" -command PanZoomDestroyDialog

    menu $panzoom(mb).edit -tearoff 0 -selectcolor $menu(selectcolor)
    $panzoom(mb).edit add command -label "Copy" -state disabled
    $panzoom(mb).edit add command -label "Paste" -state disabled
    $panzoom(mb).edit add command -label "Clear" -state disabled

    menu $panzoom(mb).coord -tearoff 0 -selectcolor $menu(selectcolor)
    $panzoom(mb).coord add radiobutton -label "WCS" \
	-variable panzoom(system) -value wcs -command PanZoomDialogCoord
    $panzoom(mb).coord add cascade -label "Multiple WCS" \
	-menu $panzoom(mb).coord.wcs
    $panzoom(mb).coord add separator
    $panzoom(mb).coord add radiobutton -label "Image" \
	-variable panzoom(system) -value image -command PanZoomDialogCoord
    $panzoom(mb).coord add radiobutton -label "Physical" \
	-variable panzoom(system) -value physical -command PanZoomDialogCoord
    if {$ds9(amp,det)} {
	$panzoom(mb).coord add radiobutton -label "Amplifier" \
	    -variable panzoom(system) -value amplifier \
	    -command PanZoomDialogCoord
	$panzoom(mb).coord add radiobutton -label "Detector" \
	    -variable panzoom(system) -value detector \
	    -command PanZoomDialogCoord
    }
    $panzoom(mb).coord add separator
    $panzoom(mb).coord add radiobutton -label "Equatorial B1950" \
	-variable panzoom(sky) -value fk4  -command PanZoomDialogCoord
    $panzoom(mb).coord add radiobutton -label "Equatorial J2000" \
	-variable panzoom(sky) -value fk5  -command PanZoomDialogCoord
    $panzoom(mb).coord add radiobutton -label "ICRS" \
	-variable panzoom(sky) -value icrs  -command PanZoomDialogCoord
    $panzoom(mb).coord add radiobutton -label "Galactic" \
	-variable panzoom(sky) -value galactic  -command PanZoomDialogCoord
    $panzoom(mb).coord add radiobutton -label "Ecliptic" \
	-variable panzoom(sky) -value ecliptic  -command PanZoomDialogCoord
    $panzoom(mb).coord add separator
    $panzoom(mb).coord add radiobutton -label "Degrees" \
	-variable panzoom(skyformat) -value degrees \
	-command PanZoomDialogCoord
    $panzoom(mb).coord add radiobutton -label "Sexagesimal" \
	-variable panzoom(skyformat) -value sexagesimal  \
	-command PanZoomDialogCoord

    menu $panzoom(mb).coord.wcs -tearoff $menu(tearoff) \
	-selectcolor $menu(selectcolor)
    foreach l {A B C D E F G H I J K L M N O P Q R S T U V W X Y Z} {
	$panzoom(mb).coord.wcs add radiobutton -label "WCS $l" \
	    -variable panzoom(system) -value "wcs$l" \
	    -command PanZoomDialogCoord
    }

    menu $panzoom(mb).pan -tearoff 0 -selectcolor $menu(selectcolor)
    $panzoom(mb).pan add command -label "Center Image" \
	-command CenterCurrentFrame

    menu $panzoom(mb).zoom -tearoff 0 -selectcolor $menu(selectcolor)
    $panzoom(mb).zoom add checkbutton -label "Align to WCS" \
	-variable wcs(align) -command AlignWCSFrame
    $panzoom(mb).zoom add cascade -label "Align Coord System" \
	-menu $panzoom(mb).zoom.align
    $panzoom(mb).zoom add separator
    $panzoom(mb).zoom add command -label "Zoom In" -command "Zoom 2"
    $panzoom(mb).zoom add command -label "Zoom Out" -command "Zoom .5"
    $panzoom(mb).zoom add separator
    $panzoom(mb).zoom add command -label "Zoom to Fit Frame" \
	-command "ZoomToFit"
    $panzoom(mb).zoom add separator
    $panzoom(mb).zoom add radiobutton -label "1/16" \
	-variable current(zoom) -value 0.0625 -command ChangeZoom
    $panzoom(mb).zoom add radiobutton -label "1/8" \
	-variable current(zoom) -value 0.125 -command ChangeZoom
    $panzoom(mb).zoom add radiobutton -label "1/4" \
	-variable current(zoom) -value 0.25 -command ChangeZoom
    $panzoom(mb).zoom add radiobutton -label "1/2" \
	-variable current(zoom) -value 0.5 -command ChangeZoom
    $panzoom(mb).zoom add radiobutton -label "1" \
	-variable current(zoom) -value 1 -command ChangeZoom
    $panzoom(mb).zoom add radiobutton -label "2" \
	-variable current(zoom) -value 2 -command ChangeZoom
    $panzoom(mb).zoom add radiobutton -label "4" \
	-variable current(zoom) -value 4 -command ChangeZoom
    $panzoom(mb).zoom add radiobutton -label "8" \
	-variable current(zoom) -value 8 -command ChangeZoom
    $panzoom(mb).zoom add radiobutton -label "16" \
	-variable current(zoom) -value 16 -command ChangeZoom

    menu $panzoom(mb).zoom.align -tearoff $menu(tearoff) \
	-selectcolor $menu(selectcolor) 
    $panzoom(mb).zoom.align add radiobutton -label "WCS" \
	-variable wcs(align,system) -value wcs -command AlignWCSFrame
    $panzoom(mb).zoom.align add cascade -label "Multiple WCS" \
	-menu $panzoom(mb).zoom.align.wcs
    $panzoom(mb).zoom.align add separator
    $panzoom(mb).zoom.align add radiobutton -label "Equatorial B1950" \
	-variable wcs(align,sky -value fk4) -command AlignWCSFrame
    $panzoom(mb).zoom.align add radiobutton -label "Equatorial J2000" \
	-variable wcs(align,sky) -value fk5 -command AlignWCSFrame
    $panzoom(mb).zoom.align add radiobutton -label "ICRS" \
	-variable wcs(align,sky) -value icrs -command AlignWCSFrame
    $panzoom(mb).zoom.align add radiobutton -label "Galactic" \
	-variable wcs(align,sky) -value galactic -command AlignWCSFrame
    $panzoom(mb).zoom.align add radiobutton -label "Ecliptic" \
	-variable wcs(align,sky) -value ecliptic -command AlignWCSFrame

    menu $panzoom(mb).zoom.align.wcs -tearoff $menu(tearoff) \
	-selectcolor $menu(selectcolor)
    foreach l {A B C D E F G H I J K L M N O P Q R S T U V W X Y Z} {
	$panzoom(mb).zoom.align.wcs add radiobutton -label "WCS $l" \
	    -variable wcs(align,system) -value "wcs$l" -command AlignWCSFrame
    }

    menu $panzoom(mb).orient -tearoff 0 -selectcolor $menu(selectcolor)
    $panzoom(mb).orient add radiobutton  -label "None" \
	-variable current(orient) -value none -command ChangeOrient
    $panzoom(mb).orient add radiobutton  -label "Invert X" \
	-variable current(orient) -value x -command ChangeOrient
    $panzoom(mb).orient add radiobutton -label "Invert Y" \
	-variable current(orient) -value y -command ChangeOrient
    $panzoom(mb).orient add radiobutton -label "Invert X&Y" \
	-variable current(orient) -value xy -command ChangeOrient

    menu $panzoom(mb).rotate -tearoff 0 -selectcolor $menu(selectcolor)
    $panzoom(mb).rotate add radiobutton -label "0 deg" \
	-variable current(rotate) -value 0 -command ChangeRotate
    $panzoom(mb).rotate add radiobutton -label "90 deg" \
	-variable current(rotate) -value 90 -command ChangeRotate
    $panzoom(mb).rotate add radiobutton -label "180 deg" \
	-variable current(rotate) -value 180 -command ChangeRotate
    $panzoom(mb).rotate add radiobutton -label "270 deg" \
	-variable current(rotate) -value 270 -command ChangeRotate

    frame $w.param -relief groove -borderwidth 2
    frame $w.buttons -relief groove -borderwidth 2
    pack $w.param $w.buttons -fill x -ipadx 4 -ipady 4

    label $w.param.zoomtitle -text "Zoom"
    entry $w.param.zoomvalue -textvariable panzoom(zoom) -width 14

    label $w.param.rottitle -text "Rotate"
    entry $w.param.rotvalue -textvariable panzoom(rotate) -width 14
    label $w.param.rottitle2 -text "degrees"

    label $w.param.pantitle -text "Pan"
    set panzoom(xname) [label $w.param.panxtitle -text "" -width 1]
    entry $w.param.panx -textvariable panzoom(x) -width 14
    set panzoom(yname) [label $w.param.panytitle -text "" -width 1]
    entry $w.param.pany -textvariable panzoom(y) -width 14
    label $w.param.pansystem -textvariable panzoom(clabel) \
	-width 10 -relief groove
    
    grid rowconfigure $w.param 0 -pad 4
    grid rowconfigure $w.param 1 -pad 4
    grid rowconfigure $w.param 2 -pad 4

    grid $w.param.zoomtitle -row 0 -column 0 -columnspan 2 -padx 4 -sticky w
    grid $w.param.zoomvalue -row 0 -column 2 -padx 4 -sticky w

    grid $w.param.rottitle -row 1 -column 0 -columnspan 2 -padx 4 -sticky w
    grid $w.param.rotvalue -row 1 -column 2 -padx 4 -sticky w
    grid $w.param.rottitle2 -row 1 -column 3 -columnspan 2 -padx 4 -sticky w

    grid $w.param.pantitle -row 2 -column 0 -padx 4 -sticky w
    grid $panzoom(xname) -row 2 -column 1 -padx 4 -sticky w
    grid $w.param.panx -row 2 -column 2 -padx 4 -sticky w
    grid $panzoom(yname) -row 2 -column 3 -padx 4 -sticky w
    grid $w.param.pany -row 2 -column 4 -padx 4 -sticky w
    grid $w.param.pansystem -row 2 -column 5 -padx 4 -sticky w

    button $w.buttons.apply -text "Apply" -command PanZoomApplyDialog
    button $w.buttons.close -text "Close" -command PanZoomDestroyDialog
    pack $w.buttons.apply $w.buttons.close -side left -padx 10 -expand true

    $w.param.zoomvalue select range 0 end

    UpdatePanZoomDialog
}

proc PanZoomApplyDialog {} {
    global current
    global panzoom

    if {$current(frame) != ""} {
	set current(zoom) $panzoom(zoom)
	set current(rotate) $panzoom(rotate)

	SetWatchCursor
	$current(frame) zoom to $current(zoom) about \
	    $panzoom(system) $panzoom(sky) $panzoom(x) $panzoom(y) 
	$current(frame) rotate to $current(rotate)

	RefreshInfoBox $current(frame)
	UpdateGraphXAxis
	GridUpdateZoom
	UnsetWatchCursor
    }
}

proc PanZoomDestroyDialog {} {
    global panzoom

    destroy $panzoom(top)
    destroy $panzoom(mb)

    unset panzoom(x)
    unset panzoom(y)
    unset panzoom(xname)
    unset panzoom(yname)
    unset panzoom(zoom)
    unset panzoom(rotate)
    unset panzoom(clabel)
}

proc PanZoomDialogCoord {} {
    global panzoom
    global current

    UpdateCoordMenu $current(frame) $panzoom(mb).coord
    AdjustCoord $current(frame) panzoom(system)

    if {$current(frame) != ""} {
	set coord [$current(frame) get cursor $panzoom(system) $panzoom(sky) \
		       $panzoom(skyformat)]
	set panzoom(x) [lindex $coord 0]
	set panzoom(y) [lindex $coord 1]
    } else {
	set panzoom(x) {}
	set panzoom(y) {}
    }

    $panzoom(xname) configure -text "X" -font {helvetica 9}
    $panzoom(yname) configure -text "Y" -font {helvetica 9}
    switch -- $panzoom(system) {
	image -
	physical -
	amplifier -
	detector {}
	default {
	    if [$current(frame) has wcs equatorial $panzoom(system)] {
		switch -- $panzoom(sky) {
		    fk4 -
		    fk5 -
		    icrs {
			$panzoom(xname) configure -text "a" -font {symbol 12}
			$panzoom(yname) configure -text "d" -font {symbol 12}
		    }
		    galactic {
			$panzoom(xname) configure -text "l" \
			    -font {-family times -slant italic}
			$panzoom(yname) configure -text "b" \
			    -font {-family times -slant italic}
		    }
		    ecliptic {
			$panzoom(xname) configure -text "l" -font {symbol 12}
			$panzoom(yname) configure -text "b" -font {symbol 12}
		    }
		}
	    }
	}
    }

    set panzoom(clabel) $panzoom(system)
    switch -- $panzoom(system) {
	image -
	physical -
	amplifier -
	detector {}
	default {
	    if [$current(frame) has wcs $panzoom(system)] {
		if [$current(frame) has wcs equatorial $panzoom(system)] {
		    set panzoom(clabel) $panzoom(sky)
		} else {
		    set name [$current(frame) get wcs name $panzoom(system)]
		    if {$name != ""} {
			set panzoom(clabel) $name
		    }
		}
	    }
	}
    }
}

proc UpdatePanZoomDialog {} {
    global panzoom
    global current

    if {$current(frame) != ""} {
	global debug
	if {$debug(tcl,update)} {
	    puts "UpdatePanZoomDialog"
	}

	if {[winfo exist $panzoom(top)]} {
	    if {$current(frame) != ""} {
		set panzoom(zoom) [$current(frame) get zoom]
		set panzoom(rotate) [$current(frame) get rotate]
	    } else {
		set panzoom(zoom) {}
		set panzoom(rotate) {}
	    }

	    PanZoomDialogCoord
	    UpdateAlignWCSMenu $panzoom(mb).zoom
	}
    }
}

proc UpdatePanZoomMenu {} {
    global current
    global wcs
    global buttons
    global ds9
    global menu
    global panzoom
    
    global debug
    if {$debug(tcl,update)} {
	puts "UpdatePanZoomMenu"
    }

    UpdateAlignWCSMenu $ds9(mb).zoom

    if {$current(frame) != ""} {
	$ds9(mb) entryconfig $menu(zoom) -state normal
	$buttons(name).major.zoom configure -state normal

	set panzoom(preserve) [$current(frame) get pan preserve]

	if {[$current(frame) has fits]} {
	    $ds9(mb).zoom entryconfig "Zoom to Fit Frame" -state normal
	} else {
	    $ds9(mb).zoom entryconfig "Zoom to Fit Frame" -state disabled
	}

	set current(zoom) [$current(frame) get zoom]
	set current(rotate) [$current(frame) get rotate]
	set current(orient) [$current(frame) get orient]
	set align [$current(frame) get wcs align]

	set wcs(align) [lindex $align 0]
	set wcs(align,system) [lindex $align 1]
	set wcs(align,sky) [lindex $align 2]
    } else {
	$ds9(mb) entryconfig $menu(zoom) -state disabled
	$buttons(name).major.zoom configure -state disabled

	set panzoom(preserve) 0

	set current(zoom) 1
	set current(rotate) 0
	set current(orient) none

	set wcs(align) 0
	set wcs(align,system) wcs
	set wcs(align,sky) fk5
    }
}

proc ProcessPanCmd {varname iname} {
    upvar $varname var
    upvar $iname i

    switch -- [string tolower [lindex $var $i]] {
	to {
	    set sys [lindex $var [expr $i+3]]
	    set sky [lindex $var [expr $i+4]]
	    switch -- $sys {
		fk4 -
		fk5 -
		icrs -
		galactic -
		ecliptic {
		    PanTo [lindex $var [expr $i+1]] [lindex $var [expr $i+2]] \
			wcs $sys
		    incr i 3
		}
		wcs -
		wcsa -
		wcsb -
		wcsc -
		wcsd -
		wcse -
		wcsf -
		wcsg -
		wcsh -
		wcsi -
		wcsj -
		wcsk -
		wcsl -
		wcsm -
		wcsn -
		wcso -
		wcsp -
		wcsq -
		wcsr -
		wcss -
		wcst -
		wcsu -
		wcsv -
		wcsw -
		wcsx -
		wcsy -
		wcsz {
		    PanTo [lindex $var [expr $i+1]] [lindex $var [expr $i+2]] \
			$sys $sky
		    incr i 4
		}
		image -
		physical -
		amplifier -
		detector {
		    PanTo [lindex $var [expr $i+1]] [lindex $var [expr $i+2]] \
			$sys {}
		    incr i 3
		}
		{} -
		default {
		    PanTo [lindex $var [expr $i+1]] [lindex $var [expr $i+2]] \
			physical {}
		    incr i 2
		}
	    }
	}
	default {
	    set sys [lindex $var [expr $i+2]]
	    set sky [lindex $var [expr $i+3]]
	    switch -- $sys {
		fk4 -
		fk5 -
		icrs -
		galactic -
		ecliptic {
		    Pan [lindex $var $i] [lindex $var [expr $i+1]] \
			wcs $sys
		    incr i 2
		}
		wcs -
		wcsa -
		wcsb -
		wcsc -
		wcsd -
		wcse -
		wcsf -
		wcsg -
		wcsh -
		wcsi -
		wcsj -
		wcsk -
		wcsl -
		wcsm -
		wcsn -
		wcso -
		wcsp -
		wcsq -
		wcsr -
		wcss -
		wcst -
		wcsu -
		wcsv -
		wcsw -
		wcsx -
		wcsy -
		wcsz {
		    Pan [lindex $var $i] [lindex $var [expr $i+1]] \
			$sys $sky
		    incr i 3
		}
		image -
		physical -
		amplifier -
		detector {
		    Pan [lindex $var $i] [lindex $var [expr $i+1]] \
			$sys {}
		    incr i 2
		}
		{} -
		default {
		    Pan [lindex $var $i] [lindex $var [expr $i+1]] \
			physical {}
		    incr i 1
		}

	    }
	}
    }
}

proc ProcessZoomCmd {varname iname} {
    upvar $varname var
    upvar $iname i

    global current

    switch -- [string tolower [lindex $var $i]] {
	to {
	    switch -- [string tolower [lindex $var [expr $i+1]]] {
		fit {ZoomToFit}
		default {
		    set current(zoom) [lindex $var [expr $i+1]]
		    ChangeZoom
		}
	    }
	    incr i 1
	}
	default {Zoom [lindex $var $i]}
    }
}

proc ProcessOrientCmd {varname iname} {
    upvar $varname var
    upvar $iname i

    global current
    set current(orient) [string tolower [lindex $var $i]]
    ChangeOrient
}

proc ProcessRotateCmd {varname iname} {
    upvar $varname var
    upvar $iname i

    global current
    switch -- [string tolower [lindex $var $i]] {
	to {
	    set current(rotate) [lindex $var [expr $i+1]]
	    ChangeRotate
	    incr i
	}
	default {Rotate [lindex $var $i]}
    }
}
