// Copyright (C) 1999-2004
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include <stdlib.h>
#include <limits.h>
#include <string.h>

#include "colorbartrue16.h"
#include "util.h"

// Tk Canvas Widget Function Declarations

int ColorbarTrueColor16CreateProc(Tcl_Interp*, Tk_Canvas, Tk_Item*, int, 
				  Tcl_Obj *const []);

// Colorbar Specs

static Tk_CustomOption tagsOption = {
  Tk_CanvasTagsParseProc, Tk_CanvasTagsPrintProc, NULL
};

struct ColorbarTrueColor16Options {
  Tk_Item item;              // required by tk
  int x, y;                  // Coordinates of positioning point on canvas
  int width;                 // widget width
  int height;                // widget height
  Tk_Anchor anchor;          // Where to anchor widget relative to x,y
  char* cmdName;             // Suggested Tcl command name

  Widget* widget;            // pointer to widget class

  int colors;
};

static Tk_ConfigSpec colorbarTrueColor16Specs[] = {

  {TK_CONFIG_STRING, "-command", NULL, NULL, "colorbar",
   Tk_Offset(ColorbarTrueColor16Options, cmdName), TK_CONFIG_OPTION_SPECIFIED,
   NULL},
  {TK_CONFIG_INT, "-x", NULL, NULL, "1",
   Tk_Offset(ColorbarTrueColor16Options, x), TK_CONFIG_OPTION_SPECIFIED, NULL},
  {TK_CONFIG_INT, "-y", NULL, NULL, "1",
   Tk_Offset(ColorbarTrueColor16Options, y), TK_CONFIG_OPTION_SPECIFIED, NULL},
  {TK_CONFIG_INT, "-width", NULL, NULL, "512",
   Tk_Offset(ColorbarTrueColor16Options, width), TK_CONFIG_OPTION_SPECIFIED, 
   NULL},
  {TK_CONFIG_INT, "-height", NULL, NULL, "22",
   Tk_Offset(ColorbarTrueColor16Options, height), TK_CONFIG_OPTION_SPECIFIED, 
   NULL},
  {TK_CONFIG_ANCHOR, "-anchor", NULL, NULL, "nw",
   Tk_Offset(ColorbarTrueColor16Options, anchor), 0, NULL},
  {TK_CONFIG_CUSTOM, "-tags", NULL, NULL, NULL,
   0, TK_CONFIG_NULL_OK, &tagsOption},

  {TK_CONFIG_INT, "-colors", NULL, NULL, "1024",
   Tk_Offset(ColorbarTrueColor16Options, colors), 0, NULL},

  {TK_CONFIG_END, NULL, NULL, NULL, NULL, 0, 0, NULL},
};

// Tk Static Structure

static Tk_ItemType colorbarTrueColor16Type = {
  "colorbartruecolor16",        // name
  sizeof(ColorbarTrueColor16Options), // size
  ColorbarTrueColor16CreateProc, // configProc
  colorbarTrueColor16Specs,     // configSpecs
  WidgetConfigProc,             // configProc
  WidgetCoordProc,              // coordProc
  WidgetDeleteProc,             // deleteProc
  WidgetDisplayProc,            // displayProc
  0,                            // alwaysRedraw
  WidgetPointProc,              // pointProc
  WidgetAreaProc,               // areaProc
  WidgetPostscriptProc,         // postscriptProc
  WidgetScaleProc,              // scaleProc
  WidgetTranslateProc,          // translateProc
  (Tk_ItemIndexProc*)NULL,      // indexProc
  (Tk_ItemCursorProc*)NULL,     // icursorProc
  (Tk_ItemSelectionProc*)NULL,  // selectionProc
  (Tk_ItemInsertProc*)NULL,     // insertProc
  (Tk_ItemDCharsProc*)NULL,     // dCharsProc
  (Tk_ItemType*)NULL            // nextPtr
};

// Non-Member Functions

int ColorbarTrueColor16_Init(Tcl_Interp* interp)
{
  Tk_CreateItemType(&colorbarTrueColor16Type);
  return TCL_OK;
}

int ColorbarTrueColor16CreateProc(Tcl_Interp* interp, Tk_Canvas canvas, 
				  Tk_Item* item, int argc, 
				  Tcl_Obj *const argv[])
{
  ColorbarTrueColor16* colorbar = 
    new ColorbarTrueColor16(interp, canvas, item);

  // and set default configuration

  if (colorbar->configure(argc, (const char**)argv, 0) != TCL_OK) {
    delete colorbar;
    Tcl_AppendResult(interp, " error occured while creating colorbar.", NULL);
    return TCL_ERROR;
  }

  return TCL_OK;
}

// ColorbarTrueColor16

ColorbarTrueColor16::ColorbarTrueColor16(Tcl_Interp* i, Tk_Canvas c, 
					 Tk_Item* item) 
  : ColorbarTrueColor(i, c, item)
{
  configSpecs = colorbarTrueColor16Specs;  // colorbar configure options

  loadDefaultCMaps();
}

int ColorbarTrueColor16::initColormap()
{
  colorCount = (((ColorbarTrueColor16Options*)options)->colors);
  return initColormapForReal();
}

void ColorbarTrueColor16::updateColors()
{
  updateColorCells();

  // fill in xmap
  // make sure we have a pixmap

  if (!pixmap || !xmap)
    return;

  int& width = options->width;
  int& height = options->height;
  char* data = xmap->data;
    
  // Fill in colorbar data
  // Calculate first row
  // check for bits/pixel

  int rs,gs,bs;
#ifndef _WIN32
  unsigned short rm = decodeMask((unsigned short)visual->red_mask, &rs);
  unsigned short gm = decodeMask((unsigned short)visual->green_mask, &gs);
  unsigned short bm = decodeMask((unsigned short)visual->blue_mask, &bs);
#else
  // windows masks are plain wrong
  unsigned short rm = decodeMask((unsigned short)0x7C00, &rs);
  unsigned short gm = decodeMask((unsigned short)0x03E0, &gs);
  unsigned short bm = decodeMask((unsigned short)0x001F, &bs);
#endif

  // if we have cross platforms, we need to byte swap

  if ((!xmap->byte_order && lsb()) || (xmap->byte_order && !lsb())) {
    for (int i=0; i<width; i++) {
      int j = ((int)(double(i)/width*colorCount))*3;
      unsigned short r = colorCells[j+2];
      unsigned short g = colorCells[j+1];
      unsigned short b = colorCells[j];
      unsigned short a = 0;
      a |= rs>0 ? ((r & rm) << rs) : ((r & rm) >> -rs);
      a |= gs>0 ? ((g & gm) << gs) : ((g & gm) >> -gs);
      a |= bs>0 ? ((b & bm) << bs) : ((b & bm) >> -bs);

      memcpy(data+i*2, &a, 2);
    }
  }
  else {
    for (int i=0; i<width; i++) {
      int j = ((int)(double(i)/width*colorCount))*3;
      unsigned short r = colorCells[j+2];
      unsigned short g = colorCells[j+1];
      unsigned short b = colorCells[j];
      unsigned short a = 0;
      a |= rs>0 ? ((r & rm) << rs) : ((r & rm) >> -rs);
      a |= gs>0 ? ((g & gm) << gs) : ((g & gm) >> -gs);
      a |= bs>0 ? ((b & bm) << bs) : ((b & bm) >> -bs);

      unsigned char* rr = (unsigned char*)(&a);
      *(data+i*2) = *(rr+1);
      *(data+i*2+1) = *(rr);
    }
  }

  // --and duplicate for remaining rows

  for (int j=1; j<height; j++)
    memcpy(data+(j*xmap->bytes_per_line), data, xmap->bytes_per_line);

  // update the pixmap from map image and clean up

  XPutImage(display, pixmap, gc, xmap, 0, 0, 0, 0, width, height);

  redraw();
}

