// Copyright (C) 1999-2004
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <limits.h>
#include <float.h>

#if __GNUC__ >= 3
#include <iostream>
#include <sstream>
#include <iomanip>
using namespace std;
#else
#include <iostream.h>
#include <strstream.h>
#include <iomanip.h>
#endif

#include "column.h"
#include "head.h"
#include "util.h"

FitsColumn::FitsColumn(FitsHead* head, int i, int off)
{
  index_ = i;
  width_ = 0;
  offset_ = off;
  type_ = ' ';

  tform_ = head->getString(keycat("TFORM",i));
  ttype_ = head->getString(keycat("TTYPE",i));
  tunit_ = head->getString(keycat("TUNIT",i));
  tscal_ = head->getReal(keycat("TSCAL",i), 1);
  tzero_ = head->getReal(keycat("TZERO",i), 0);
  hastnull_ = head->find(keycat("TNULL",i)) ? 1:0;
  tnull_ = head->getInteger(keycat("TNULL",i), 0);

  char* td = head->find(keycat("TDMAX",i));
  char* tl = head->find(keycat("TLMAX",i));
  char* ta = head->find(keycat("TALEN",i));
  char* ax = head->find(keycat("AXLEN",i));

  // this provides backward compatibility
  if (td) {
    hastlmin_ = head->find(keycat("TDMIN",i)) ? 1:0;
    hastlmax_ = 1;
    tlmin_ = head->getReal(keycat("TDMIN",i), 0);
    tlmax_ = head->getReal(keycat("TDMAX",i), 0);
  }
  else if (tl) {
    hastlmin_ = head->find(keycat("TLMIN",i)) ? 1:0;
    hastlmax_ = 1;
    tlmin_ = head->getReal(keycat("TLMIN",i), 0);
    tlmax_ = head->getReal(keycat("TLMAX",i), 0);
  }
  else if (ta) {
    hastlmin_ = 0;
    hastlmax_ = 1;
    tlmin_ = 1;
    tlmax_ = head->getReal(keycat("TALEN",i), 0);
  }
  else if (ax) {
    hastlmin_ = 0;
    hastlmax_ = 1;
    tlmin_ = 1;
    tlmax_ = head->getReal(keycat("AXLEN",i), 0);
  }
  else {
    hastlmin_ = 0;
    hastlmax_ = 0;
    tlmin_ = 0;
    tlmax_ = 0;
  }

  // now, make sure they are valid
  if (tlmin_>tlmax_) {
    hastlmin_ = 0;
    hastlmax_ = 0;
    tlmin_ = 0;
    tlmax_ = 0;
  }

  // use tlmin/tlmax if available
  if (hastlmin_ || hastlmax_) {
    min_ = tlmin_;
    max_ = tlmax_;
  }
  else {
    min_ = -DBL_MAX;
    max_ = DBL_MAX;
  }
}

FitsColumn::~FitsColumn()
{
  if (tform_)
    delete [] tform_;
  if (tunit_)
    delete [] tunit_;
  if (ttype_)
    delete [] ttype_;
}

#if __GNUC__ >= 3
char* FitsColumn::keycat(const char* name, int i)
{
  ostringstream str;
  str << name << i << ends;
  memcpy(keybuf,str.str().c_str(),str.str().length());
  return keybuf;
}
#else
char* FitsColumn::keycat(const char* name, int i)
{
  ostrstream str(keybuf, 9);
  str << name << i << ends;
  return keybuf;
}
#endif

#if __GNUC__ >= 3
FitsAsciiColumn::FitsAsciiColumn(FitsHead* head, int i, int offset)
  : FitsColumn(head, i, offset)
{
  prec_ = 0;
  int tbcol = head->getInteger(keycat("TBCOL",i),0);
  if (tbcol)
    offset_ = tbcol-1;

  // parse TFORM
  if (tform_) {
    string x(tform_);
    istringstream str(x);
    str >> type_ >> width_;
    if (type_ == 'F' || type_ == 'E' || type_ == 'D') {
      char s;
      str >> s >> prec_;
    }
  }
}
#else
FitsAsciiColumn::FitsAsciiColumn(FitsHead* head, int i, int offset)
  : FitsColumn(head, i, offset)
{
  prec_ = 0;
  int tbcol = head->getInteger(keycat("TBCOL",i),0);
  if (tbcol)
    offset_ = tbcol-1;

  // parse TFORM
  if (tform_) {
    istrstream str(tform_);
    str >> type_ >> width_;
    if (type_ == 'F' || type_ == 'E' || type_ == 'D') {
      char s;
      str >> s >> prec_;
    }
  }
}
#endif

#if __GNUC__ >= 3
double FitsAsciiColumn::value(const char* ptr, int i)
{
  string x(ptr+offset_);
  istringstream str(x);
  double r;
  str >> r;

  return r;
}
#else
double FitsAsciiColumn::value(const char* ptr, int i)
{
  istrstream str(ptr+offset_);
  double r;
  str >> r;

  return r;
}
#endif

char* FitsAsciiColumn::str(const char* ptr, int i)
{
  strncpy(buf_, ptr+offset_, width_);
  buf_[width_] = '\0';
  return buf_;
}

Vector FitsAsciiColumn::dimension()
{
  if (hastlmin_ || hastlmax_)
    return Vector(tlmin_,tlmax_);
  else
    switch (type_) {
    case 'A':
      return Vector(0,UCHAR_MAX);
      break;
    case 'I':
      return Vector(INT_MIN,INT_MAX);
      break;
    case 'F':
    case 'E':
      return Vector(FLT_MIN,FLT_MAX);
      break;
    case 'D':
      return Vector(DBL_MIN,DBL_MAX);
      break;
    }
}

FitsBinColumn::FitsBinColumn(FitsHead* head, int i, int offset)
  : FitsColumn(head, i, offset)
{
  repeat_ = 1;
  tdisp_ = head->getString(keycat("TDISP",i));
}

FitsBinColumn::~FitsBinColumn()
{
  if (tdisp_)
    delete [] tdisp_;
}

#if __GNUC__ >= 3
FitsBinColumnA::FitsBinColumnA(FitsHead* head, int i, int offset)
  : FitsBinColumn(head, i, offset)
{
  Awidth_ = 0;
  Aterm_ = 0;

  // parse TFORM
  if (tform_) {
    string x(tform_);
    istringstream str(x);
    if (isalpha(tform_[0]))
      str >> type_;
    else
      str >> repeat_ >> type_;

    char s;
    str >> s >> s >> s >> s >> s >> Awidth_ >> s >> oct >> Aterm_;
  }

  // and determine field width
  width_ = repeat_ * (Awidth_+1);
}
#else
FitsBinColumnA::FitsBinColumnA(FitsHead* head, int i, int offset)
  : FitsBinColumn(head, i, offset)
{
  Awidth_ = 0;
  Aterm_ = 0;

  // parse TFORM
  if (tform_) {
    istrstream str(tform_);
    if (isalpha(tform_[0]))
      str >> type_;
    else
      str >> repeat_ >> type_;

    char s;
    str >> s >> s >> s >> s >> s >> Awidth_ >> s >> oct >> Aterm_;
  }

  // and determine field width
  width_ = repeat_ * (Awidth_+1);
}
#endif

#if __GNUC__ >= 3
double FitsBinColumnA::value(const char* ptr, int i)
{
  double r;
  string x(ptr+offset_);
  istringstream str(x);
  str >> r;

  return r;
}
#else
double FitsBinColumnA::value(const char* ptr, int i)
{
  double r;
  istrstream str(ptr+offset_, width_);
  str >> r;

  return r;
}
#endif

char* FitsBinColumnA::str(const char* ptr, int i)
{
  strncpy(buf_, ptr+offset_, width_);
  buf_[width_] = '\0';
  return buf_;
}

Vector FitsBinColumnA::dimension()
{
  return (hastlmin_ || hastlmax_) ? Vector(tlmin_,tlmax_) : 
    Vector(-DBL_MAX,DBL_MAX);
}

#if __GNUC__ >= 3
FitsBinColumnB::FitsBinColumnB(FitsHead* head, int i, int offset)
  : FitsBinColumn(head, i, offset)
{
  byteswap = lsb();

  // parse TFORM
  if (tform_) {
    string x(tform_);
    istringstream str(x);
    if (isalpha(tform_[0]))
      str >> type_;
    else
      str >> repeat_ >> type_;
  }
}
#else
FitsBinColumnB::FitsBinColumnB(FitsHead* head, int i, int offset)
  : FitsBinColumn(head, i, offset)
{
  byteswap = lsb();

  // parse TFORM
  if (tform_) {
    istrstream str(tform_);
    if (isalpha(tform_[0]))
      str >> type_;
    else
      str >> repeat_ >> type_;
  }
}
#endif

FitsBinColumnBit::FitsBinColumnBit(FitsHead* head, int i, int off)
  : FitsBinColumnB(head, i, off)
{
  // and determine field width
  width_ = (repeat_+7)/8;
}

double FitsBinColumnBit::value(const char* ptr, int i)
{
  // not implemented
  return 0;
}

char* FitsBinColumnBit::str(const char* ptr, int i)
{
  // not implemented
  return NULL;
}

Vector FitsBinColumnBit::dimension()
{
  return Vector(0,repeat_);
}

template<class T> FitsBinColumnT<T>::FitsBinColumnT(FitsHead* head,
						    int i, int off)
  : FitsBinColumnB(head, i, off)
{
  // and determine field width
  width_ = repeat_ * size();
}

double FitsBinColumnT<unsigned char>::value(const char* ptr, int i)
{
  return (unsigned char)(*(ptr+offset_+i));
}

double FitsBinColumnT<short>::value(const char* ptr, int i)
{
  const char* p = ptr+offset_+i*2;
  union {
    char c[2];
    short s;
  } u;

  if (byteswap) {
    u.c[1] = *p++;
    u.c[0] = *p;
  }
  else {
    u.c[0] = *p++;
    u.c[1] = *p;
  }

  return u.s;
}

double FitsBinColumnT<unsigned short>::value(const char* ptr, int i)
{
  const char* p = ptr+offset_+i*2;
  union {
    char c[2];
    unsigned short s;
  } u;

  if (byteswap) {
    u.c[1] = *p++;
    u.c[0] = *p;
  }
  else {
    u.c[0] = *p++;
    u.c[1] = *p;
  }

  return u.s;
}

double FitsBinColumnT<int>::value(const char* ptr, int i)
{
  const char* p = ptr+offset_+i*4;
  union {
    char c[4];
    int i;
  } u;

  if (byteswap) {
    u.c[3] = *p++;
    u.c[2] = *p++;
    u.c[1] = *p++;
    u.c[0] = *p;
  }
  else
    memcpy(u.c,p,4);

  return u.i;
}

double FitsBinColumnT<unsigned int>::value(const char* ptr, int i)
{
  const char* p = ptr+offset_+i*4;
  union {
    char c[4];
    unsigned int i;
  } u;

  if (byteswap) {
    u.c[3] = *p++;
    u.c[2] = *p++;
    u.c[1] = *p++;
    u.c[0] = *p;
  }
  else
    memcpy(u.c,p,4);

  return u.i;
}

double FitsBinColumnT<float>::value(const char* ptr, int i)
{
  const char* p = ptr+offset_+i*4;
  union {
    char c[4];
    float f;
  } u;

  if (byteswap) {
    u.c[3] = *p++;
    u.c[2] = *p++;
    u.c[1] = *p++;
    u.c[0] = *p;
  }
  else
    memcpy(u.c,p,4);

  return u.f;
}

double FitsBinColumnT<double>::value(const char* ptr, int i)
{
  const char* p = ptr+offset_+i*8;
  union {
    char c[8];
    double d;
  } u;

  if (byteswap) {
    u.c[7] = *p++;
    u.c[6] = *p++;
    u.c[5] = *p++;
    u.c[4] = *p++;
    u.c[3] = *p++;
    u.c[2] = *p++;
    u.c[1] = *p++;
    u.c[0] = *p;
  }
  else
    memcpy(u.c,p,8);

  return u.d;
}

#if __GNUC__ >= 3
template<class T> char* FitsBinColumnT<T>::str(const char* ptr, int i)
{
  ostringstream ost;
  ost << value(ptr,i) << ends;
  return (char*)ost.str().c_str();
}
#else
template<class T> char* FitsBinColumnT<T>::str(const char* ptr, int i)
{
  ostrstream ost(buf_,64);
  ost << value(ptr,i) << ends;
  return buf_;
}
#endif

Vector FitsBinColumnT<unsigned char>::dimension()
{
  return (hastlmin_ || hastlmax_) ? Vector(tlmin_-.5,tlmax_+.5) 
    : Vector(0,UCHAR_MAX);
}

Vector FitsBinColumnT<short>::dimension()
{
  return (hastlmin_ || hastlmax_) ? Vector(tlmin_-.5,tlmax_+.5) 
    : Vector(SHRT_MIN,SHRT_MAX);
}

Vector FitsBinColumnT<unsigned short>::dimension()
{
  return (hastlmin_ || hastlmax_) ? Vector(tlmin_-.5,tlmax_+.5) 
    : Vector(0,USHRT_MAX);
}

Vector FitsBinColumnT<int>::dimension()
{
  return (hastlmin_ || hastlmax_) ? Vector(tlmin_-.5,tlmax_+.5) 
    : Vector(INT_MIN,INT_MAX);
}

Vector FitsBinColumnT<unsigned int>::dimension()
{
  return (hastlmin_ || hastlmax_) ? Vector(tlmin_-.5,tlmax_+.5) 
    : Vector(0,UINT_MAX);
}

Vector FitsBinColumnT<float>::dimension()
{
  return (hastlmin_ || hastlmax_) ? Vector(tlmin_,tlmax_) 
    : Vector(-FLT_MAX,FLT_MAX);
}

Vector FitsBinColumnT<double>::dimension()
{
  return (hastlmin_ || hastlmax_) ? Vector(tlmin_,tlmax_) 
    : Vector(-DBL_MAX,DBL_MAX);
}

template class FitsBinColumnT<unsigned char>;
template class FitsBinColumnT<short>;
template class FitsBinColumnT<unsigned short>;
template class FitsBinColumnT<int>;
template class FitsBinColumnT<unsigned int>;
template class FitsBinColumnT<float>;
template class FitsBinColumnT<double>;
