
#ifndef IIRBASE_HH
#define IIRBASE_HH

// Copyright (c) 1996-1999 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.


// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	phil.wilsey@uc.edu
//          Dale E. Martin	dmartin@ece.uc.edu

//---------------------------------------------------------------------------

#include "IRBasicDataTypes.hh"

class IIR_Identifier;
class IIR_DesignFile;

/** This is the base node definition for the intermediate form.  All nodes
    in IIR are descendants of this node.  */

class IIRBase {
public:

  /** Returns the IIR_Kind of this node.  This is useful
      in determining the type of a node when an IIR * is
      all that's available */
  virtual IIR_Kind get_kind() const = 0;

  /** Returns the kind of this node in text form.  This is here for output
      and debugging purposes. */
  virtual IIR_Char *get_kind_text() const = 0;

  /** Accessor to set the VHDL file name that this node was parsed from. 
      @param file_name The IIR_Identifier representation of the file name.
   */
  void set_file_name(IIR_Identifier *file_name);
  
  /** Accessor to set the line number of the VHDL this node corresponds
      to. 
      @param line_number The IIR_Int32 corresponding to the line number.
  */
  void set_line_number(IIR_Int32 line_number) {
    iir_line_number = line_number;
  }

  /** Read the file name from the node. */
  IIR_Identifier *get_file_name();

  /** Read the line number from the node. */
  IIR_Int32 get_line_number()  const { return iir_line_number; }
  
#ifdef SCHEMATIC_INFO
  /** The following nodes apply to schematic capture tools only and have to
      be #ifdefed in with SCHEMATIC_INFO. */
  //@{
  
  /** Set the sheet name of this node (if instantiated from schematic capture).
      @param sheet_name The IIR_Identifier representation of the sheet name.
   */
  void set_sheet_name(IIR_Identifier *sheet_name) {
    iir_sheet_name = sheet_name;
  }

  /** Set the X coordinate of this node (if instantiated from schematic capture).
      @param x_coordinate The IIR_Int32 representation of the X coordinate.
   */
  void set_x_coordinate(IIR_Int32 x_coordinate) {
    iir_x_coordinate = x_coordinate;
  }

  /** Set the Y coordinate of this node (if instantiated from schematic capture).
      @param y_coordinate The IIR_Int32 representation of the Y coordinate.
  */
  void set_y_coordinate(IIR_Int32 y_coordinate) {
    iir_y_coordinate = y_coordinate;
  }

  /** Get the sheet name. */
  IIR_Identifier *get_sheet_name() {return iir_sheet_name;}

  /** Get the X coordinate. */
  IIR_Int32 get_x_coordinate()     {return iir_x_coordinate;}

  /** Get the Y coordinate. */
  IIR_Int32 get_y_coordinate()     {return iir_y_coordinate;}

  //@}

#endif

#ifdef EXTRA_LOCATERS
  /** The following allow the analyzer to supply extra locator info
      if desired.  Savant doesn't supply this info.  To enable the space in
      the Savant IIR, #define EXTRA_LOCATERS. */

  //@{

  /** Set the column offset.
      @param column_offset The IIR_Int32 representation of the column offset of this
      node in the VHDL source. */
  void set_column_offset(IIR_Int32 column_offset) {
    iir_column_offset = column_offset;
  }

  /** Set the character offset.
      @param character_offset The IIR_Int32 representation of the character offset of this
      node in the VHDL source. */
  void set_character_offset(IIR_Int32 character_offset) {
    iir_character_offset = character_offset;
  }

  /** Get the column offset */
  IIR_Int32 get_column_offset()    {return iir_column_offset;}

  /** Get the character offset */
  IIR_Int32 get_character_offset() {return iir_character_offset;}
  //@}

#endif  

protected:
  IIRBase();
  virtual ~IIRBase() = 0;

  /** The Savant implementation of IIR stores a pointer back to the design
      file.  This allows access to the file name, as well as the symbol
      table associated with the file. */
  IIR_DesignFile *_my_design_file;
    
private:
  /** Where the line number is stored. */
  IIR_Int32  iir_line_number;
  
#ifdef SCHEMATIC_INFO
  /** Where the X coordinate is stored. */
  IIR_Int32  iir_x_coordinate;

  /** Where the Y coordinate is stored. */
  IIR_Int32  iir_y_coordinate;

  /** Where the sheet name is stored. */
  IIR_Identifier *iir_sheet_name;
#endif

#ifdef EXTRA_LOCATERS
  /** Where the column offset is stored. */
  IIR_Int32  iir_column_offset;

  /** Where the character offset is stored. */
  IIR_Int32  iir_character_offset;
#endif
  
};
#endif
